/*******************************************************************************
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/

package com.freescale.sa.ui.calltree.utils;

import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Logger;

/**
 * This class is built based on data from a CallTreeEditorTab and exports it
 * into dot format. It will create a subgraph for each software context (PID).
 * 
 * @author B46903
 * 
 */
public class DotExport {
	private static class NodesProperties extends HashMap<String, Properties> {
		private static final long serialVersionUID = 1L;

		@Override
		public Properties put(String nodeName, Properties nodeProps) {
			Properties props = get(nodeName);
			if (props == null) {
				super.put(nodeName, nodeProps);
				return nodeProps;
			}

			props.append(nodeProps);
			return props;
		}

		@Override
		public String toString() {
			StringBuilder builder = new StringBuilder();

			for (Map.Entry<String, Properties> entry : entrySet()) {
				builder.append(Arc.ARC_INDENT + "\"" + entry.getKey() + "\" " + entry.getValue() + "\n"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			}

			return builder.toString();
		}
	}

	private static class Properties extends HashMap<String, String> {
		private static final long serialVersionUID = 1L;

		@Override
		public String toString() {
			StringBuilder out = new StringBuilder("["); //$NON-NLS-1$
			Boolean first = true;

			for (Map.Entry<String, String> entry : entrySet()) {
				if (first) {
					out.append(entry.getKey() + " = \"" + entry.getValue() + "\""); //$NON-NLS-1$  //$NON-NLS-2$
					first = false;
				} else {
					out.append(", " + entry.getKey() + " = \"" + entry.getValue() + "\""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				}
			}
			out.append("]"); //$NON-NLS-1$

			return out.toString();
		}

		public void append(Properties nodeProps) {
			for (Map.Entry<String, String> entry : nodeProps.entrySet()) {
				String propName = entry.getKey();
				if (!containsKey(propName)) {
					put(propName, entry.getValue());
				}
			}
		};
	}

	/**
	 * This class defines an arc between two nodes from a graph
	 * 
	 * @author B46903
	 * 
	 */
	private static class Arc {
		private String source;
		private String destination;

		private Boolean isInMaxCallStack;
		private Boolean isInMaxSizeStack;
		private Integer numCalls;

		private static final String ARC_INDENT = "\t\t"; //$NON-NLS-1$

		// Properties
		private static final String LABEL_PROP = "label"; //$NON-NLS-1$
		private static final String COLOR_PROP = "color"; //$NON-NLS-1$
		private static final String FILL_COLOR_PROP = "fillcolor"; //$NON-NLS-1$
		private static final String STYLE_PROP = "style"; //$NON-NLS-1$
		private static final String FONT_COLOR_PROP = "fontcolor"; //$NON-NLS-1$

		private static final String FILLED = "filled"; //$NON-NLS-1$

		// Colors
		private static final String DARK_GREEN_COLOR = "darkgreen"; //$NON-NLS-1$
		private static final String WHITE_COLOR = "white"; //$NON-NLS-1$

		public Arc(final String source, final String destination, final Boolean isInMaxCallStack,
				final Boolean isInMaxSizeStack) {
			this.source = source;
			this.destination = destination;

			this.isInMaxCallStack = isInMaxCallStack;
			this.isInMaxSizeStack = isInMaxSizeStack;
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + ((destination == null) ? 0 : destination.hashCode());
			result = prime * result + ((source == null) ? 0 : source.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			Arc other = (Arc) obj;
			if (destination == null) {
				if (other.destination != null)
					return false;
			} else if (!destination.equals(other.destination))
				return false;
			if (source == null) {
				if (other.source != null)
					return false;
			} else if (!source.equals(other.source))
				return false;
			return true;
		}

		public void serialize(StringBuilder output, Integer occurrences, String contextName, NodesProperties nodes) {
			/*
			 * Add context name to generate unique ids for each node from the
			 * generated graph
			 */
			String sourceId = source + contextName;
			String destinationId = destination + contextName;

			Properties sourceProperties = getProperties(source, contextName);
			Properties destinationProperties = getProperties(destination, contextName);
			Properties labelProperties = getLabelProperties(occurrences);

			nodes.put(sourceId, sourceProperties);
			nodes.put(destinationId, destinationProperties);

			output.append(ARC_INDENT + "\"" + sourceId + "\" -> \"" + destinationId + "\""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			output.append(" " + labelProperties + ";\n"); //$NON-NLS-1$ //$NON-NLS-2$
		}

		private Properties getLabelProperties(Integer occurrences) {
			Properties properties = new Properties();

			properties.put(LABEL_PROP, occurrences.toString());
			if (isInMaxCallStack) {
				properties.put(COLOR_PROP, DARK_GREEN_COLOR);
			}

			return properties;
		}

		private Properties getProperties(String node, String contextName) {
			Properties properties = new Properties();

			properties.put(LABEL_PROP, node);

			if (isInMaxCallStack) {
				properties.put(COLOR_PROP, DARK_GREEN_COLOR);
			}

			if (isInMaxSizeStack) {
				properties.put(STYLE_PROP, FILLED);
				properties.put(FONT_COLOR_PROP, WHITE_COLOR);
				properties.put(FILL_COLOR_PROP, DARK_GREEN_COLOR);
			}

			return properties;
		}

		public Integer getNumCalls() {
			return numCalls;
		}

		public void setNumCalls(Integer numCalls) {
			this.numCalls = numCalls;
		}
	}

	/**
	 * Holds all arcs from a graph and number of occurrences of the arcs.
	 * 
	 * @author B46903
	 */
	static private class ContextGraph extends HashMap<Arc, Integer> {
		private static final long serialVersionUID = 1L;
		private static final String CLUSTER_INDENT = "\t"; //$NON-NLS-1$
		private String name;

		public ContextGraph(String name) {
			this.name = name;
		}

		public void addArc(Arc arc) {
			Integer value = get(arc);
			if (value == null) {
				put(arc, arc.getNumCalls());
			} else {
				put(arc, value + arc.getNumCalls());
			}
		}

		private String dotEscape(String str) {
			return str.replaceAll(" ", "_").replaceAll("/", "_"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}

		public void serialize(StringBuilder output) {
			NodesProperties props = new NodesProperties();
			output.append("\tsubgraph cluster_" + dotEscape(name) + " {\n"); //$NON-NLS-1$ //$NON-NLS-2$

			for (Map.Entry<Arc, Integer> entry : entrySet()) {
				entry.getKey().serialize(output, entry.getValue(), dotEscape(name), props);				
			}

			output.append(props);
			output.append(CLUSTER_INDENT + "\tcolor=lightgrey;\n"); //$NON-NLS-1$
			output.append(CLUSTER_INDENT + "\tlabel = \"" + name + "\";\n"); //$NON-NLS-1$ //$NON-NLS-2$
			output.append(CLUSTER_INDENT + "}\n"); //$NON-NLS-1$
		}
	}

	private Collection<CallTreeContext> contexts;
	private Collection<ContextGraph> graphs;
	private static final Logger LOGGER = Logger.getLogger(DotExport.class);

	public DotExport(Collection<CallTreeContext> contexts) {
		this.contexts = contexts;

		graphs = new ArrayList<ContextGraph>();
	}

	/**
	 * Exports contexts into dot format.
	 * 
	 * @param filePath
	 *            The output file path
	 */
	public void export(String filePath) {
		/* Building */
		for (CallTreeContext context : contexts) {
			ContextGraph graph = parseContext(context);
			graphs.add(graph);
		}

		StringBuilder output = new StringBuilder();
		output.append("digraph G {\n"); //$NON-NLS-1$

		/* Serialization */
		for (ContextGraph graph : graphs) {
			graph.serialize(output);
		}

		output.append("}\n"); //$NON-NLS-1$

		/* Writing */
		try {
			PrintWriter writer = new PrintWriter(filePath, "UTF-8"); //$NON-NLS-1$
			writer.print(output);
			writer.close();
		} catch (FileNotFoundException e) {
			LOGGER.debug(e.getLocalizedMessage());
		} catch (UnsupportedEncodingException e) {
			LOGGER.debug(e.getLocalizedMessage());
		}
	}

	/**
	 * Iterate over functions from a context and build a graph
	 * 
	 * @param context
	 *            The context to be parsed
	 */
	private ContextGraph parseContext(CallTreeContext context) {
		ContextGraph graph = new ContextGraph(context.getName());

		for (EdgeRec edge : context.getEdges()) {
			Arc arc = new Arc(edge.getCallerShortName(), edge.calleeShortName, edge.isInMaxCallStack(),
					edge.isInMaxSizeStack());
			arc.setNumCalls(edge.numCalls);
			graph.addArc(arc);
		}

		return graph;
	}
}
