/*******************************************************************************
 * Copyright (c) 2014, 2015 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/

package com.freescale.sa.ui.calltree.utils;

import java.math.BigInteger;

import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableCellConfiguration;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableRecordConfiguration;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableRecordData;
import com.freescale.sa.util.CommonConstants;

/**
 * This is an adapter for a TableRecordData object. It is built on a
 * TableRecordConfiguration and interprets TableRecordData columns based on the
 * given configuration.
 * 
 * @author B46903
 * 
 */
public class ContextDataIntepretor {
	private static final Integer INVALID_COLUMN_INDEX = -1;

	private static final Float FLOAT_DEFAULT_VALUE = 0F;
	private static final Boolean BOOL_DEFAULT_VALUE = false;
	private static final Integer INTEGER_DEFAULT_VALUE = 0;
	private static final String STRING_DEFAULT_VALUE = CommonConstants.EMPTY_STRING;
	private static final Double DOUBLE_DEFAULT_VALUE = 0D;
	private static final BigInteger BIG_INTEGER_DEFAULT_VALUE = new BigInteger("0"); //$NON-NLS-1$

	private int callerNameIndex = INVALID_COLUMN_INDEX;
	private int calleeNameIndex = INVALID_COLUMN_INDEX;
	private int callerIdIndex = INVALID_COLUMN_INDEX;
	private int calleeIdIndex = INVALID_COLUMN_INDEX;
	private int callerFunctionIdIndex = INVALID_COLUMN_INDEX;
	private int calleeFunctionIdIndex = INVALID_COLUMN_INDEX;
	private int callSiteIndex = INVALID_COLUMN_INDEX;
	private int numberOfCallsIndex = INVALID_COLUMN_INDEX;
	private int sourceFileIndex = INVALID_COLUMN_INDEX;
	private int lineNoIndex = INVALID_COLUMN_INDEX;
	private int maxCallStackIndex = INVALID_COLUMN_INDEX;
	private int maxSizeStackIndex = INVALID_COLUMN_INDEX;
	private int totalCallsOfParentIndex = INVALID_COLUMN_INDEX;
	private int totalTimeItWasCalledIndex = INVALID_COLUMN_INDEX;
	private int inclusiveTimeIndex = INVALID_COLUMN_INDEX;
	private int grayedOutIndex = INVALID_COLUMN_INDEX;

	public ContextDataIntepretor(TableRecordConfiguration functionConfig) {
		// Read the file's header to see the order of data
		for (int i = 0; i < functionConfig.cells.size(); i++) {
			TableCellConfiguration cell = functionConfig.cells.get(i);

			if (cell.name.equals(UIConstants.flatdata_caller_name)) {
				callerNameIndex = i;
			}
			if (cell.name.equals(UIConstants.flatdata_callee_name)) {
				calleeNameIndex = i;
			}
			if (cell.name.equals(UIConstants.flatdata_caller_ID)) {
				callerIdIndex = i;
			}
			if (cell.name.equals(UIConstants.flatdata_callee_ID)) {
				calleeIdIndex = i;
			}
			if (cell.name.equals(UIConstants.flatdata_caller_function_ID)) {
				callerFunctionIdIndex = i;
			}
			if (cell.name.equals(UIConstants.flatdata_callee_function_ID)) {
				calleeFunctionIdIndex = i;
			}
			if (cell.name.equals(UIConstants.flatdata_call_site)) {
				callSiteIndex = i;
			}
			if (cell.name.equals(UIConstants.flatdata_no_of_calls)) {
				numberOfCallsIndex = i;
			}
			if (cell.name.equals(UIConstants.flatdata_source_file_path)) {
				sourceFileIndex = i;
			}
			if (cell.name.equals(UIConstants.flatdata_line_no)) {
				lineNoIndex = i;
			}
			if (cell.name.equals(UIConstants.flatdata_max_call_stack)) {
				maxCallStackIndex = i;
			}
			if (cell.name.equals(UIConstants.flatdata_max_size_stack)) {
				maxSizeStackIndex = i;
			}
			if (cell.name.equals(UIConstants.flatdata_total_calls_of_parent)) {
				totalCallsOfParentIndex = i;
			}
			if (cell.name.equals(UIConstants.flatdata_total_time_it_was_called)) {
				totalTimeItWasCalledIndex = i;
			}
			if (cell.name.equals(UIConstants.flatdata_inclusive_time)) {
				inclusiveTimeIndex = i;
			}
			if (cell.name.equals(UIConstants.flatdata_grayed_out_symbol)) {
				grayedOutIndex = i;
			}
		}
	}

	public EdgeRec buildEdgeRec(TableRecordData record) {
		return new EdgeRec(getCallerId(record), getCallerFunctionId(record), getCallerName(record), getCallSite(record), getSourceFile(record),
				getLineNo(record), getCalleeId(record), getCalleeFunctionId(record), getCalleeName(record), getNumberOfCalls(record),
				getMaxCallStack(record), getMaxSizeStack(record), getTotalCallsOfParent(record),
				getTotalTimeItWasCalled(record), getInclusiveTime(record), getGrayedOut(record));
	}

	private int getIntOrMaxInt(String value) {
		int returnValue;
		try {
			returnValue = Integer.parseInt(value);
		} catch (NumberFormatException e) {
			returnValue = Integer.MAX_VALUE;
		}
		return returnValue;
	}

	private double getDouble(String value) {
		double returnValue;
		try {
			returnValue = Double.parseDouble(value);
		} catch (NumberFormatException e) {
			returnValue = Double.MAX_VALUE;
		}
		return returnValue;
	}

	/**
	 * It gets a BigInteger value from string input. String input is expected to
	 * be in decimal format.
	 * 
	 * @param value
	 *            String input in decimal format.
	 * @return BigInteger value for string input.
	 */
	private BigInteger getBigInteger(String value) {
		BigInteger returnValue;
		try {
			returnValue = new BigInteger(value);
		} catch (NumberFormatException e) {
			returnValue = BigInteger.ZERO;
		}
		return returnValue;
	}

	// Integers
	public Integer getCallerId(TableRecordData record) {
		if (callerIdIndex == INVALID_COLUMN_INDEX) {
			return INTEGER_DEFAULT_VALUE;
		}

		return getIntOrMaxInt(record.cells.get(callerIdIndex).value);
	}

	public Integer getCalleeId(TableRecordData record) {
		if (calleeIdIndex == INVALID_COLUMN_INDEX) {
			return INTEGER_DEFAULT_VALUE;
		}
		
		return getIntOrMaxInt(record.cells.get(calleeIdIndex).value);
	}
	
	public Integer getCallerFunctionId(TableRecordData record) {
		if (callerFunctionIdIndex == INVALID_COLUMN_INDEX) {
			return EdgeRec.INVALID_FUNCTION_ID;
		}

		return getIntOrMaxInt(record.cells.get(callerFunctionIdIndex).value);
	}

	public Integer getCalleeFunctionId(TableRecordData record) {
		if (calleeFunctionIdIndex == INVALID_COLUMN_INDEX) {
			return EdgeRec.INVALID_FUNCTION_ID;
		}
		
		return getIntOrMaxInt(record.cells.get(calleeFunctionIdIndex).value);
	}

	public BigInteger getCallSite(TableRecordData record) {
		if (callSiteIndex == INVALID_COLUMN_INDEX) {
			return BIG_INTEGER_DEFAULT_VALUE;
		}
		
		return getBigInteger(record.cells.get(callSiteIndex).value);
	}

	public Integer getLineNo(TableRecordData record) {
		if (lineNoIndex == INVALID_COLUMN_INDEX) {
			return INTEGER_DEFAULT_VALUE;
		}
		return getIntOrMaxInt(record.cells.get(lineNoIndex).value);
	}

	public Integer getNumberOfCalls(TableRecordData record) {
		if (numberOfCallsIndex == INVALID_COLUMN_INDEX) {
			return INTEGER_DEFAULT_VALUE;
		}
		return getIntOrMaxInt(record.cells.get(numberOfCallsIndex).value);
	}

	// Strings
	public String getCallerName(TableRecordData record) {
		if (callerNameIndex == INVALID_COLUMN_INDEX) {
			return STRING_DEFAULT_VALUE;
		}
		return record.cells.get(callerNameIndex).value;
	}

	public String getCalleeName(TableRecordData record) {
		if (calleeNameIndex == INVALID_COLUMN_INDEX) {
			return STRING_DEFAULT_VALUE;
		}
		return record.cells.get(calleeNameIndex).value;
	}

	public String getSourceFile(TableRecordData record) {
		if (sourceFileIndex == INVALID_COLUMN_INDEX) {
			return STRING_DEFAULT_VALUE;
		}
		return record.cells.get(sourceFileIndex).value;
	}

	public String getMaxCallStack(TableRecordData record) {
		if (maxCallStackIndex == INVALID_COLUMN_INDEX) {
			return STRING_DEFAULT_VALUE;
		}
		return record.cells.get(maxCallStackIndex).value;
	}

	public String getMaxSizeStack(TableRecordData record) {
		if (maxSizeStackIndex == INVALID_COLUMN_INDEX) {
			return STRING_DEFAULT_VALUE;
		}
		return record.cells.get(maxSizeStackIndex).value;
	}

	// Floats
	public Float getTotalCallsOfParent(TableRecordData record) {
		if (totalCallsOfParentIndex == INVALID_COLUMN_INDEX) {
			return FLOAT_DEFAULT_VALUE;
		}
		return Float.parseFloat(record.cells.get(totalCallsOfParentIndex).value);
	}

	public Float getTotalTimeItWasCalled(TableRecordData record) {
		if (totalTimeItWasCalledIndex == INVALID_COLUMN_INDEX) {
			return FLOAT_DEFAULT_VALUE;
		}
		return Float.parseFloat(record.cells.get(totalTimeItWasCalledIndex).value);
	}

	// Doubles
	public Double getInclusiveTime(TableRecordData record) {
		if (inclusiveTimeIndex == INVALID_COLUMN_INDEX) {
			return DOUBLE_DEFAULT_VALUE;
		}
		return getDouble(record.cells.get(inclusiveTimeIndex).value);
	}

	// Booleans
	public Boolean getGrayedOut(TableRecordData record) {
		if (grayedOutIndex == INVALID_COLUMN_INDEX) {
			return BOOL_DEFAULT_VALUE;
		}
		return Boolean.parseBoolean(record.cells.get(grayedOutIndex).value);
	}
}
