/*******************************************************************************
 * Copyright (c) 2014, 2015 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/

package com.freescale.sa.ui.calltree.utils;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Vector;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import com.freescale.sa.ui.calltree.editor.CallStackRec;
import com.freescale.sa.ui.common.SAErrorDialog;
import com.freescale.sa.ui.common.editor.flatprofiler.FlatProfilerFileInput;
import com.freescale.sa.ui.common.IHierarchicalItem;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.SubsectionData;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableData;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableRecordConfiguration;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.TableRecordData;

/*
 * This class describes data needed to build a subtree for a software context into a CallTree tab
 */
public class CallTreeContext implements IHierarchicalItem {

	private ArrayList<EdgeRec> edges;
	private SubsectionData subsection;
	private Vector<CallStackRec> maxCallStack;
	private Vector<CallStackRec> maxSizeStack;

	private static BigInteger NO_CALL_SITE = new BigInteger("-1");//$NON-NLS-1$

	private Function root;

	public CallTreeContext(FlatProfilerFileInput flatProfiler, SubsectionData subsection) {
		this.subsection = subsection;

		TableData functionData = subsection.get(0L);
		edges = new ArrayList<EdgeRec>();

		TableRecordConfiguration functionConfig = null;
		ContextDataIntepretor dataIntepretor = null;

		functionConfig = flatProfiler.tableRecordsConfigurations.get(UIConstants.flatdata_callPair_line);

		if (functionConfig == null) {
			SAErrorDialog.showErrorDialog(UIMessages.CallTreeEditor_18, UIMessages.CallTreeEditor_18, new Status(
					IStatus.ERROR, UIConstants.PLUGIN_ID, UIMessages.CallTreeEditor_19));
			return;
		}

		dataIntepretor = new ContextDataIntepretor(functionConfig);

		// Read data for each record
		for (TableRecordData record : functionData.getRecords()) {
			if (!record.recordConfigurationName.equals(UIConstants.flatdata_callPair_line)) {
				continue;
			}

			EdgeRec edge = dataIntepretor.buildEdgeRec(record);
			edges.add(edge);
		}

		initRootFunction();
	}

	public void initRootFunction() {
		Integer id = 0;

		if (edges == null || edges.isEmpty()) {
			return;
		}

		if (maxCallStack == null || maxSizeStack == null) {
			setMaxCallStacks();
			setMaxSizeStack();
		}

		for (int i = 0; i < edges.size(); i++) {
			if (edges.get(i).getCallerId() == id) {
				root = new Function(edges.get(i).getCallerShortName(), edges.get(i).getCallerId(), null, NO_CALL_SITE,
						this, 0, edges.get(i).isGrayedOut(), edges.get(i).getCallerFunctionId());
				root.addFunction(root);
				break;
			}
		}
	}

	public String getName() {
		return subsection.getName();
	}

	public void setMaxCallStacks() {
		maxCallStack = CallStackRec.getMaxCallStack(edges);
	}

	public void setMaxSizeStack() {
		maxSizeStack = CallStackRec.getMaxSizeStack(edges);
	}

	/**
	 * Returns the edges of the context
	 * 
	 * @return The edges
	 */
	public ArrayList<EdgeRec> getEdges() {
		return edges;
	}

	public Vector<CallStackRec> getMaxCallStack() {
		return maxCallStack;
	}

	public Vector<CallStackRec> getMaxSizeStack() {
		return maxSizeStack;
	}

	@Override
	public IHierarchicalItem getParent() {
		return null;
	}

	@Override
	public <T extends IHierarchicalItem> void setParent(T parent) {
		// TODO : Nothing to do
	}

	@Override
	public Collection<? extends IHierarchicalItem> getChildren() {
		if (root == null) {
			return null;
		}

		ArrayList<Function> functions = new ArrayList<Function>();
		functions.add(root);
		return functions;
	}

	@Override
	public boolean hasChildren() {
		return (root != null);
	}

	public Function getRootFunction() {
		return root;
	}
}
