/*******************************************************************************
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.calltree.editor;

import java.text.DecimalFormat;

import org.apache.log4j.Logger;
import org.eclipse.jface.viewers.IFontProvider;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ITableColorProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;

import com.freescale.sa.ui.calltree.utils.CallTreeContext;
import com.freescale.sa.ui.calltree.utils.Function;
import com.freescale.sa.ui.calltree.utils.UIConstants;
import com.freescale.sa.ui.common.SAUILSPlugin;
import com.freescale.sa.ui.common.utils.ResourceManager;
import com.freescale.sa.ui.common.utils.TimeUnitGroup;
import com.freescale.sa.ui.common.utils.TimeUnitUtils;
import com.freescale.sa.util.CommonConstants;

class CallTreeLabelProvider extends LabelProvider implements IFontProvider, ITableLabelProvider, ITableColorProvider {

	public static final int MAX_SIZE_STACK_COL = SWT.COLOR_GREEN;
	public static final int GRAYED_OUT_COL = SWT.COLOR_DARK_GRAY;
	public static final int GRAYED_OUT_BKG = SWT.COLOR_GRAY;

	private Image m_imgIsr = null;
	private Image m_imgFunc = null;

	// Comma as thousands separator.
	private DecimalFormat comma1000sSeparator = new DecimalFormat(",##0"); //$NON-NLS-1$

	// There are columns that need time unit and frequency to be formatted.
	private TimeUnitGroup m_TimeUnitGroup;

	private static Logger LOGGER = Logger.getLogger(CallTreeLabelProvider.class);

	// Constructs a PlayerLabelProvider
	public CallTreeLabelProvider(TimeUnitGroup timeFormatGroup) {
		m_TimeUnitGroup = timeFormatGroup;
	}

	public Font getFont(Object obj) {
		if (obj instanceof CallTreeContext) {
			return ResourceManager.getFont("Tahoma", 8, SWT.BOLD); //$NON-NLS-1$			
		}

		if (((Function) obj).isRecursive()) {
			return ResourceManager.getFont("Tahoma", 8, SWT.BOLD); //$NON-NLS-1$
		} else {
			return ResourceManager.getFont("Tahoma", 8, SWT.NORMAL); //$NON-NLS-1$
		}
	}

	private void loadImages() {
		if (m_imgIsr != null && m_imgFunc != null) {
			return;
		}

		/*
		 * when using the resource manager this widget (CallTreeViewer) must be
		 * registered as user to ResourceManager see CallTreeEditor
		 */
		try {
			m_imgIsr = SAUILSPlugin.imageDescriptorFromPlugin(UIConstants.PLUGIN_ID, UIConstants.analysis_isr_iconfile)
					.createImage();
			m_imgFunc = SAUILSPlugin.imageDescriptorFromPlugin(UIConstants.PLUGIN_ID,
					UIConstants.analysis_function_iconfile).createImage();
		} catch (Exception e) {
			LOGGER.debug(e.getLocalizedMessage());
		}
	}

	/**
	 * Gets the image for the specified column
	 * 
	 * @param arg0
	 *            the player
	 * @param arg1
	 *            the column
	 * @return Image
	 */
	public Image getColumnImage(Object arg0, int columnIndex) {
		Image image = null;

		if (arg0 instanceof CallTreeContext) {
			return null;
		}

		if (columnIndex == 0) {
			loadImages();
			if (((Function) arg0).getName().contains("IntVec")) { //$NON-NLS-1$
				image = m_imgIsr;
			} else {
				image = m_imgFunc;
			}
		}

		return image;
	}

	/**
	 * Gets the text for the specified column
	 * 
	 * @param arg0
	 *            the player
	 * @param arg1
	 *            the column
	 * @return String
	 */
	public String getColumnText(Object arg0, int columnIndex) {

		String text = CommonConstants.EMPTY_STRING;

		if (arg0 instanceof CallTreeContext) {
			if (columnIndex == 0) {
				return ((CallTreeContext) arg0).getName();
			}

			return null;
		}

		if ((((Function) arg0).getId() != 0) || (columnIndex == 0)) {
			switch (columnIndex) {
			case 0:
				text = ((Function) arg0).getName();
				break;

			case 1:
				text = comma1000sSeparator.format(((Function) arg0).getNumCalls());
				break;

			case 2:
				text = String.format("%.2f", ((Function) arg0).getPctTotalCallsOfParent()); //$NON-NLS-1$
				break;

			case 3:
				text = String.format("%.2f", ((Function) arg0).getPctTotalTimesItWasCalled()); //$NON-NLS-1$
				break;

			case 4:
				/*
				 * Inclusive time column is formatted according with time unit
				 * and frequency.
				 */
				double inclusiveTime = ((Function) arg0).getInclusiveTime();
				if (m_TimeUnitGroup != null) {
					text = TimeUnitUtils.convertCycles(String.valueOf(inclusiveTime),
							m_TimeUnitGroup.getTimeUnitSelection(), m_TimeUnitGroup.getFreqSelection());
				} else {
					LOGGER.error("Error : Unable to format column text because time unit and frequency provider is null."); //$NON-NLS-1$
				}
				break;
			default:
				return text;
			}
		}
		return text;
	}

	public Color getForeground(Object element, int columnIndex) {
		if (element instanceof CallTreeContext) {
			return null;
		}

		if (columnIndex == 0) {
			if (((Function) element).isOnMaxSizeStack()) {
				return Display.getCurrent().getSystemColor(MAX_SIZE_STACK_COL);
			} else if (((Function) element).isOnMaxCallStack()) {
				return Display.getCurrent().getSystemColor(SWT.COLOR_WHITE);
			} else if (((Function) element).isExcludedSymbol()) {
				return Display.getCurrent().getSystemColor(GRAYED_OUT_COL);
			}
		}
		return null;
	}

	/** @see IColorProvider#getBackground(Object) */
	public Color getBackground(Object element, int columnIndex) {
		if (element instanceof CallTreeContext) {
			return null;
		}

		if (columnIndex == 0) {
			if (((Function) element).isExcludedSymbol()) {
				return Display.getCurrent().getSystemColor(GRAYED_OUT_BKG);
			} else if (((Function) element).isOnMaxCallStack()) {
				return Display.getCurrent().getSystemColor(SWT.COLOR_DARK_GRAY);
			}
		}
		return null;
	}

	/**
	 * Adds a listener
	 * 
	 * @param arg0
	 *            the listener
	 */
	public void addListener(ILabelProviderListener arg0) {// Throw it away
	}

	/**
	 * Dispose any created resources
	 */
	public void dispose() {
		/*
		 * ResourceManager will dispose the resources when the CallTreeViewer
		 * gets disposed
		 */
		if (m_imgIsr != null) {
			m_imgIsr.dispose();
		}

		if (m_imgFunc != null) {
			m_imgFunc.dispose();
		}
	}

	/**
	 * Removes the specified listener
	 * 
	 * @param arg0
	 *            the listener
	 */
	public void removeListener(ILabelProviderListener arg0) {
		// Do nothing
	}
}
