/*******************************************************************************
 * Copyright (c) 2014-2015 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/
package com.freescale.sa.ui.calltree.editor;

import java.util.ArrayList;
import java.util.Collection;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Device;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;

import com.freescale.sa.model.AnalysisManager;
import com.freescale.sa.model.IAnalyzer;
import com.freescale.sa.ui.common.AbstractSaEditor;
import com.freescale.sa.ui.common.IConfigurableDisplay;
import com.freescale.sa.ui.common.IRefreshableDisplay;
import com.freescale.sa.ui.common.editor.DragAndDropManager;
import com.freescale.sa.ui.common.editor.flatprofiler.FlatProfilerFileInput;

public class CallTreeInput implements IConfigurableDisplay, IRefreshableDisplay {
	/**
	 * <p>
	 * The unique ID of this editor.
	 * </p>
	 */
	public static final String ID = "com.freescale.sa.ui.callTreeEditor"; //$NON-NLS-1$

	private static Logger LOGGER = Logger.getLogger(CallTreeInput.class);

	private Composite m_previousParent = null;

	private CallTreeEditorTab m_activeTab = null;
	private ArrayList<CallTreeEditorTab> m_tabs;
	private Collection<String> m_sections;
	private FlatProfilerFileInput m_fileInput = null;
	private AbstractSaEditor m_editor = null;
	private String m_storageDir;
	private String m_archID;
	private IAnalyzer m_analyzer;

	/**
	 * Creates a call tree editor
	 * 
	 * @param fileName
	 *            The path to input file
	 */
	public CallTreeInput(String fileName) {
		m_fileInput = new FlatProfilerFileInput(fileName);
		m_fileInput.initializeConfiguration();
		m_sections = m_fileInput.getSections();

		m_tabs = new ArrayList<CallTreeEditorTab>();
	}

	/** Creates a call tree editor */
	public CallTreeInput(String fileName, String baseName, String storageDir, String arch, String projectName) {
		this(fileName);
		m_storageDir = storageDir;
		m_archID = arch;

		if (m_fileInput.tableRecordsConfigurations.size() > 0) {

			try {
				m_analyzer = AnalysisManager.getDefault().getAnalyzer(m_archID);
			} catch (CoreException e) {
				LOGGER.debug("[CallTreeInput] unable to get analyzer. ", e); //$NON-NLS-1$
			}
		}
	}
	
	/**
	 * Returns the input file path.
	 * @return The input file path
	 */
	public String getFilePath() {
		if (m_fileInput == null) {
			return null;
		}
		
		return m_fileInput.m_fileName;
	}

	/**
	 * Destroys all internal tabs
	 */
	private void disposeEditors() {
		if (m_tabs == null)
			return;

		for (CallTreeEditorTab tab : m_tabs) {
			tab.dispose();
		}

		m_tabs.clear();
	}

	/**
	 * Draws all editor's tabs
	 * 
	 * @param parent
	 *            Parent composite
	 */
	protected void draw(Composite parent) {
		if (m_previousParent != parent) {
			m_previousParent = parent;
			disposeEditors();

			Composite composite = new Composite(parent, SWT.FILL);
			composite.setLayout(new FillLayout());

			final CTabFolder folder = new CTabFolder(composite, SWT.TOP | SWT.BOTTOM);
			folder.addSelectionListener(new SelectionAdapter() {
				@Override
				public void widgetDefaultSelected(SelectionEvent e) {
					CTabItem selection = folder.getSelection();
					if (selection == null)
						return;

					m_activeTab = m_tabs.get(folder.getSelectionIndex());
				}
			});
			
			//Add drag and drop support
			//Add only if there are more than 1 sections (sources)
			if (m_sections != null) {
				if (m_sections.size() > 1) {
					final DragAndDropManager dndm = DragAndDropManager.getInstance();
								
					String drSrc = dndm.getDraggedString();
					m_fileInput.setIsFilteredFile(drSrc != null && !drSrc.isEmpty());
					
					if (!m_fileInput.getIsFilteredFile()) {		
						dndm.addListeners(folder, m_fileInput.m_fileName);					
					} else {
						if (drSrc.length() > 0 && m_editor != null) {
							m_fileInput.addFilteredSection(drSrc);
							m_editor.setEditorTitle(drSrc);
							dndm.clearDraggedString();
						}
					}
				}
	
				for (String section : m_sections) {
					if (m_fileInput.getIsSourceFiltered(section)) {
						CTabItem tab = new CTabItem(folder, SWT.NONE);
						tab.setShowClose(true);
						tab.setText(section);
						CallTreeEditorTab ctab = new CallTreeEditorTab(m_fileInput, section, folder, m_analyzer);
						ctab.setStorageDir(m_storageDir);
						Device device = Display.getCurrent ();
						Color white = new Color (device, 255, 255, 255);
						ctab.getComposite().setBackground(white);
						ctab.getComposite().setBackgroundMode(SWT.INHERIT_FORCE);
						tab.setControl(ctab.getComposite());
						m_tabs.add(ctab);
					}
				}
			}
			
			//Select first tab
			if (folder.getItemCount() > 0) {
				folder.setSelection(0);
			}
		}

		for (CallTreeEditorTab ctab : m_tabs) {
			ctab.draw();
		}
	}

	@Override
	public void refreshDisplay() {
		for (CallTreeEditorTab tab : m_tabs) {
			tab.refresh(false);
		}
	}

	@Override
	public String getDisplayId() {
		if (m_activeTab == null)
			return null;

		return m_activeTab.getDisplayId();
	}

	@Override
	public void configure(CONFIG_ELEMENT elementType, String value) {
		if (m_activeTab == null)
			return;

		m_activeTab.configure(elementType, value);
	}

	/**
	 * It refreshes active tab with or without reloading data input.
	 * 
	 * @param reloadData
	 *            True for reload data input using new collected trace. False to
	 *            perform refresh on tree without reloading data input, only
	 *            changed items will be refreshed.
	 */
	public void refresh(boolean b) {
		if (m_activeTab == null)
			return;

		m_activeTab.refresh(true);
	}

	/**
	 * Getter for input data
	 * 
	 * @return Internal data used to build internal tabs
	 */
	public FlatProfilerFileInput getFlatProfiler() {
		return m_fileInput;
	}
	
	public String getTraceType(){
		if (m_fileInput != null) {
			return m_fileInput.getTraceType();
		}
		
		return null;
	}
	
	public AbstractSaEditor getEditor() {
		return m_editor;
	}

	public void setEditor(AbstractSaEditor m_editor) {
		this.m_editor = m_editor;
	}
}
