/*******************************************************************************
 * Copyright 2016-2018 NXP.
 *
 * Copyright (c) 2014 Freescale Semiconductor, Inc. All rights reserved.
 * Freescale Internal Only. Not for distribution
 *******************************************************************************/

package com.freescale.sa.ui.calltree.editor;

import java.io.File;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;

import org.apache.log4j.Logger;
import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.AbstractTreeViewer;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.ide.IGotoMarker;
import org.eclipse.ui.texteditor.AbstractTextEditor;

import com.freescale.sa.model.IAnalyzer;
import com.freescale.sa.model.PlatformType;
import com.freescale.sa.ui.calltree.utils.CallTreeContext;
import com.freescale.sa.ui.calltree.utils.DotExport;
import com.freescale.sa.ui.calltree.utils.Function;
import com.freescale.sa.ui.calltree.utils.UIConstants;
import com.freescale.sa.ui.calltree.utils.UIMessages;
import com.freescale.sa.ui.common.IConfigurableDisplay;
import com.freescale.sa.ui.common.IRefreshableDisplay;
import com.freescale.sa.ui.common.editor.ConfigureResultsDialog;
import com.freescale.sa.ui.common.editor.ConfigureResultsDialog.ConfigureResultsTabId;
import com.freescale.sa.ui.common.editor.flatprofiler.FlatProfilerFileInput;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.SectionData;
import com.freescale.sa.ui.common.flatdata.FlatDataProvider.SubsectionData;
import com.freescale.sa.ui.common.utils.ResourceManager;
import com.freescale.sa.ui.common.utils.SaUiLsUtils;
import com.freescale.sa.ui.common.utils.TimeUnitGroup;
import com.freescale.sa.ui.common.utils.TimeUnitUtils;
import com.freescale.sa.util.CommonConstants;
import com.freescale.sa.util.LaunchConfigUtils;

public class CallTreeEditorTab implements IConfigurableDisplay, IRefreshableDisplay {
	/**
	 * <p>
	 * The unique ID of this editor.
	 * </p>
	 */
	public static final String ID = "com.freescale.sa.ui.callTreeEditor"; //$NON-NLS-1$

	private ArrayList<CallTreeContext> contexts = null;
	private Menu calltreeMenu = null;
	private IMarker bookmark = null;
	private FlatProfilerFileInput flatProfiler = null;

	private Composite m_composite;

	private TreeViewer tree;
	private CallTreeContentProvider m_contentProvider = new CallTreeContentProvider();
	private AbstractTextEditor editor = null;
	private String sourceFileName = null;
	private int lineNumber = 0;

	private static Logger LOGGER = Logger.getLogger(CallTreeInput.class);
	private static final String START_MARKER = "<START>"; //$NON-NLS-1$

	private String projectName;
	private String sectionName;
	private IAnalyzer analyzer;

	/**
	 * Menu for configuring the time unit.
	 */
	protected TimeUnitGroup m_timeUnitGroup;

	private String storageDir;

	public CallTreeEditorTab(FlatProfilerFileInput fileInput, String section, Composite parent, IAnalyzer analyzer) {
		this.projectName = "unknown"; //$NON-NLS-1$
		flatProfiler = fileInput;
		m_composite = new Composite(parent, SWT.FILL);
		m_composite.setLayout(new GridLayout());
		sectionName = section;
		this.analyzer = analyzer;

		contexts = getDataFromFile();
	}

	public void setFlatProfiler(String fileName) {
		flatProfiler = new FlatProfilerFileInput(fileName);
		flatProfiler.initializeConfiguration();
	}

	@SuppressWarnings("unused")
	private void showSource() {
		IFileStore srcFileStore = null;
		IStructuredSelection selection = (IStructuredSelection) tree.getSelection();

		if (selection != null) {
			Function selFunction = (Function) selection.getFirstElement();

			if ((selFunction.getName().equals(START_MARKER)) == false) {
				sourceFileName = selFunction.getFuncFile();

				lineNumber = selFunction.getFuncLineNo();
				if ((sourceFileName.equals("unknown")) == false) { //$NON-NLS-1$
					IResource resourceFile = null;
					IWorkspace workspace = ResourcesPlugin.getWorkspace();
					IWorkspaceRoot wspRoot = workspace.getRoot();
					sourceFileName = getArchivedFile(sourceFileName);
					Path newFilePath = new Path(sourceFileName);

					resourceFile = wspRoot.getFileForLocation(newFilePath);
					try {
						IWorkbench workBench = PlatformUI.getWorkbench();
						IWorkbenchWindow wbWindow = workBench.getActiveWorkbenchWindow();
						IWorkbenchPage wbPage = wbWindow.getActivePage();
						if (resourceFile == null) {
							// try again to get a resource to add bookmark -
							// maybe is an external file
							IPath filePath = new Path(sourceFileName);
							if (filePath.toFile().exists()) {
								resourceFile = (IResource) EFS.getLocalFileSystem().getStore(filePath);
							} else {
								// nothing to do
								return;
							}
						}

						if ((bookmark != null) && (bookmark.exists())) {
							bookmark.delete();
						}

						bookmark = resourceFile.createMarker(IMarker.BOOKMARK);
						lineNumber = selFunction.getFuncLineNo();

						if (resourceFile instanceof IFileStore) {
							// external file editor
							editor = (AbstractTextEditor) IDE.openEditorOnFileStore(wbPage, srcFileStore);
						} else {
							// TODO : Not yet supported on LS
							// open a file from workspace
							// editor = (AbstractTextEditor)
							// wbPage.openEditor(EditorUtility.getEditorInput(resourceFile),
							// CEditorID);
						}

						if (bookmark.exists()) {
							bookmark.setAttribute(IMarker.LINE_NUMBER, lineNumber);
							// Now scroll to and select the bookmark
							IGotoMarker gtm = (IGotoMarker) editor.getAdapter(IGotoMarker.class);
							gtm.gotoMarker(bookmark);
						}

					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
		}

	}

	public void draw() {
		// Create tool bar.
		ToolBar toolBar = createToolBar(m_composite);
		toolBar.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		// Create tree viewer.
		createTreeViewer(m_composite);
	}

	private void createTreeViewer(Composite parentComposite) {
		tree = new TreeViewer(parentComposite, SWT.NONE);

		GridLayout viewerLayout = new GridLayout();

		viewerLayout.marginLeft = 1;
		viewerLayout.marginRight = 1;

		tree.getTree().setLayout(viewerLayout);

		GridData viewerLData = new GridData(GridData.FILL_BOTH);

		viewerLData.grabExcessVerticalSpace = true;
		viewerLData.grabExcessHorizontalSpace = true;
		viewerLData.horizontalAlignment = GridData.FILL;
		viewerLData.verticalAlignment = GridData.FILL;
		tree.getTree().setLayoutData(viewerLData);

		/**
		 * the label provider will use image resources (for icons). Before using
		 * any resource, the resource manager must know about its clients
		 * 
		 * @see ResourceManager.registerResourceUser();
		 */
		ResourceManager.registerResourceUser(parentComposite);

		tree.setContentProvider(m_contentProvider);
		tree.setLabelProvider(new CallTreeLabelProvider(m_timeUnitGroup));

		// Get maximum call and size stacks
		setMaxStacks();
		setInputTreeViewer();

		new TreeColumn(tree.getTree(), SWT.LEFT).setText(UIMessages.CallTreeEditor_1);
		new TreeColumn(tree.getTree(), SWT.RIGHT).setText(UIMessages.CallTreeEditor_2);
		new TreeColumn(tree.getTree(), SWT.RIGHT).setText(UIMessages.CallTreeEditor_3);
		new TreeColumn(tree.getTree(), SWT.RIGHT).setText(UIMessages.CallTreeEditor_4);

		String timeColText = UIMessages.CallTreeEditor_5;
		if (m_timeUnitGroup != null)
			timeColText = TimeUnitUtils.getTimeColumnNameWithDescription(timeColText,
					m_timeUnitGroup.getTimeUnitSelection(), m_timeUnitGroup.getFreqSelection());

		new TreeColumn(tree.getTree(), SWT.RIGHT).setText(timeColText);

		tree.getTree().getColumn(0).setToolTipText(UIMessages.CallTreeEditor_6);
		tree.getTree().getColumn(1).setToolTipText(UIMessages.CallTreeEditor_7);
		tree.getTree().getColumn(2).setToolTipText(UIMessages.CallTreeEditor_8);
		tree.getTree().getColumn(3).setToolTipText(UIMessages.CallTreeEditor_9);
		tree.getTree().getColumn(4).setToolTipText(UIMessages.CallTreeEditor_10);

		// Set columns movable
		for (int i = 1, n = tree.getTree().getColumnCount(); i < n; i++) {
			tree.getTree().getColumn(i).setMoveable(true);
		}

		// Pack the columns
		for (int i = 0, n = tree.getTree().getColumnCount(); i < n; i++) {
			tree.getTree().getColumn(i).pack();
		}
		tree.getTree().getColumn(0).setWidth(350);
		// Turn on the header and the lines
		tree.getTree().setHeaderVisible(true);
		tree.getTree().setLinesVisible(true);

		tree.addDoubleClickListener(new IDoubleClickListener() {

			public void doubleClick(DoubleClickEvent evt) {
				// TODO : Not yet supported on LS
				// showSource();
			}
		});

		// create the contextual menu
		calltreeMenu = new Menu(tree.getTree().getShell(), SWT.POP_UP);
		// create expand node menu item
		MenuItem expandItem = new MenuItem(calltreeMenu, SWT.CASCADE);

		expandItem.setText(UIMessages.CallTreeEditor_12);
		expandItem.addListener(SWT.Selection, new Listener() {

			public void handleEvent(Event event) {
				TreeItem[] treeItemsSelection = tree.getTree().getSelection();

				if (treeItemsSelection == null || treeItemsSelection.length == 0) {
					return;
				}

				tree.expandToLevel(treeItemsSelection[0].getData(), AbstractTreeViewer.ALL_LEVELS);
			}
		});
		// create expand all nodes menu item
		MenuItem expandAllItem = new MenuItem(calltreeMenu, SWT.PUSH);

		expandAllItem.setText(UIMessages.CallTreeEditor_0);
		expandAllItem.addListener(SWT.Selection, new Listener() {

			public void handleEvent(Event event) {
				tree.expandAll();
			}
		});
		// add a separator
		new MenuItem(calltreeMenu, SWT.BAR);
		// create collapse all menu item
		MenuItem collapseAllItem = new MenuItem(calltreeMenu, SWT.PUSH);

		collapseAllItem.setText(UIMessages.CallTreeEditor_11);
		collapseAllItem.addListener(SWT.Selection, new Listener() {

			public void handleEvent(Event event) {
				tree.collapseAll();
			}
		});

		tree.getTree().setMenu(calltreeMenu);
		setInputTreeViewer();

		// Register for dispose events
		tree.getTree().addDisposeListener(new DisposeListener() {
			public void widgetDisposed(DisposeEvent e) {
				dispose();
			}
		});

	}

	/**
	 * Create a toolbar for the editor.
	 * 
	 * @param parent
	 *            the parent composite.
	 */
	private ToolBar createToolBar(Composite parent) {
		final ToolBar toolBar = new ToolBar(parent, SWT.FLAT | SWT.WRAP | SWT.RIGHT);
		drawConfigureTableButton(toolBar);
		drawExportToDotButton(toolBar);
		drawExcludeSymbolsButton(toolBar);
		
		return toolBar;
	}

	/**
	 * Creates a toolbar button that allows the user to configure the table.
	 * 
	 * @param toolBar
	 *            the editor's toolBar
	 * @return the ToolItem associated with the button
	 */
	private ToolItem drawConfigureTableButton(ToolBar toolBar) {
		final ToolItem configureColumnsToolItem = new ToolItem(toolBar, SWT.DROP_DOWN);
		URL configureColsUrl = FileLocator.find(Platform.getBundle(com.freescale.sa.ui.common.UIConstants.PLUGIN_ID),
				new Path(com.freescale.sa.ui.common.UIConstants.analysis_settings_iconfile), null); //$NON-NLS-1$

		configureColumnsToolItem.setImage(ImageDescriptor.createFromURL(configureColsUrl).createImage());
		configureColumnsToolItem.setToolTipText(UIMessages.CallTreeEditor_Configure_Table);
		// Create time unit settings menu.
		String storageFolder = flatProfiler.getAnalysisResultsPath().toOSString();
		String baseFileName = flatProfiler.getBaseName();
		String decodedTraceConfigPath = storageFolder + IPath.SEPARATOR + baseFileName + "." //$NON-NLS-1$
				+ UIConstants.decoded_trace_config_extension;
		m_timeUnitGroup = TimeUnitGroup.getGroupForConfig(decodedTraceConfigPath);
		m_timeUnitGroup.addParent((IConfigurableDisplay) this);
		m_timeUnitGroup.addParent((IRefreshableDisplay) this);

		configureColumnsToolItem.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				Shell shell = new Shell(event.display);
				MenuManager menuMgr = new MenuManager();
				Menu menu = menuMgr.createContextMenu(shell);

				m_timeUnitGroup.fillContextMenu(menuMgr);

				menu.setVisible(true);
			}
		});

		if (m_timeUnitGroup.getFreqSelection() == TimeUnitGroup.UNDEFINED_FREQUENCY) {
			configureColumnsToolItem.dispose();
		}

		return configureColumnsToolItem;
	}

	private ToolItem drawExportToDotButton(final ToolBar toolBar) {
		final ToolItem exportToDotToolItem = new ToolItem(toolBar, SWT.PUSH);
		URL exportToDotUrl = FileLocator.find(Platform.getBundle(UIConstants.PLUGIN_ID), //$NON-NLS-1$
				new Path(UIConstants.analysis_dot_iconfile), null); //$NON-NLS-1$		
		exportToDotToolItem.setImage(ImageDescriptor.createFromURL(exportToDotUrl).createImage());

		exportToDotToolItem.setToolTipText(UIMessages.CallTreeEditor_14);

		exportToDotToolItem.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				exportToDot(toolBar);
			}
		});
		return exportToDotToolItem;
	}

	private void drawExcludeSymbolsButton(ToolBar toolBar) {
		if (analyzer == null) {
			return;
		}

		final ToolItem excludeSymbolsItem = new ToolItem(toolBar, SWT.PUSH);
		URL excludeSymbolsUrl = FileLocator.find(Platform.getBundle(com.freescale.sa.ui.common.UIConstants.PLUGIN_ID), //$NON-NLS-1$
				new Path(com.freescale.sa.ui.common.UIConstants.analysis_exclude_symbols_iconfile), null); //$NON-NLS-1$
		excludeSymbolsItem.setImage(ImageDescriptor.createFromURL(excludeSymbolsUrl).createImage());

		excludeSymbolsItem.setToolTipText(UIMessages.CallTreeEditor_15);

		excludeSymbolsItem.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				IWorkbench workBench = PlatformUI.getWorkbench();
				IWorkbenchWindow wbWindow = workBench.getActiveWorkbenchWindow();

				String platformType = CommonConstants.EMPTY_STRING;
				if (!projectName.isEmpty()) {
					for (IProject project : ResourcesPlugin.getWorkspace().getRoot().getProjects()) {
						if (projectName.equals(project.getName())) {
							platformType = PlatformType.getPlatformType(project).toString();
							continue;
						}
					}
				} else {
					ILaunchConfiguration launchConfig = LaunchConfigUtils.getLaunchConfigByName(getBaseName());
					platformType = PlatformType.getPlatformType(launchConfig).toString();
				}
				ConfigureResultsDialog chooserDialog = new ConfigureResultsDialog(wbWindow.getShell(), platformType,
						analyzer.getArchName(), getBaseName(), storageDir, ConfigureResultsTabId.EXCLUDE_SYMBOLS_TAB);
				int retCode = chooserDialog.open();
				if (retCode == IDialogConstants.CANCEL_ID) {
					return;
				}
			}
		});
	}

	/**
	 * Public interface for dot exporting
	 * 
	 * @param filePath
	 *            The output file path
	 */
	public void exportDotToFile(String filePath) {
		if (filePath == null || filePath.isEmpty()) {
			return;
		}

		DotExport dotExport = new DotExport(contexts);
		dotExport.export(filePath);
	}

	public Collection<CallTreeContext> getContexts() {
		return contexts;
	}

	/**
	 * It exports CallTree statistics into dot format when UI is used.
	 * 
	 * @param toolBar
	 *            The editor's toolBar.
	 */
	public void exportToDot(ToolBar toolBar) {
		int response = SWT.NO;
		int filterIndex = 0;
		String selected = null;

		if (toolBar != null) {
			while (response == SWT.NO) {
				response = SWT.YES;

				selected = SaUiLsUtils.getExportFile(UIMessages.CallTreeEditor_14, toolBar.getShell(), new String[] {
						"*.dot", "*.gv", "*.*" }, filterIndex); //$NON-NLS-1$//$NON-NLS-2$ //$NON-NLS-3$
				if (selected == null) {
					return;
				}
				File file = new File(selected);

				if (file.exists()) {
					MessageBox messageBox = new MessageBox(toolBar.getShell(), SWT.ICON_QUESTION | SWT.YES | SWT.NO
							| SWT.CANCEL);

					messageBox.setMessage(UIMessages.CallTreeEditor_16);
					messageBox.setText(UIMessages.CallTreeEditor_17);
					response = messageBox.open();
					if (response == SWT.CANCEL) {
						return;
					}
				}
			}
		}

		exportDotToFile(selected);
	}

	public void setFocus() {
	}

	/**
	 * Initializes the list of contexts from the input file
	 * 
	 * @return The list of contexts found in file
	 */
	private ArrayList<CallTreeContext> getDataFromFile() {
		ArrayList<CallTreeContext> tempElements = new ArrayList<CallTreeContext>();

		SectionData section = flatProfiler.getSection(sectionName);

		for (SubsectionData subsection : section.values()) {
			tempElements.add(new CallTreeContext(flatProfiler, subsection));
		}

		return tempElements;
	}

	/**
	 * It refreshes tree with or without reloading data input.
	 * 
	 * @param reloadData
	 *            True for reload data input using new collected trace. False to
	 *            perform refresh on tree without reloading data input, only
	 *            changed items will be refreshed.
	 * @Note Make sure hashCode() and equals() methods are defined for objects
	 *       used by TreeContentProvider, i.e., Function objects used by
	 *       CallTreeContentProvider to provide the specified methods.
	 */
	public void refresh(boolean reloadData) {

		if (reloadData) {
			// Clear old data and load new one.
			flatProfiler.tableRecordsConfigurations.clear();
			flatProfiler.initializeConfiguration();
			tree.getTree().clearAll(true);
			contexts = getDataFromFile();

			// Get maximum call stack for each context
			for (CallTreeContext context : contexts) {
				context.setMaxCallStacks();
				context.setMaxSizeStack();
			}

			setInputTreeViewer();
		}

		TreeColumn[] columns = tree.getTree().getColumns();
		for (TreeColumn column : columns) {
			String colText = column.getText();
			if (colText.toLowerCase().contains(UIMessages.CallTreeEditor_5.toLowerCase())) {
				colText = TimeUnitUtils.getTimeColumnNameWithDescription(colText,
						m_timeUnitGroup.getTimeUnitSelection(), m_timeUnitGroup.getFreqSelection());
				column.setText(colText);
			}
		}

		refreshTreeViewer();
	}

	/**
	 * Wrapper over setInput method of org.eclipse.jface.viewers.TreeViewer.
	 * Before every setInput on tree, it marks tree as not having root in order
	 * to ensure the <START> label as root of the tree.
	 */
	private void setInputTreeViewer() {
		for (CallTreeContext context : contexts) {
			context.initRootFunction();
		}
		tree.setInput(contexts);
	}

	/**
	 * Wrapper over refresh method of org.eclipse.jface.viewers.TreeViewer.
	 * Before every refresh on tree, it marks tree as not having root in order
	 * to ensure the <START> label as root of the tree.
	 */
	private void refreshTreeViewer() {
		tree.refresh();
	}

	public FlatProfilerFileInput getFlatProfiler() {
		return flatProfiler;
	}

	public void setFlatProfiler(FlatProfilerFileInput flatProfiler) {
		this.flatProfiler = flatProfiler;
	}

	private String getArchiveName() {
		int lastDot = flatProfiler.m_fileName.lastIndexOf('.');
		StringBuilder archiveName = new StringBuilder();
		archiveName.append(flatProfiler.m_fileName.substring(0, lastDot));
		archiveName.append(UIConstants.ZIP_EXTENSION);
		return archiveName.toString();
	}

	private String getArchivedFile(String fileName) {
		// Get the file from the archive if the corresponding archive exists
		String archiveName = getArchiveName();
		File archivefile = new File(archiveName);
		if (archivefile.exists()) {
			// TODO : Not yet supported on LS
			// fileName = SourceFilesArchiver.getSourceFile(archiveName,
			// fileName);
		}

		return fileName;
	}

	@Override
	public void refreshDisplay() {
		refresh(false);
	}

	@Override
	public String getDisplayId() {
		return "Call Tree Display Tab"; //$NON-NLS-1$
	}

	@Override
	public void configure(CONFIG_ELEMENT elementType, String value) {
		/*
		 * This method is called when one of the config elements (e.g.: time
		 * unit, freq) is changed by the user form the UI, using the 'configure
		 * table' menu actions.
		 */

		switch (elementType) {
		case TIME_UNIT_ELM:
			configureTimeUnit(value);
			break;

		case FREQ_ELM:
			configureFrequency(value);
			break;

		default:
			LOGGER.error("[configure]: unknown CONFIG_ELEMENT: " + elementType); //$NON-NLS-1$
		}
	}

	/**
	 * Update the time unit in the viewer.
	 * 
	 * @param timeUnit
	 *            the new time unit (see types defined in
	 *            {@link IConfigurableDisplay}).
	 */
	private void configureTimeUnit(String timeUnit) {
		// Save it in the decoded trace configuration file.
		String storageFolder = flatProfiler.getAnalysisResultsPath().toOSString();
		String baseFileName = flatProfiler.getBaseName();

		if (storageFolder != null && baseFileName != null) {
			String decodedTraceConfigPath = storageFolder + IPath.SEPARATOR + baseFileName + "." //$NON-NLS-1$
					+ UIConstants.decoded_trace_config_extension;
			TimeUnitUtils.saveTimeUnit(decodedTraceConfigPath, timeUnit);
		}
	}

	/**
	 * Convert the frequency value from string to double and update its value.
	 * 
	 * @param value
	 *            The frequency value, as a string.
	 */
	private void configureFrequency(String value) {
		double freq = -1;
		try {
			freq = Double.parseDouble(value);
		} catch (NumberFormatException e) {
			e.printStackTrace();
		}

		// Save it in the decoded trace configuration file.
		String storageFolder = flatProfiler.getAnalysisResultsPath().toOSString();
		String baseFileName = flatProfiler.getBaseName();

		if (storageFolder != null && baseFileName != null) {
			String decodedTraceConfigPath = storageFolder + IPath.SEPARATOR + baseFileName + "." //$NON-NLS-1$
					+ UIConstants.decoded_trace_config_extension;
			TimeUnitUtils.saveFrequency(decodedTraceConfigPath, freq);
		}
	}

	/**
	 * Destroys internal components
	 */
	void dispose() {
		if (m_timeUnitGroup != null) {
			m_timeUnitGroup.removeParent((IConfigurableDisplay) this);
			m_timeUnitGroup.removeParent((IRefreshableDisplay) this);
		}

		if ((bookmark != null) && (bookmark.exists())) {
			try {
				bookmark.delete();
			} catch (CoreException e) {
				e.printStackTrace();
			}
		}

		editor = null;
	}

	private void setMaxStacks() {
		for (CallTreeContext context : contexts) {
			context.setMaxCallStacks();
			context.setMaxSizeStack();
		}
	}

	public Composite getComposite() {
		return m_composite;
	}

	private String getBaseName() {
		if (flatProfiler == null) {
			return null;
		}

		return flatProfiler.getBaseName();
	}

	public void setStorageDir(String storageDir) {
		this.storageDir = storageDir;
	}
}
