/**
 * Copyright 2017-2019 NXP
 */
package com.nxp.swtools.periphs.gui.view.provider;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.controller.Messages;
import com.nxp.swtools.periphs.model.data.mcu.IMcu;
import com.nxp.swtools.provider.configuration.sources.ISourceFile;
import com.nxp.swtools.provider.configuration.sources.SourceFileProviderFactory;
import com.nxp.swtools.utils.scripting.ScriptResult;
import com.nxp.swtools.utils.sources.ISourcesContentProvider;

/**
 * Class used for providing content to the Sources View.
 * @author Juraj Ondruska
 */
public class SourcesViewProvider implements ISourcesContentProvider {

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.sources.ISourcesContentProvider#getContent()
	 */
	@Override
	public @NonNull Collection<@NonNull ISourceFile> getContent() {
		Controller controller = Controller.getInstance();
		ScriptResult scriptResult = controller.getGeneratedCode();
		Map<@NonNull String, @NonNull String> code = new LinkedHashMap<>();
		if (scriptResult != null) {
			code.putAll(scriptResult.getOpenedFiles()); // add all core-unrelated files
			IMcu mcu = controller.getMcu();
			for (String core : scriptResult.getCores()) {
				Map<@NonNull String, @NonNull String> coreFiles = scriptResult.getOpenedFiles(core);
				if (coreFiles != null) {
					for (Entry<@NonNull String, @NonNull String> entry : coreFiles.entrySet()) {
						@Nullable String newKey = entry.getKey();
						if (mcu.getCores().size() > 1) { // is multi-core?
							newKey += UtilsText.AT_SIGN + mcu.getCoreUiName(core);
						}
						code.put(newKey, entry.getValue());
					}
				}
			}
		}
		return SourceFileProviderFactory.buildSourceFileList(code);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.sources.ISourcesContentProvider#getErrorText()
	 */
	@Override
	public @Nullable String getErrorText() {
		final Controller controller = Controller.getInstance();
		if (!controller.isTurnedOn() || controller.isGeneratingCode()) {
			return null;
		}
		ScriptResult scriptResult = controller.getGeneratedCode();
		String mcuError = controller.getToolError();
		if (mcuError != null) {
			return mcuError;
		}
		return ((scriptResult != null) && !scriptResult.isFailed())
				? scriptResult.getErrors()
				: com.nxp.swtools.periphs.controller.Messages.get().Controller_SourcesGeneration_CannotGenerateDueToInternalError;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.sources.ISourcesContentProvider#getWarningText()
	 */
	@Override
	public @Nullable String getWarningText() {
		final Controller controller = Controller.getInstance();
		if (!controller.isTurnedOn() || controller.isGeneratingCode()) {
			return null;
		}
		ScriptResult scriptResult = controller.getGeneratedCode();
		return scriptResult != null ? scriptResult.getWarnings() : null;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.sources.ISourcesContentProvider#isOutOfSync()
	 */
	@Override
	public boolean isOutOfSync() {
		return Controller.getInstance().isGeneratingCode();
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.sources.ISourcesContentProvider#getStatus()
	 */
	@Override
	public @Nullable String getStatus() {
		final Controller controller = Controller.getInstance();
		if (!controller.isTurnedOn() || controller.isGeneratingCode()) {
			return null;
		}
		ScriptResult scriptResult = controller.getGeneratedCode();
		String mcuError = controller.getToolError();
		if (mcuError != null) {
			return mcuError;
		} else if ((scriptResult == null) || scriptResult.isFailed()) {
			return Messages.get().Controller_SourcesGeneration_CannotGenerateDueToInternalError;
		} else if (!UtilsText.isEmpty(scriptResult.getErrors())) {
			return Messages.get().SourcesViewProvider_SourcesGeneration_GeneratedWithErrors;
		} else if (!UtilsText.isEmpty(scriptResult.getWarnings())) {
			return Messages.get().SourcesViewProvider_SourcesGeneration_GeneratedWithWarnings;
		}
		return UtilsText.EMPTY_STRING;
	}

}
