/**
 * Copyright 2017-2018 NXP
 */
package com.nxp.swtools.periphs.gui.view.provider;

import java.util.Collection;
import java.util.logging.Logger;

import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.configuration.SwToolsProduct;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.controller.PeripheralsProblem;
import com.nxp.swtools.periphs.gui.view.componentsettings.ComponentSettingView;
import com.nxp.swtools.periphs.model.config.FunctionalGroup;
import com.nxp.swtools.validation.engine.IBaseProblem;
import com.nxp.swtools.validation.engine.IProblemOpenHandler;
import com.nxp.swtools.validation.engine.IProblemOpenProvider;

/**
 * Class provides open handlers for peripherals problems.
 * @author Juraj Ondruska
 */
public class PeripheralsProblemOpenProvider implements IProblemOpenProvider {
	/** Logger of the class */
	static final @NonNull Logger LOGGER = LogManager.getLogger(PeripheralsProblemOpenProvider.class);
	/** The handlers */
	private static final @NonNull Collection<@NonNull IProblemOpenHandler> HANDLERS =
			CollectionsUtils.asList(new PeripheralsProblemOpenHandler());

	/* (non-Javadoc)
	 * @see com.nxp.swtools.validation.engine.IProblemOpenProvider#getOpenHandlers(com.nxp.swtools.validation.engine.IBaseProblem)
	 */
	@Override
	public @NonNull Collection<@NonNull IProblemOpenHandler> getOpenHandlers(@NonNull IBaseProblem problem) {
		if ((problem instanceof PeripheralsProblem) && (((PeripheralsProblem) problem).getComponentType() != null)) {
			// return handlers only for peripherals problem which has the component type
			return HANDLERS;
		} else {
			return CollectionsUtils.emptyList();
		}
	}

	/**
	 * Class used for opening view for peripherals problems.
	 * @author Juraj Ondruska
	 */
	private static class PeripheralsProblemOpenHandler implements IProblemOpenHandler {
		
		/**
		 * Constructor.
		 */
		PeripheralsProblemOpenHandler() {
			// Do not allow instantiating from outside
		}

		/* (non-Javadoc)
		 * @see com.nxp.swtools.validation.engine.IProblemOpenHandler#openProblem(com.nxp.swtools.validation.engine.IBaseProblem)
		 */
		@Override
		public void openProblem(@NonNull IBaseProblem problem) {
			if (problem instanceof PeripheralsProblem) {
				openTool(SwToolsProduct.PRODUCT_ID_PERIPHERALS_TOOL);
				PeripheralsProblem peripheralsProblem = (PeripheralsProblem) problem;
				String componentType = peripheralsProblem.getComponentType();
				String componentName = peripheralsProblem.getComponentInstanceName();
				String functionalGroupName = peripheralsProblem.getFunctionalGroupName();
				if (componentType != null) {
					// Switch functional group when needed. Fires listeners for functional group change event
					FunctionalGroup group = Controller.getInstance().getProfile().getFunctionalGroup(UtilsText.safeString(functionalGroupName));
					if (group != null) {
						if (!Controller.getInstance().getFunctionalGroup().getUiName().equals(functionalGroupName)) {
							// If current functional group is different then switch to group of this problem
							Controller.getInstance().setFunctionalGroup(group, this);
						}
					}
					// This opens required component setting view after functional group switches and everything is ready
					Display display = Display.getCurrent();
					assert display != null;
					display.asyncExec(new Runnable() {
						@Override
						public void run() {
							IWorkbenchWindow activeWorkbenchWindow = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
							if (activeWorkbenchWindow != null) {
								IWorkbenchPage activePage = activeWorkbenchWindow.getActivePage();
								if (activePage != null) {
									ComponentSettingView.open(activePage, componentType, UtilsText.safeString(componentName), componentName == null, true);
								} else {
									LOGGER.warning("Cannot open component setting view because there is no active page"); //$NON-NLS-1$
								}
							} else{
								LOGGER.warning("Cannot open component setting view because there is no active workbench window"); //$NON-NLS-1$
							}
						}
					});
				}
			} else {
				LOGGER.warning(PeripheralsProblemOpenHandler.class.getSimpleName() + " accepts only " //$NON-NLS-1$
						+ PeripheralsProblem.class.getSimpleName());
			}
		}

		/* (non-Javadoc)
		 * @see com.nxp.swtools.validation.engine.IProblemOpenHandler#getTargetToolId()
		 */
		@Override
		public @Nullable String getTargetToolId() {
			return SwToolsProduct.PRODUCT_ID_PERIPHERALS_TOOL;
		}
		
	}
	
}
