/**
 * Copyright 2017-2019 NXP
 */
package com.nxp.swtools.periphs.gui.view.provider;

import java.text.MessageFormat;
import java.util.List;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.controller.PeriphControllerWrapper;
import com.nxp.swtools.periphs.gui.view.PeripheralsView;
import com.nxp.swtools.periphs.model.config.ComponentInstanceConfig;
import com.nxp.swtools.periphs.model.config.FunctionalGroup;
import com.nxp.swtools.provider.configuration.dependencies.IDependency;
import com.nxp.swtools.provider.configuration.storage.StorageDependencyFeature;
import com.nxp.swtools.validation.engine.IBaseProblem;
import com.nxp.swtools.validation.engine.IValidationProblem;

/**
 * Problem handler used for responding to peripheral uninitialized problems.
 * @author Juraj Ondruska
 */
public final class PeripheralInitResolveHandler implements IPeripheralsProblemResolveHandler {
	/** Singleton instance of the class */
	public static final @NonNull PeripheralInitResolveHandler INSTANCE = new PeripheralInitResolveHandler();

	/** Associated controller wrapper*/
	protected final @NonNull IControllerWrapper controllerWrapper = PeriphControllerWrapper.getInstance();
	/**
	 * Private constructor avoids creating instances from outside.
	 */
	private PeripheralInitResolveHandler() {
		// intentionally empty
	}
	
	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.provider.IProblemResolveHandlerExt#canHandle(com.nxp.swtools.validation.engine.IBaseProblem)
	 */
	@Override
	public boolean canHandle(@NonNull IBaseProblem problem) {
		if (problem instanceof IValidationProblem) {
			IValidationProblem validationProblem = (IValidationProblem) problem;
			if (IDependency.RESOURCE_PERIPHERAL.equals(validationProblem.getDependency().getResourceType())) {
				return StorageDependencyFeature.FEATURE_INITIALIZED.equals(validationProblem.getFeature().getName());
			}
		}
		return false;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.validation.engine.IProblemResolveHandler#resolveProblem(com.nxp.swtools.validation.engine.IBaseProblem)
	 */
	@Override
	public boolean resolveProblem(@NonNull IBaseProblem problem) {
		if (problem instanceof IValidationProblem) {
			IValidationProblem validationProblem = (IValidationProblem) problem;
			String peripheralInstance = validationProblem.getDependency().getResourceId();
			List<@NonNull ComponentInstanceConfig> componentInstanceConfigs = controllerWrapper.getController().getComponentInstanceConfigs(peripheralInstance);
			if (!componentInstanceConfigs.isEmpty()) {
				controllerWrapper.getController().setComponentInstancesEnabled(componentInstanceConfigs, true, getClass());
				return true;
			}
			String functionalGroupName = validationProblem.getConfiguration().getConfigurationName();
			FunctionalGroup functionalGroup = Controller.getInstance().getProfile().getFunctionalGroup(functionalGroupName);
			if (functionalGroup != null) {
				PeripheralsView.configurePeripheral(peripheralInstance, functionalGroup, this, controllerWrapper);
				return true;
			}
		}
		return false;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.validation.engine.IProblemResolveHandler#getResolveProblemLabel(com.nxp.swtools.validation.engine.IBaseProblem)
	 */
	@Override
	public @NonNull String getResolveProblemLabel(@NonNull IBaseProblem problem) {
		if (problem instanceof IValidationProblem) {
			IValidationProblem validationProblem = (IValidationProblem) problem;
			String peripheralInstance = validationProblem.getDependency().getResourceId();
			List<@NonNull ComponentInstanceConfig> componentInstanceConfigs = controllerWrapper.getController().getComponentInstanceConfigs(peripheralInstance);
			if (!componentInstanceConfigs.isEmpty()) {
				List<@NonNull String> instanceNames = componentInstanceConfigs.stream().map(x -> x.getName()).collect(CollectorsUtils.toList());
				return MessageFormat.format(Messages.get().PeripheralInitResolveHandler_Enable_Instances, instanceNames);
			}
			return MessageFormat.format(Messages.get().PeripheralInitResolveHandler_ConfigurePeripheral, peripheralInstance);
		}
		return UtilsText.EMPTY_STRING;
	}
}
