/**
 * Copyright 2017-2019 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;

import java.util.List;
import java.util.logging.Level;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.EditingSupport;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

import com.nxp.swtools.common.ui.utils.editors.CellEditorComboInstantSearchWithImages;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.configuration.properties.SWToolsProperties;
import com.nxp.swtools.periphs.controller.UiValueIdHolder;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl;
import com.nxp.swtools.periphs.model.config.ArrayConfig;
import com.nxp.swtools.periphs.model.config.IChild;
import com.nxp.swtools.periphs.model.config.ISettingConfig;
import com.nxp.swtools.periphs.model.config.ScalarConfig;
import com.nxp.swtools.periphs.model.config.ScalarConfig.Type;
import com.nxp.swtools.periphs.model.config.SetConfig.SetPresence;
import com.nxp.swtools.periphs.model.config.SettingConfigFactory;
import com.nxp.swtools.periphs.model.data.XMLConstants;
import com.nxp.swtools.periphs.model.data.setting.IBaseModel;
import com.nxp.swtools.utils.progress.ProgressUtils;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;

/**
 * Class representing tabular representation of an array config with items represented as rows.
 * @author Juraj Ondruska
 * @author Marek Ciz
 * @author Viktar Paklonski
 */
public class ArrayControlTabularVertical extends AArrayControlTabular {

	/**
	 * Constructor.
	 * @param arrayConfig for which to create the table
	 * @param controllerWrapper containing the generic controller
	 */
	protected ArrayControlTabularVertical(@NonNull ArrayConfig arrayConfig, @NonNull IControllerWrapper controllerWrapper) {
		super(arrayConfig, controllerWrapper);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#tableViewerHandling(org.eclipse.jface.viewers.TableViewer)
	 */
	@Override
	protected void tableViewerHandling(@NonNull TableViewer tableViewerLoc) {
		final Table table = tableViewerLoc.getTable();
		tableViewerLoc.addSelectionChangedListener(new ISelectionChangedListener() {
			@Override
			public void selectionChanged(SelectionChangedEvent event) {
				ISelection selection = event.getSelection();
				if (selection instanceof StructuredSelection) {
					Object selectedElement = ((StructuredSelection) selection).getFirstElement();
					if (selectedElement instanceof IChildControl) {
						IChildControl newSelection = (IChildControl) selectedElement;
						setSelectedChild(newSelection);
						updateButtons(UpdateType.NORMAL);
					}
				}
			}
		});
		// create the indices column only in case indices are allowed
		if (shouldShowIndices()) {
			createIndicesTableViewerColumn(tableViewerLoc);
		}
		// create some temporary fake settingConfig to retrieve proper column titles
		ISettingConfig childForHeaderLabels = SettingConfigFactory.createSettingConfig(UtilsText.EMPTY_STRING, UtilsText.EMPTY_STRING,
				arrayConfig.getModelData().getReferenceType(), arrayConfig.getChildContext(), arrayConfig.getChildContext().getProfile().getMcu());
		// create tableViewers columns based on the labels retrieved from the fake settingConfig
		for (@NonNull IChild childLoc : getSettingsFlat(childForHeaderLabels)) {
			IBaseModel baseModel = childLoc.getModelData();
			if ((baseModel != null) && (table != null)) {
				if (SWToolsProperties.isVerificationOn()) {
					if (baseModel.getUINameExprString() != null) {
						LOGGER.log(Level.WARNING, String.format("%1s is used as table column title from %2s; column titles are only updated on component (re-)open", //$NON-NLS-1$
								XMLConstants.LABEL_EXPR, baseModel));
					}
				}
				createTableViewerColumnVertical(baseModel, tableViewerLoc, shouldShowIndices());
			}
		}
		// remove the temporary fake settingConfig
		arrayConfig.remove(childForHeaderLabels);
		if (table != null) {
			// false -> not horizontal layout
			setTableLayout(table, arrayConfig.getId(), true);
			if (SWToolsProperties.isVerificationOn()) {
				if (arrayConfig.getModelData().getUINameExprString() != null) {
					LOGGER.log(Level.WARNING, String.format("%1s is used as table title from %2s; table titles are only updated on component (re-)open", //$NON-NLS-1$
							XMLConstants.LABEL_EXPR, arrayConfig));
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#getMenuContext(org.eclipse.jface.viewers.TableViewer)
	 */
	@Override
	protected @NonNull ArrayControlItemMenuContext getMenuContext(@NonNull TableViewer tableViewerLoc) {
		return new ArrayControlItemMenuContextVertical(this, tableViewerLoc);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlGroup#createLabelControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public Control createLabelControl(Composite composite) {
		return null;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#updateTableViewer(org.eclipse.jface.viewers.TableViewer)
	 */
	@Override
	protected void updateTableViewer(@NonNull TableViewer viewerLoc) {
		// input updated data
		viewerLoc.setInput(children);
		if (contentContainer != null) {
			contentContainer.requestLayout();
		}
		viewerLoc.getTable().setEnabled(arrayConfig.isEnabled());
		IChildControl childToSelect = getChildToSelect();
		if (childToSelect != null) {
			viewerLoc.setSelection(new StructuredSelection(childToSelect), true);
		}
	}

	/**
	 * Create table column from given configuration in given Vertical table
	 * @param baseModel of the newly created column
	 * @param tableViewerLoc in which the new column will be created
	 * @param isIndices if indices column is required
	 */
	protected void createTableViewerColumnVertical(@NonNull IBaseModel baseModel, @NonNull TableViewer tableViewerLoc, boolean isIndices) {
		final TableViewerColumn viewerColumn = new TableViewerColumn(tableViewer, SWT.NONE);
		viewerColumn.getColumn().addControlListener(new ControlAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.ControlAdapter#controlResized(org.eclipse.swt.events.ControlEvent)
			 */
			@Override
			public void controlResized(ControlEvent e) {
				updateScrollSize();
			}
		});
		// decide if indices column is generated
		final int rightOffset = isIndices ? 2 : 1;
		final int columnIndex = tableViewerLoc.getTable().getColumnCount() - rightOffset;
		viewerColumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				String shownText = null;
				IChild currentChild = getChild(element, columnIndex);
				// assign child for column listener
				if ((currentChild instanceof ScalarConfig)) {
					if (currentChild.isAvailable()) {
						ScalarConfig currentSettingLoc = (ScalarConfig) currentChild;
						shownText = AArrayControlTabular.getText(currentSettingLoc);
					} else {
						shownText = NOT_APPLICABLE;
					}
				}
				return shownText;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getFont(java.lang.Object)
			 */
			@Override
			public Font getFont(Object element) {
				IChild currentChild = getChild(element, columnIndex);
				if ((currentChild != null) && (!currentChild.isAvailable())) {
					return italicFont;
				}
				return null;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getImage(java.lang.Object)
			 */
			@Override
		    public Image getImage(Object element) {
				Image shownIcon = null;
				IChild currentChild = getChild(element, columnIndex);
				if (currentChild != null) {
					shownIcon = AArrayControlTabular.getImage(currentChild);
				}
				return shownIcon;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getForeground(java.lang.Object)
			 */
			@Override
			public Color getForeground(Object element) {
				IChild currentChild = getChild(element, columnIndex);
				if ((currentChild != null) && !currentChild.isEnabled()) {
					return DISABLED_TEXT_COLOR;
				}
				return super.getForeground(element);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getBackground(java.lang.Object)
			 */
			@Override
			public Color getBackground(Object element) {
				IChild currentChild = getChild(element, columnIndex);
				if (currentChild != null) {
					// decorate background with proper color
					return AArrayControlTabular.getBackgroundColor(currentChild);
				}
				return super.getBackground(element);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.CellLabelProvider#getToolTipText(java.lang.Object)
			 */
			@Override
			public String getToolTipText(Object element) {
				String tooltip = UtilsText.EMPTY_STRING;
				IChild currentChild = getChild(element, columnIndex);
				if (currentChild != null) {
					tooltip = ToolTipableFormatter.getToolTipText(new ToolTipableMarkdownDescriptionDecorator(currentChild));
				}
				return tooltip;
			}
		});
		viewerColumn.setEditingSupport(new EditingSupport(tableViewerLoc) {
			/* (non-Javadoc)
			 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.ASettingEditingSupport#getCellEditor(java.lang.Object)
			 */
			@Override
			protected @Nullable CellEditor getCellEditor(Object element) {
				CellEditor editor = null;
				IChild currentChild = getChild(element, columnIndex);
				if (currentChild instanceof ScalarConfig) {
					ScalarConfig currentSettingLoc = ((ScalarConfig) currentChild);
					// checkboxes options
					if (currentSettingLoc.getType() == Type.BOOL) {
						// change value of check box
						ProgressUtils.run((m) -> {
							controllerWrapper.getController().setValue(currentSettingLoc, currentSettingLoc.getValue().equals(Boolean.FALSE)
									? UtilsText.safeString(Boolean.TRUE.toString()) : UtilsText.safeString(Boolean.FALSE.toString()), tableViewerLoc);
						});
					} else if (currentSettingLoc.getType() == Type.ENUM) {
						List<@NonNull UiValueIdHolder> enumItems = controllerWrapper.getController().getEnumItems((ISettingConfig) currentChild);
						CellEditorComboInstantSearchWithImages editorCombo = new CellEditorComboInstantSearchWithImages(tableViewerLoc.getTable(), SWT.NONE);
						for (UiValueIdHolder enumItem : enumItems) {
							editorCombo.add(enumItem.getUiValue());
							editorCombo.addItemToolTip(enumItem.getUiValue(), enumItem.getDescription());
						}
						editor = editorCombo;
					} else if ((currentSettingLoc.getType() == Type.INTEGER) || (currentSettingLoc.getType() == Type.FLOAT)
							|| (currentSettingLoc.getType() == Type.STRING)) {
						editor = new TextCellEditor(tableViewerLoc.getTable());
					}
				} else if (currentChild instanceof SetPresence) {
					SetPresence setPresence = (SetPresence) currentChild;
					// change value of check box
					ProgressUtils.run((m) -> {
						controllerWrapper.getController().setSetPresence(setPresence, !setPresence.getBoolValue(), tableViewerLoc);
					});
				}
				return editor;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.EditingSupport#canEdit(java.lang.Object)
			 */
			@Override
			protected boolean canEdit(Object element) {
				boolean canEdit = true;
				if (element instanceof IChildControl){
					List<@NonNull IChild> childList = getSettingsFlat(((IChildControl) element).getChild());
					IChild currentChild = childList.get(columnIndex);
					canEdit = isCellEnabled(currentChild);
				}
				return canEdit;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.EditingSupport#getValue(java.lang.Object)
			 */
			@Override
			protected @Nullable Object getValue(@Nullable Object element) {
				String selectedItem = UtilsText.EMPTY_STRING;
				IChild currentChild = getChild(element, columnIndex);
				if ((currentChild instanceof ScalarConfig) && currentChild.isAvailable()) {
					selectedItem = AArrayControlTabular.getText((ScalarConfig) currentChild);
				}
				return selectedItem;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.EditingSupport#setValue(java.lang.Object, java.lang.Object)
			 */
			@Override
			protected void setValue(Object element, Object userInputValue) {
				IChild currentChild = getChild(element, columnIndex);
				if (currentChild instanceof ScalarConfig) {
					ScalarConfig currentSettingLoc = ((ScalarConfig) currentChild);
					// check-boxes options
					if (currentSettingLoc.getType() == Type.ENUM) {
						List<@NonNull UiValueIdHolder> enumItems = controllerWrapper.getController().getEnumItems(currentSettingLoc);
						if (userInputValue instanceof Integer) {
							int indexRow = ((Integer) userInputValue).intValue();
							if ((enumItems.size() >= indexRow) && (indexRow >= 0)) {
								ProgressUtils.run((m) -> {
									controllerWrapper.getController().setValue(currentSettingLoc, UtilsText.safeString(enumItems.get(indexRow).getId()), tableViewerLoc);
								});
							}
						}
					} else if ((currentSettingLoc.getType() == Type.INTEGER) || (currentSettingLoc.getType() == Type.FLOAT)
							|| (currentSettingLoc.getType() == Type.STRING)) {
						controllerWrapper.getController().setValue(currentSettingLoc,
								(userInputValue == null) ? UtilsText.EMPTY_STRING : UtilsText.safeString(userInputValue.toString()), tableViewerLoc);
					}
				}
			}
		});
		// set title to a column
		final TableColumn column = viewerColumn.getColumn();
		String title = baseModel.getUIName(arrayConfig.getExpressionContext());
		column.setText(title);
		column.setToolTipText(title);
		column.pack();
		setColumnWidth(column);
	}

	/**
	 * Get proper child from the element.
	 * @param element from the table's input
	 * @param column of the child
	 * @return child or {@code null} if there is no child for the current column in the element
	 */
	protected @Nullable IChild getChild(@Nullable Object element, int column) {
		if (element instanceof IChildControl){
			List<@NonNull IChild> childList = getSettingsFlat(((IChildControl) element).getChild());
			if ((column >= 0) && (column < childList.size())) {
				return childList.get(column);
			}
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#getSelection()
	 */
	@Override
	public ISettingConfig getSelection() {
		TableViewer tableViewerLoc = tableViewer;
		if (tableViewerLoc != null) {
			ISelection selection = tableViewerLoc.getSelection();
			return super.getChildFromSelection(selection);
		}
		return null;
	}
}
