/**
 * Copyright 2017-2018 NXP                 
 * Created: 15. 12. 2017
 */
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;

import java.util.ArrayList;
import java.util.List;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl;
import com.nxp.swtools.periphs.model.config.ArrayConfig;
import com.nxp.swtools.periphs.model.config.IChild;
import com.nxp.swtools.periphs.model.config.ISettingConfig;
import com.nxp.swtools.periphs.model.config.SettingConfigFactory;
import com.nxp.swtools.periphs.model.data.setting.IBaseModel;

/**
 * Class, which transforms horizontal IChildControl Lists into vertical ISettingConfig
 * @author Marek Ciz
 */
public class ArrayControlTabularHorizontalProvider {
	/** Already vertically transformed children */
	@NonNull List<@NonNull List<@NonNull IChild>> transformedChildren;
	/** Original children */
	@NonNull List<@NonNull IChildControl> children;
	/** Extracted UI values from original children */
	@NonNull List<@NonNull List<@NonNull IChild>> flattenedChildren;
	/** Class representing configuration of an array setting */
	@NonNull ArrayConfig arrayConfig;
	/** Menu control UI label in the component table */
	static final @NonNull String MENU_HEADER_TITLE = UtilsText.EMPTY_STRING;
	/** Header names that are displayed in table */
	@NonNull List<@NonNull String> displayedHeaderNames;
	/** List of objects, where are stored SettingConfig lists. Flag if current setting should be used for remove button */
	@NonNull List<@NonNull TabularHorizontalItem> horizontalItems;
	
	/**
	 * Constructor.
	 * @param arrayConfig Class representing configuration of an array setting
	 * @param children Original children
	 */
	protected ArrayControlTabularHorizontalProvider(@NonNull ArrayConfig arrayConfig, @NonNull List<@NonNull IChildControl> children) {
		transformedChildren = new ArrayList<>();
		flattenedChildren = new ArrayList<>();
		this.arrayConfig = arrayConfig;
		this.children = children;
		displayedHeaderNames = getHeaders();
		transformedChildren = getFilledChildren();
		horizontalItems = fillHorizontalItems();		
	}
	
	/**
	 * Creates transformed list structure
	 * @param rowCount total row count of a table
	 * @return empty list in required size
	 */
	private @NonNull static List<@NonNull List<@NonNull IChild>> createTransformedChildrenList(int rowCount) {
		@NonNull List<@NonNull List<@NonNull IChild>> transformedChildrenEmpty = new ArrayList<>();
		for (int i = 0; i < rowCount; i++) {
			transformedChildrenEmpty.add(new ArrayList<>());
		}
		return transformedChildrenEmpty;
	}
	
	/**
	 * Fill TabularHorizontalItems with proper values
	 * @return list of horizontal items
	 */
	private @NonNull List<@NonNull TabularHorizontalItem> fillHorizontalItems() {
		 @NonNull List<@NonNull TabularHorizontalItem> horizontalItemsLoc = new ArrayList<>();
		 @NonNull List<@NonNull String> headerNames = getHeaders();
		 int rowCount = headerNames.size() - 1;
		 for (int row = 0; row < rowCount; row++) {
			 // fill rows with proper data
			 horizontalItemsLoc.add(new TabularHorizontalItem(headerNames.get(row), transformedChildren.get(row)));
		 }
		 return horizontalItemsLoc;
	}
	
	/**
	 * Tabular Horizontal Items getter
	 * @return Tabular Horizontal Items
	 */
	@NonNull List<@NonNull TabularHorizontalItem> getHorizontalItems() {
		return horizontalItems;
	}
	
	/**
	 * Extracts ISettingConfig from IChildControl and add it to the list
	 * @return flattened list of children
	 */
	private @NonNull List<@NonNull List<@NonNull IChild>> getFlattenedChildren() {
		@NonNull List<@NonNull List<@NonNull IChild>> flattenChildrenLoc = new ArrayList<>();
		for (IChildControl childControl : children) {
			flattenChildrenLoc.add(AArrayControlTabular.getSettingsFlat(childControl.getChild()));
		}
		return flattenChildrenLoc;
	}
	
	/**
	 * Extracts list of table headers
	 * @return list of table columns(rows - in horizontal layout) headers
	 */
	@NonNull List<@NonNull String> getHeaders() {
		// create some temporary fake settingConfig to retrieve proper column titles
		@NonNull ISettingConfig childForHeaderLabels = SettingConfigFactory.createSettingConfig(UtilsText.EMPTY_STRING, UtilsText.EMPTY_STRING,
				arrayConfig.getModelData().getReferenceType(), arrayConfig.getChildContext(),
				arrayConfig.getChildContext().getProfile().getMcu());
		
		// original headers, that should be displayed in the first column
		@NonNull List<@NonNull String> originalHeaderNames = new ArrayList<>();
		for (IChild originalHeader : AArrayControlTabular.getSettingsFlat(childForHeaderLabels)) {
			IBaseModel baseModel = originalHeader.getModelData();
			if (baseModel != null) {
				originalHeaderNames.add(baseModel.getUIName(arrayConfig.getExpressionContext()));
			}
		}	
		// also add menu column title
		originalHeaderNames.add(MENU_HEADER_TITLE);
		// remove fake settings
		arrayConfig.remove(childForHeaderLabels);
		return originalHeaderNames;
	}
	
	/**
	 * Fills vertically transformed children list with extracted Settings data
	 * @return transformed children list
	 */
	private @NonNull List<@NonNull List<@NonNull IChild>> getFilledChildren() {
		@NonNull List<@NonNull List<@NonNull IChild>> transformedChildrenLoc = getChildren();
		// rowCount stores how many rows are in a new table
		int rowCount = displayedHeaderNames.size() - 1;
		transformedChildrenLoc = createTransformedChildrenList(rowCount);
		flattenedChildren = getFlattenedChildren();
		// transform horizontal list to a vertical one
		if (!flattenedChildren.isEmpty()) {
			for (int row = 0; row < rowCount; row++) {
				for (int col = 0; col < children.size(); col++) {
					transformedChildrenLoc.get(row).add(flattenedChildren.get(col).get(row));
				}
			}
		}
		return transformedChildrenLoc;
	}
	
	/**
	 * @return list of transformed children
	 */
	protected @NonNull List<@NonNull List<@NonNull IChild>> getChildren() {
		return transformedChildren;
	}
}
