/**
 * Copyright 2017-2019 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;


import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.EditingSupport;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

import com.nxp.swtools.common.ui.utils.editors.CellEditorComboInstantSearchWithImages;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.controller.UiValueIdHolder;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl;
import com.nxp.swtools.periphs.model.config.ArrayConfig;
import com.nxp.swtools.periphs.model.config.IChild;
import com.nxp.swtools.periphs.model.config.ISettingConfig;
import com.nxp.swtools.periphs.model.config.ScalarConfig;
import com.nxp.swtools.periphs.model.config.ScalarConfig.Type;
import com.nxp.swtools.periphs.model.config.SetConfig.SetPresence;
import com.nxp.swtools.utils.progress.ProgressUtils;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;

/**
 * Class representing tabular representation of an array config with items represented as columns.
 * @author Juraj Ondruska
 * @author Marek Ciz
 */
public class ArrayControlTabularHorizontal extends AArrayControlTabular {

	/**
	 * Constructor.
	 * @param arrayConfig for which to create the GUI
	 * @param controllerWrapper containing the generic controller
	 */
	protected ArrayControlTabularHorizontal(@NonNull ArrayConfig arrayConfig, @NonNull IControllerWrapper controllerWrapper) {
		super(arrayConfig, controllerWrapper);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#tableViewerHandling(org.eclipse.jface.viewers.TableViewer)
	 */
	@Override
	protected void tableViewerHandling(@NonNull TableViewer tableViewerLoc) {
		final Table table = tableViewerLoc.getTable();
		if (table != null) {
			// true - for horizontal layout
			setTableLayout(table, arrayConfig.getId(), false);
			table.setHeaderVisible(true);
		}
		// create header column
		createTableViewerHeaderColumn(tableViewerLoc);
		if (children.size() > 0) {
			for (int column = 0; column < children.size(); column++) {
				createTableViewerColumnHorizontal(column, tableViewerLoc);
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#getMenuContext(org.eclipse.jface.viewers.TableViewer)
	 */
	@Override
	protected @NonNull ArrayControlItemMenuContext getMenuContext(@NonNull TableViewer tableViewerLoc) {
		return new ArrayControlItemMenuContextHorizontal(this, tableViewerLoc);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#updateTableViewer(org.eclipse.jface.viewers.TableViewer)
	 */
	@Override
	protected void updateTableViewer(@NonNull TableViewer viewerLoc) {
		// check, if there are enough created columns to be populated with all the children
		int childrenSize = children.size();
		// there is -1 because of header column
		int columnCount = viewerLoc.getTable().getColumnCount() - 1;
		if (childrenSize > columnCount) {
			// create columns for newly added children
			for (int index = columnCount; index < childrenSize; index++) {
				createTableViewerColumnHorizontal(index, viewerLoc);
			}
		} else if (childrenSize < columnCount) {
			// dispose columns for newly added children
			for (int index = columnCount; index > childrenSize; index--) {
				viewerLoc.getTable().getColumn(index).dispose();
			}
		}
		// update header
		if (shouldShowIndices()) {
			for (int index = viewerLoc.getTable().getColumnCount() - 1; index > 0; --index) {
				final IChildControl element = children.get(index - 1);
				final ISettingConfig config = (ISettingConfig) element.getChild();
				final String key = arrayConfig.getSettingKey(config, arrayConfig.getExpressionContext());
				final String columnTitle = (key != null) ? key : element.getChild().getUiName();
				viewerLoc.getTable().getColumn(index).setText(columnTitle);
			}
		}
		// transform the children from horizontal to vertical lists
		ArrayControlTabularHorizontalProvider transformedChildrenProvider = new ArrayControlTabularHorizontalProvider(arrayConfig, children);
		// input updated data
		viewerLoc.setInput(transformedChildrenProvider.getHorizontalItems());
		// update header size
		final TableColumn headerColumn = viewerLoc.getTable().getColumn(0);
		headerColumn.pack();
		// update table window size
		if (contentContainer != null) {
			contentContainer.requestLayout();
		}
		viewerLoc.getTable().setEnabled(arrayConfig.isEnabled());
	}

	/**
	 * Provides current Setting of the cell
	 * @param itemLoc includes SettingsConfig, removeButton flag, and header names
	 * @param columnPosition column index in the setting table
	 * @return current Setting of the cell
	 */
	@Nullable IChild getChildFromItem(@NonNull TabularHorizontalItem itemLoc, int columnPosition){
		IChild currentChild = null;
		// current setting is extracted from list of @Nonnull ISettings, so there should not be a null ISetting
		if (children.size() > 0) {
			currentChild = itemLoc.getChildConfigs().get(columnPosition);
		}
		assert currentChild != null;
		return currentChild;
	}


	/**
	 * Create table column from given configuration in given Horizontal table
	 * @param columnPosition current position of the column
	 * @param tableViewerLoc tableViewer in which the new column will be created
	*/
	protected void createTableViewerColumnHorizontal(int columnPosition, @NonNull TableViewer tableViewerLoc) {
		final TableViewerColumn viewerColumn = new TableViewerColumn(tableViewerLoc, SWT.NONE);
		viewerColumn.getColumn().addControlListener(new ControlAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.ControlAdapter#controlResized(org.eclipse.swt.events.ControlEvent)
			 */
			@Override
			public void controlResized(ControlEvent e) {
				updateScrollSize();
			}
		});
		viewerColumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				String label = null;
				IChild currentChild = getChild(element, columnPosition);
				if ((currentChild instanceof ScalarConfig)) {
					if (currentChild.isAvailable()) {
						ScalarConfig currentSettingLoc = (ScalarConfig) currentChild;
						label = AArrayControlTabular.getText(currentSettingLoc);
					} else {
						label = NOT_APPLICABLE;
					}
				}
				return label;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getFont(java.lang.Object)
			 */
			@Override
			public Font getFont(Object element) {
				IChild currentChild = getChild(element, columnPosition);
				if ((currentChild != null) && (!currentChild.isAvailable())) {
					return italicFont;
				}
				return null;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getImage(java.lang.Object)
			 */
			@Override
			public Image getImage(Object element) {
				Image image = null;
				IChild currentChild = getChild(element, columnPosition);
				if (currentChild != null) {
					image = AArrayControlTabular.getImage(currentChild);
				}
				return image;
			}

			@Override
			public Color getForeground(Object element) {
				IChild currentChild = getChild(element, columnPosition);
				if ((currentChild != null) && !currentChild.isEnabled()) {
					return DISABLED_TEXT_COLOR;
				}
				return super.getForeground(element);
			}

			@Override
			public Color getBackground(Object element) {
				IChild currentChild = getChild(element, columnPosition);
				if (currentChild != null) {
					Color backgroundColor = AArrayControlTabular.getBackgroundColor(currentChild);
					// FIXME v6 TomasR This could be used to draw selection of column instead of row
//					if (backgroundColor == null) {
//						IChild selection = getSelection();
//						// XXX Hack for child inside of structure. Selected control is structure, cell being drawn contains its child
//						if (selection instanceof IChildProvidable) {
//							IChildProvidable providable = (IChildProvidable) selection;
//							if (providable.getChildren().contains(currentChild)) {
//								selection = providable.getChildren().get(providable.getChildren().indexOf(currentChild));
//							}
//						}
//						if (currentChild.equals(selection)) {
//							return viewerColumn.getColumn().getDisplay().getSystemColor(SWT.COLOR_LIST_SELECTION);
//						}
//					}
					return backgroundColor;
				}
				return super.getBackground(element);
			}

			@Override
			public String getToolTipText(Object element) {
				String tooltip = UtilsText.EMPTY_STRING;
				IChild currentChild = getChild(element, columnPosition);
				if (currentChild != null) {
					tooltip = ToolTipableFormatter.getToolTipText(new ToolTipableMarkdownDescriptionDecorator(currentChild));
				}
				return tooltip;
			}
		});

		viewerColumn.setEditingSupport(new EditingSupport(tableViewerLoc) {
			/* (non-Javadoc)
			 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.ASettingEditingSupport#getCellEditor(java.lang.Object)
			 */
			@Override
			protected @Nullable CellEditor getCellEditor(Object element) {
				CellEditor editor = null;
				IChild currentChild = getChild(element, columnPosition);
				if (currentChild instanceof ScalarConfig) {
					ScalarConfig currentSettingLoc = ((ScalarConfig) currentChild);
					// check-boxes options
					if (currentSettingLoc.getType().equals(Type.BOOL)) {
						// change value of check box
						ProgressUtils.run((m) -> {
							controllerWrapper.getController().setValue(currentSettingLoc, currentSettingLoc.getValue().equals(Boolean.FALSE)
									? UtilsText.safeString(Boolean.TRUE.toString()) : UtilsText.safeString(Boolean.FALSE.toString()), tableViewerLoc);
						});
						// instant search lists
					} else if (currentSettingLoc.getType().equals(Type.ENUM)) {
						List<@NonNull UiValueIdHolder> enumItems = controllerWrapper.getController().getEnumItems((ISettingConfig) currentChild);
						CellEditorComboInstantSearchWithImages editorCombo = new CellEditorComboInstantSearchWithImages(tableViewerLoc.getTable(), SWT.NONE);
						for (UiValueIdHolder enumItem : enumItems) {
							editorCombo.add(enumItem.getUiValue());
							editorCombo.addItemToolTip(enumItem.getUiValue(), enumItem.getDescription());
						}
						editor = editorCombo;
					} else if (currentSettingLoc.getType().equals(Type.INTEGER) || currentSettingLoc.getType().equals(Type.FLOAT)
							|| currentSettingLoc.getType().equals(Type.STRING)) {
						editor = new TextCellEditor(tableViewerLoc.getTable());
					}
				} else if (currentChild instanceof SetPresence) {
					SetPresence setPresence = (SetPresence) currentChild;
					// change value of check box
					ProgressUtils.run((m) -> {
						controllerWrapper.getController().setSetPresence(setPresence, !setPresence.getBoolValue(), tableViewerLoc);
					});
				}
				TableColumn columnLoc = viewerColumn.getColumn();
				if (columnLoc != null) {
					selectChildOfColumn(columnLoc);
				}
				return editor;
			}

			/**
			 * Selects correct child based on selected column
			 * @param column column that is selected
			 */
			private void selectChildOfColumn(@NonNull TableColumn column) {
				List<TableColumn> columnsList = Arrays.asList(tableViewerLoc.getTable().getColumns());
				int indexOfColumn = columnsList.indexOf(column);
				if (indexOfColumn != -1) {
					IChildControl childToSet = getChildren().get(indexOfColumn - 1);
					setSelectedChild(childToSet);
				}
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.EditingSupport#canEdit(java.lang.Object)
			 */
			@Override
			protected boolean canEdit(Object element) {
				boolean canEdit = true;
				IChild currentChild = getChild(element, columnPosition);
				if (currentChild != null) {
					canEdit = isCellEnabled(currentChild);
				}
				return canEdit;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.EditingSupport#getValue(java.lang.Object)
			 */
			@Override
			protected @Nullable Object getValue(Object element) {
				String selectedItem = null;
				IChild currentChild = getChild(element, columnPosition);
				if ((currentChild != null) && (currentChild.isAvailable())) {
					if (currentChild instanceof ScalarConfig) {
						selectedItem = AArrayControlTabular.getText((ScalarConfig) currentChild);
					}
				}
				return selectedItem;
			}

			/*
			 * (non-Javadoc)
			 * @see org.eclipse.jface.viewers.EditingSupport#setValue(java.lang.Object, java.lang.Object)
			 */
			@Override
			protected void setValue(Object element, Object userInputValue) {
				IChild currentChild = getChild(element, columnPosition);
				if (currentChild instanceof ScalarConfig) {
					ScalarConfig currentSettingLoc = ((ScalarConfig) currentChild);
					// instant search list options
					if (currentSettingLoc.getType().equals(Type.ENUM)) {
						List<@NonNull UiValueIdHolder> enumItems = controllerWrapper.getController().getEnumItems(currentSettingLoc);
						if (userInputValue instanceof Integer) {
							int indexRow = ((Integer) userInputValue).intValue();
							if ((enumItems.size() > indexRow) && (indexRow >= 0)) {
								ProgressUtils.run((m) -> {
									controllerWrapper.getController().setValue(currentSettingLoc, UtilsText.safeString(enumItems.get(indexRow).getId()), tableViewerLoc);
								});
							}
						}
					} else if (currentSettingLoc.getType().equals(Type.INTEGER) || currentSettingLoc.getType().equals(Type.FLOAT)
							|| currentSettingLoc.getType().equals(Type.STRING)) {
						controllerWrapper.getController().setValue(currentSettingLoc,
								(userInputValue == null) ? UtilsText.EMPTY_STRING : UtilsText.safeString(userInputValue.toString()), tableViewerLoc);
					}
				}
			}
		});

		String title = UtilsText.EMPTY_STRING;
		// set title and width to a column
		final TableColumn column = viewerColumn.getColumn();
		// hide or show column numbers in the top
		if (shouldShowIndices()) {
			title = Integer.toString(columnPosition);
			column.setText(title);
			column.setToolTipText(title);
		}
		column.pack();
		setColumnWidth(column);
		if (column.getWidth() > AArrayControlTabular.COLUMN_MAX_WIDTH) {
			column.setWidth(AArrayControlTabular.COLUMN_MAX_WIDTH);
		}
	}

	/**
	 * Creates header Column
	 * @param tableViewerLoc table's table viewer
	 */
	protected void createTableViewerHeaderColumn(@NonNull TableViewer tableViewerLoc) {
		final TableViewerColumn headerViewerColumn = new TableViewerColumn(tableViewerLoc, SWT.NONE);
		final TableColumn headerColumn = headerViewerColumn.getColumn();
		headerColumn.addControlListener(new ControlAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.ControlAdapter#controlResized(org.eclipse.swt.events.ControlEvent)
			 */
			@Override
			public void controlResized(ControlEvent e) {
				updateScrollSize();
			}
		});
		if (shouldShowIndices()) {
			headerColumn.setText("#"); //$NON-NLS-1$
		}
		headerColumn.setAlignment(SWT.LEFT);
		headerViewerColumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				String header = UtilsText.EMPTY_STRING;
				if (element instanceof TabularHorizontalItem) {
					TabularHorizontalItem item = (TabularHorizontalItem) element;
					header = item.getHeaderName();
				}
				return header;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getFont(java.lang.Object)
			 */
			@Override
			public Font getFont(Object element) {
				return boldFont;
			}
		});
		if (headerColumn.getWidth() > AArrayControlTabular.COLUMN_MAX_WIDTH) {
			headerColumn.setWidth(AArrayControlTabular.COLUMN_MAX_WIDTH);
		}
	}

	/**
	 * Get proper child from the element.
	 * @param element from the table's input
	 * @param column of the child
	 * @return child or {@code null} if there is no child for the current column in the element
	 */
	protected @Nullable IChild getChild(@Nullable Object element, int column) {
		if (element instanceof TabularHorizontalItem) {
			TabularHorizontalItem item = (TabularHorizontalItem) element;
			if (item.getChildConfigs() instanceof ArrayList<?>) {
				return getChildFromItem(item, column);
			}
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#getSelection()
	 */
	@Override
	public @Nullable ISettingConfig getSelection() {
		IChildControl selectedChildLoc = getSelectedChild();
		if (selectedChildLoc != null) {
			return super.getChildFromSelection(new StructuredSelection(selectedChildLoc));
		}
		return null;
	}
}
