/**
 * Copyright 2019 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.function.Function;

import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;

import com.nxp.swtools.common.ui.utils.swt.widgets.RadioButtonsGroup;
import com.nxp.swtools.common.ui.utils.swt.widgets.RadioButtonsGroupVertical;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.lang.CollectionsUtils.Pair;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlFactory;
import com.nxp.swtools.periphs.gui.view.componentsettings.ComponentSettingView;
import com.nxp.swtools.periphs.gui.view.componentsettings.ControlOptions;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl;
import com.nxp.swtools.periphs.model.config.ArrayConfig;
import com.nxp.swtools.periphs.model.config.IChild;
import com.nxp.swtools.periphs.model.config.ISettingConfig;
import com.nxp.swtools.periphs.model.config.StructConfig;
import com.nxp.swtools.periphs.model.data.setting.ISetting;
import com.nxp.swtools.periphs.model.data.setting.StructSetting;

/**
 * Class represents array of radio button groups. Usable only for enum or structure of enums
 * @author Tomas Rudolf
 */
public class ArrayControlRadioHorizontal extends ArrayControlRadio {
	/** Number of columns in the item composite */
	public static final int GUI_ITEM_COLS = ComponentSettingView.CONFIG_SET_COLS + 1;

	/**
	 * Constructor.
	 * @param arrayConfig for which to create the GUI
	 * @param controllerWrapper containing the generic controller
	 */
	protected ArrayControlRadioHorizontal(@NonNull ArrayConfig arrayConfig, @NonNull IControllerWrapper controllerWrapper) {
		super(arrayConfig, controllerWrapper);
	}

	@Override
	protected void createContent(@NonNull Composite contentCompositeLoc, final @NonNull List<@NonNull String> names, final @NonNull List<@NonNull String> ids) {
		// Show GUI
		for (int itemIndex = 0; itemIndex < children.size(); itemIndex++) {
			createItem(contentCompositeLoc, names, ids, itemIndex);
		}
	}

	/**
	 * Creates item in given composite with given names, ids and index
	 * @param composite composite in which to create radio button group
	 * @param names list of names for labels
	 * @param ids list of ids to be used as values
	 * @param itemIndex index for newly created item
	 */
	private void createItem(@NonNull Composite composite, final List<@NonNull String> names, final List<@NonNull String> ids, int itemIndex) {
		RadioButtonsGroup group = new RadioButtonsGroupVertical(composite, SWT.NONE);
		group.setNumberOfButtons(names.size());
		final int finalItemIndex = itemIndex;
		String finalItemIndexString = String.valueOf(finalItemIndex);
		Function<Integer, String> keySelectorSupplier = (column) -> {
			// Key_selector
			IChild childById = getChild().getChild(finalItemIndexString);
			if (childById instanceof ISettingConfig) {
				return getChild().getSettingKey((ISettingConfig) childById, childById.getExpressionContext());
			}
			// Or column number if something fails
			return finalItemIndexString;
		};
		group.setHorizontalHeaderSupplier(keySelectorSupplier);
		if (itemIndex == 0) {
			group.setHorizontalHeaderSupplier((column) -> {
				int intValue = column.intValue();
				if (intValue == 0) {
					return UtilsText.EMPTY_STRING;
				}
				return keySelectorSupplier.apply(column);
			});
			group.setVerticalHeaderSupplier((line) -> names.get(line.intValue()));
		}
		group.setLayoutData(new GridDataComponents(SWT.CENTER, SWT.CENTER, false, false, 1, names.size()));
		group.addListener(SWT.Selection, new Listener() {
			@Override
			public void handleEvent(@NonNull Event event) {
				int selectedButtonIndex = event.detail;
				String idToSet = UtilsText.safeString(ids.get(selectedButtonIndex));
				changeValue(getConfigById(finalItemIndex), idToSet);
			}
		});
		groupsMap.put(group, getChild().getChildren().get(itemIndex));
		updateGroup(ids, group);
	}
	

	/**
	 * @param ids
	 * @param group
	 */
	private void updateGroup(final List<@NonNull String> ids, @NonNull RadioButtonsGroup group) {
		String selectedValue = UtilsText.EMPTY_STRING;
		int whichChildOfStructure = getWhichChildOfStructure();
		ISettingConfig config = groupsMap.get(group);
		if (config != null) {
			if (whichChildOfStructure == NOT_STRUCTURED_SETTING_INDEX) {
				selectedValue = config.getValueName();
			} else {
				StructConfig structConfig = (StructConfig) config;
				List<ISettingConfig> settings = new ArrayList<>(structConfig.getChildren());
				selectedValue = settings.get(whichChildOfStructure).getValueName();
			}
		}
		int selectedIndex = ids.indexOf(selectedValue);
		if (selectedIndex == -1) {
			selectedIndex = 0;
		}
		group.setCurrentSelection(selectedIndex, true);
		group.update();
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.ArrayControlRadio#updateMainCommonImpl(com.nxp.swtools.periphs.gui.view.
	 * componentsettings.IChildControl.UpdateType, org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected boolean updateMainCommonImpl(@NonNull UpdateType updateType, @NonNull Composite composite) {
		Iterator<@NonNull Entry<RadioButtonsGroup, ISettingConfig>> groupsIterator = groupsMap.entrySet().iterator();
		Iterator<@NonNull ISettingConfig> arrayItemsIterator = getChild().getChildren().iterator();
		@Nullable
		ISettingConfig arrayItem = arrayItemsIterator.hasNext() ? arrayItemsIterator.next() : null;
		// dispose controls of all removed items
		while (groupsIterator.hasNext()) {
			@SuppressWarnings("null") // For analysis purpose only! Entry is always non-null or exception is thrown
			Entry<RadioButtonsGroup, ISettingConfig> entry = groupsIterator.next();
			ISettingConfig config = entry.getValue();
			RadioButtonsGroup group = entry.getKey();
			if (!config.equals(arrayItem)) {
				// Remove child
				groupsIterator.remove();
				group.dispose();
				IChildControl childToRemove = CollectionsUtils.nullableOptionalGet(children.stream()
					.filter(x -> x.getChild().equals(config)).findAny());
				if (childToRemove != null) {
					childToRemove.dispose();
					children.remove(childToRemove);
				}
			} else { // match
				arrayItem = arrayItemsIterator.hasNext() ? arrayItemsIterator.next() : null;
			}
		}
		// update all the children
		for (Entry<@NonNull RadioButtonsGroup, @NonNull ISettingConfig> entry : groupsMap.entrySet()) {
			Pair<List<@NonNull String>, List<@NonNull String>> namesAndIds = getNamesAndIds(entry.getValue());
			updateGroup(namesAndIds.getSecond(), entry.getKey());
		}
		// add newly added items to the GUI
		while (arrayItem != null) {
			IChildControl currentChildControl = ChildControlFactory.create(arrayItem, new ControlOptions(), controllerWrapper);
			if (currentChildControl != null) {
				Pair<List<@NonNull String>, List<@NonNull String>> namesAndIds = getNamesAndIds(arrayItem);
				children.add(currentChildControl);
				createItem(composite, namesAndIds.getFirst(), namesAndIds.getSecond(), children.size() - 1);
			}
			arrayItem = arrayItemsIterator.hasNext() ? arrayItemsIterator.next() : null;
		}
		return false;
	}

	/**
	 * Returns pair of list with names and list with ids of given config
	 * @param config to get names and ids from
	 * @return pair with list of names and list of ids
	 */
	private @NonNull Pair<List<@NonNull String>, List<@NonNull String>> getNamesAndIds(@NonNull ISettingConfig config) {
		List<@NonNull String> names = new ArrayList<>();
		List<@NonNull String> ids = new ArrayList<>();
		int whichChildOfStructure = getWhichChildOfStructure();
		ISetting modelData = config.getModelData();
		if (whichChildOfStructure == -1) {
			names.addAll(getNames(modelData));
			ids.addAll(getIds(modelData));
		} else {
			StructSetting structSetting = (StructSetting) modelData;
			List<ISetting> settings = new ArrayList<>(structSetting.getSettings().values());
			ISetting setting = settings.get(whichChildOfStructure);
			if (setting != null) {
				names.addAll(getNames(setting));
				ids.addAll(getIds(setting));
			}
		}
		return new Pair<>(names, ids);
	}
}
