/**
 * Copyright 2019 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;

import com.nxp.swtools.common.ui.utils.swt.widgets.RadioButtonsGroup;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.model.config.ArrayConfig;
import com.nxp.swtools.periphs.model.config.IChild;
import com.nxp.swtools.periphs.model.config.ISettingConfig;
import com.nxp.swtools.periphs.model.config.ScalarConfig;
import com.nxp.swtools.periphs.model.config.StructConfig;
import com.nxp.swtools.periphs.model.data.setting.BoolSetting;
import com.nxp.swtools.periphs.model.data.setting.EnumSetting;
import com.nxp.swtools.periphs.model.data.setting.ISetting;
import com.nxp.swtools.periphs.model.data.setting.StructSetting;

/**
 * Class represents array of radio button groups. Usable only for enum, boolean or structure of enums or booleans
 * @author Tomas Rudolf - nxf31690
 */
public abstract class ArrayControlRadio extends AArrayControlInternal {
	/** Index used in case of not structured setting of array */
	public static final int NOT_STRUCTURED_SETTING_INDEX = -1;
	/** Composite with the content */
	protected @Nullable Composite contentComposite;
	/** Labels used for boolean setting */
	private static List<@NonNull String> BOOL_LABELS = new ArrayList<>();
	/** Values used for boolean setting */
	private static List<@NonNull String> BOOL_IDS = new ArrayList<>();
	/** Stores information which child of structure should be shown as independent array */
	private int whichChildOfStructure = NOT_STRUCTURED_SETTING_INDEX;
	/** Links to other copies of this class created because of structure was given as input */
	private List<ArrayControlRadio> otherChildrenOfStructure = new ArrayList<>();
	/** Map containing radio buttons group widget as key to its config */
	Map<@NonNull RadioButtonsGroup, @NonNull ISettingConfig> groupsMap = new LinkedHashMap<>();

	static {
		BOOL_LABELS.add("True"); //$NON-NLS-1$
		BOOL_LABELS.add("False"); //$NON-NLS-1$
		BOOL_IDS.add("true"); //$NON-NLS-1$
		BOOL_IDS.add("false"); //$NON-NLS-1$
	}

	/**
	 * Constructor.
	 * @param arrayConfig for which to create the GUI
	 * @param controllerWrapper containing the generic controller
	 */
	protected ArrayControlRadio(@NonNull ArrayConfig arrayConfig, @NonNull IControllerWrapper controllerWrapper) {
		super(arrayConfig, controllerWrapper);
	}

	/**
	 * Sets which child of structure should be used
	 * @param index of child in structure
	 */
	public void setWhichChildOfStructure(int index) {
		this.whichChildOfStructure = index;
	}

	/**
	 * Returns which child of structure should be used
	 * @return index of child in structure
	 */
	public int getWhichChildOfStructure() {
		return whichChildOfStructure;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#create(org.eclipse.swt.widgets.Composite, int)
	 */
	@Override
	public void create(@NonNull Composite composite, int colSpan) {
		ISetting settingInsideArray = getChild().getInstanceSetting();
		if (!shouldShowMoreInstances()) {
			super.create(composite, colSpan);
		} else {
			// Create other settings
			if (settingInsideArray instanceof StructSetting) {
				StructSetting structSetting = (StructSetting) settingInsideArray;
				LinkedHashMap<@NonNull String, @NonNull ISetting> settings = structSetting.getSettings();
				// First create this control with first setting
				setWhichChildOfStructure(0);
				super.create(composite, colSpan);
				// Create rest of controls in parent composite
				for (int childIndex = 1; childIndex < settings.size(); childIndex++) {
					ArrayControlRadio radioGroup = new ArrayControlRadioHorizontal(getChild(), getControllerWrapper());
					otherChildrenOfStructure.add(radioGroup);
					radioGroup.setWhichChildOfStructure(childIndex);
					radioGroup.create(composite, colSpan);
				}
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#update(com.nxp.swtools.periphs.gui.view.componentsettings.
	 * IChildControl.UpdateType)
	 */
	@Override
	public void update(@NonNull UpdateType updateType) {
		super.update(updateType);
		otherChildrenOfStructure.forEach(x -> x.update(updateType));
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#updateLabelContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	protected void updateLabelContent(@NonNull Control label, @NonNull UpdateType updateType) {
		super.updateLabelContent(label, updateType);
		if (label instanceof Label) {
			Label labelLoc = (Label) label;
			if (getWhichChildOfStructure() == NOT_STRUCTURED_SETTING_INDEX) {
				super.updateLabelContent(label, updateType);
			} else {
				StructSetting structSetting = (StructSetting) getChild().getInstanceSetting();
				List<ISetting> settings = new ArrayList<>(structSetting.getSettings().values());
				String uiName = settings.get(getWhichChildOfStructure()).getUINameString();
				labelLoc.setText(uiName);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#createLabelControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @Nullable Control createLabelControl(@NonNull Composite composite) {
		return null;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.ArrayControlStandard#createMainControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @NonNull Control createMainControl(@NonNull Composite composite) {
		createLabelWithControls(composite);
		// create main horizontally scroll-able area
		ScrolledComposite scrolledComposite = new ScrolledComposite(composite, SWT.H_SCROLL | SWT.BORDER);
		scrolledComposite.setExpandHorizontal(true);
		scrolledComposite.setExpandVertical(true);
		// create the content in the content composite
		Composite contentCompositeLoc = createComposite(scrolledComposite, getColumnCount());
		contentCompositeLoc.setLayoutData(new GridDataComponents());
		contentComposite = contentCompositeLoc;

		ISetting settingInsideArray = getChild().getInstanceSetting();
		List<@NonNull String> names = new ArrayList<>();
		List<@NonNull String> ids = new ArrayList<>();
		if (getWhichChildOfStructure() == NOT_STRUCTURED_SETTING_INDEX) {
			// Single setting array
			names.addAll(getNames(settingInsideArray));
			ids.addAll(getIds(settingInsideArray));
		} else {
			// Structure setting array
			if (settingInsideArray instanceof StructSetting) {
				StructSetting structSetting = (StructSetting) settingInsideArray;
				ArrayList<ISetting> settings = new ArrayList<>(structSetting.getSettings().values());
				ISetting settingInStructure = settings.get(getWhichChildOfStructure());
				if (settingInStructure != null) {
					names.addAll(getNames(settingInStructure));
					ids.addAll(getIds(settingInStructure));
				}
			}
		}
		// Show GUI
		if (names.isEmpty()) {
			Label label = new Label(contentCompositeLoc, SWT.BOLD);
			label.setText("No usable settings found for this type of representation"); //$NON-NLS-1$
		} else {
			createContent(contentCompositeLoc, names, ids);
		}
		// configure the scrolled composite to display the content properly
		scrolledComposite.setContent(contentCompositeLoc);
		scrolledComposite.setMinSize(contentCompositeLoc.computeSize(SWT.DEFAULT, SWT.DEFAULT));
		// add listener for proper resizing of the scrolled composite
		contentCompositeLoc.addListener(SWT.Resize, e -> {
			scrolledComposite.setMinSize(contentCompositeLoc.computeSize(SWT.DEFAULT, SWT.DEFAULT));
			scrolledComposite.requestLayout();
		});
		return scrolledComposite;
	}

	/**
	 * Returns number of columns required to show content
	 * @return number of columns
	 */
	private int getColumnCount() {
		return children.size() + (isUiArrayFixedSpecified() ? 0 : 1) + 1 /* Left header */;
	}

	/**
	 * Check whether it is needed to show multiple array controls for inner setting of this array
	 * @return {@code true} if inner setting is structure and has more than one setting inside, otherwise returns {@code false}
	 */
	private boolean shouldShowMoreInstances() {
		// Ends recursion for sub-children of structure
		if (getWhichChildOfStructure() != NOT_STRUCTURED_SETTING_INDEX) {
			return false;
		}
		ISetting settingInsideArray = getChild().getInstanceSetting();
		if (settingInsideArray instanceof StructSetting) {
			StructSetting structSetting = (StructSetting) settingInsideArray;
			LinkedHashMap<@NonNull String, @NonNull ISetting> settings = structSetting.getSettings();
			return settings.size() > 1;
		}
		return false;
	}

	/**
	 * Implementation of create main control in classes that extend this class
	 * @param composite composite in which the content should be created
	 * @param names list of names
	 * @param ids list of ids
	 */
	protected abstract void createContent(@NonNull Composite composite, final @NonNull List<@NonNull String> names, final @NonNull List<@NonNull String> ids);

	/**
	 * Returns list of names of given setting
	 * @param setting enumeration or boolean setting
	 * @return list of names of given setting or empty list for unsupported type of setting
	 */
	protected static List<@NonNull String> getNames(@NonNull ISetting setting) {
		ArrayList<@NonNull String> names = new ArrayList<>();
		if (setting instanceof EnumSetting) {
			EnumSetting enumSetting = (EnumSetting) setting;
			names.addAll(enumSetting.getItems().stream().map(x -> UtilsText.safeString(x.getUINameString())).collect(CollectorsUtils.toList()));
		}
		if (setting instanceof BoolSetting) {
			names.addAll(BOOL_LABELS);
		}
		return names;
	}

	/**
	 * Returns list of ids of given setting
	 * @param setting enumeration or boolean setting
	 * @return list of ids of given setting or empty list for unsupported type of setting
	 */
	protected static List<@NonNull String> getIds(@NonNull ISetting setting) {
		ArrayList<@NonNull String> names = new ArrayList<>();
		if (setting instanceof EnumSetting) {
			EnumSetting enumSetting = (EnumSetting) setting;
			names.addAll(enumSetting.getItems().stream().map(x -> x.getId()).collect(CollectorsUtils.toList()));
		}
		if (setting instanceof BoolSetting) {
			names.addAll(BOOL_IDS);
		}
		return names;
	}

	/**
	 * Changes value of given config
	 * @param configToChange config for change
	 * @param value value which should be set
	 * @return {@code true} if value was changed successfully or {@code false} if not. Returns {@code false} when given config is not scalar
	 */
	protected boolean changeValue(@Nullable ISettingConfig configToChange, @NonNull String value) {
		if (configToChange instanceof ScalarConfig) {
			ScalarConfig scalarConfig = (ScalarConfig) configToChange;
			return getControllerWrapper().getController().setValue(scalarConfig, value, this.getClass());
		}
		return false;
	}

	/**
	 * Implementation of update main control in classes that extend this class
	 * @param updateType type of update
	 * @param composite composite in which the update occurs
	 * @return {@code true} if something changed, otherwise returns {@code false}
	 */
	protected abstract boolean updateMainCommonImpl(@NonNull UpdateType updateType, @NonNull Composite composite);

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#updateMainContent(org.eclipse.swt.widgets.Control,
	 * com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	public void updateMainContent(@NonNull Control contentControl, @NonNull UpdateType updateType) {
		// FIXME TomasR v6 This code is copied. Use better suited super class
		ScrolledComposite scrolledComposite = (ScrolledComposite) contentControl;
		Composite contentCompositeLoc = contentComposite;
		if (contentCompositeLoc != null) {
			boolean layoutChange = updateMainCommonImpl(updateType, contentCompositeLoc);
			((GridLayoutComponents) contentCompositeLoc.getLayout()).numColumns = getColumnCount();
			// refresh GUI in case of some change
			if (layoutChange) {
				contentCompositeLoc.requestLayout();
				scrolledComposite.setMinSize(contentCompositeLoc.computeSize(SWT.DEFAULT, SWT.DEFAULT));
			}
		}
		updateErrorDecoration(contentControl);

		// Update others which were created by this class
		for (ArrayControlRadio other : otherChildrenOfStructure) {
			Control mainControlLoc = other.mainControl;
			if (mainControlLoc != null) {
				other.updateMainContent(mainControlLoc, updateType);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#updateButtons(com.nxp.swtools.periphs.gui.view.
	 * componentsettings.IChildControl.UpdateType)
	 */
	@Override
	protected void updateButtons(@Nullable UpdateType updateType) {
		updateAddButton(updateType);
		updateRemoveButton(updateType);
		updateUpButton(updateType);
		updateDownButton(updateType);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#setAllSettingsTo(com.nxp.swtools.periphs.gui.view.
	 * componentsettings.internal.ArrayControlItemMenu, com.nxp.swtools.periphs.gui.view.componentsettings.internal.IArrayControlItemMenuControl)
	 */
	@Override
	public void setAllSettingsTo(@NonNull ArrayControlItemMenu caller, @NonNull IArrayControlItemMenuControl control) {
		// Not supported in this GUI representation
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#getSelection()
	 */
	@Override
	public @Nullable ISettingConfig getSelection() {
		// Always return the last one
		List<@NonNull ISettingConfig> arrayChildren = getChild().getChildren();
		return arrayChildren.get(arrayChildren.size() - 1);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#dispose()
	 */
	@Override
	public void dispose() {
		super.dispose();
		// Reset this value to unselected/single setting because of function create can be called multiple times
		setWhichChildOfStructure(NOT_STRUCTURED_SETTING_INDEX);
		groupsMap.clear();
		for (ArrayControlRadio other : otherChildrenOfStructure) {
			other.dispose();
		}
		otherChildrenOfStructure.clear();
	}

	/**
	 * Returns config with given id or {@code null}
	 * @param index of config to be returned
	 * @return config if config with given id exists or {@code null} if it does not exist
	 */
	public @Nullable ISettingConfig getConfigById(int index) {
		IChild configToChange = getChild().getChild(String.valueOf(index));
		if (configToChange instanceof ScalarConfig) {
			ScalarConfig scalarConfig = (ScalarConfig) configToChange;
			return scalarConfig;
		}
		if (configToChange instanceof StructConfig) {
			StructConfig structConfig = (StructConfig) configToChange;
			// FIXME TomasR v6 Remove this when order of configs is same as order of settings
			List<ISetting> settings = new ArrayList<>(structConfig.getModelData().getSettings().values());
			ISetting setting = settings.get(getWhichChildOfStructure());
			ISettingConfig configInStructure = CollectionsUtils
					.nullableOptionalGet(structConfig.getSettings().values().stream().filter(x -> x.getId().endsWith(setting.getId())).findFirst());
			return configInStructure;
		}
		return null;
	}
}
