/**
* Copyright 2018-2019 NXP
* Created: Feb 6, 2018
*/
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.view.componentsettings.ArrayControl;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType;
import com.nxp.swtools.periphs.model.config.IChild;
import com.nxp.swtools.periphs.model.config.ISettingConfig;

/**
 * A pop-up menu for items in {@link ArrayControl}
 * @author Viktar Paklonski
 */
public class ArrayControlItemMenu {
	/** instance owner */
	protected @NonNull AArrayControlInternal owner;
	/** menu control */
	protected @NonNull IArrayControlItemMenuControl control;
	/** pop-up menu */
	protected @NonNull Menu menu;

	/**
	 * Prepares menu for the specified widget.
	 * @param owner of the instance
	 * @param control of the menu
	 * @param parent a control which will be the parent of the new menu instance
	 */
	public ArrayControlItemMenu(@NonNull AArrayControlInternal owner, @NonNull IArrayControlItemMenuControl control, @NonNull Control parent) {
		this.owner = owner;
		this.control = control;
		menu = new Menu(parent);
		update(UpdateType.NORMAL);
	}

	/**
	 * Destroys private members.
	 */
	public void dispose() {
		menu.dispose();
	}

	/**
	 * @return {@link #menu}
	 */
	public @NonNull Menu getMenu() {
		return menu;
	}

	/**
	 * Sets menu location under the specified bounds and marks it as visible.
	 * @param bounds of the control
	 * @param parent of the control
	 * @param display of the widget
	 */
	public void showUnder(@NonNull Rectangle bounds, @NonNull Control parent, @NonNull Display display) {
		menu.setLocation(display.map(parent, null, new Point(
				bounds.x, bounds.y + bounds.height)));
        menu.setVisible(true);
	}

	/**
	 * Refreshes menu button and action availability.
	 * @param updateType or {@code null} if update type is unknown
	 */
	@SuppressWarnings("unused") //to avoid warning, that result of new MenuItem is not used
	public void update(@Nullable UpdateType updateType) {
		for (MenuItem menuItem : menu.getItems()) {
			menuItem.dispose();
		}
		final boolean allowed = (updateType != UpdateType.FORCE_DISABLE);
		final boolean first = owner.isControlFirst(control.getSelectedItemHint());
		final boolean last = owner.isControlLast(control.getSelectedItemHint());
		// create menu actions
		MenuItem moveUpItem = createMenuItem(() -> owner.moveItemFront(this, control),
				UtilsText.safeString(Messages.get().ArrayControl_ItemMenu_MoveUp));
		MenuItem moveDownItem = createMenuItem(() -> owner.moveItemBack(this, control),
				UtilsText.safeString(Messages.get().ArrayControl_ItemMenu_MoveDown));
		MenuItem moveTopItem = createMenuItem(() -> owner.moveItemBeginning(this, control),
				UtilsText.safeString(Messages.get().ArrayControl_ItemMenu_MoveTop));
		MenuItem moveBottomItem = createMenuItem(() -> owner.moveItemEnd(this, control),
				UtilsText.safeString(Messages.get().ArrayControl_ItemMenu_MoveBottom));
		if (control.getSelectedSettingHint() != null) {
			new MenuItem(menu, SWT.SEPARATOR);
			MenuItem setToValueItem = createMenuItem(() -> owner.setAllSettingsTo(this, control),
					UtilsText.safeString(Messages.get().ArrayControl_ItemMenu_SetAllItems));
			IChild selectedSetting = control.getSelectedSettingHint();
			setToValueItem.setEnabled((selectedSetting != null) && selectedSetting.isAvailable());
		}
		new MenuItem(menu, SWT.SEPARATOR);
		MenuItem removeItem = createMenuItem(() -> owner.removeItem(this, control),
				UtilsText.safeString(Messages.get().ArrayControl_ItemMenu_Remove));
		ISettingConfig currentItemConfig = owner.getCurrentItemConfig(control);
		if (currentItemConfig != null) {
			MenuItem copyItem = createMenuItem(() -> owner.getControllerWrapper().getController().copy(currentItemConfig),
					UtilsText.safeString(Messages.get().ChildControl_Copy));
			copyItem.setEnabled(allowed && owner.getControllerWrapper().getController().canCopy(currentItemConfig));
			MenuItem pasteItem = createMenuItem(() -> owner.getControllerWrapper().getController().paste(currentItemConfig),
				UtilsText.safeString(Messages.get().ChildControl_Paste));
			pasteItem.setEnabled(allowed && owner.getControllerWrapper().getController().canPaste(currentItemConfig));
		}
		// set proper enable state
		moveUpItem.setEnabled(allowed && owner.isUiArrayReorderSpecified() && !first);
		moveDownItem.setEnabled(allowed && owner.isUiArrayReorderSpecified() && !last);
		moveTopItem.setEnabled(allowed && owner.isUiArrayReorderSpecified() && !first);
		moveBottomItem.setEnabled(allowed && owner.isUiArrayReorderSpecified() && !last);
		removeItem.setEnabled(allowed && !owner.isUiArrayFixedSpecified() && owner.getChild().canRemoveItem());
	}

	/**
	 * Creates and appends a new item to the menu
	 * @param method associated action
	 * @param text user-friendly description
	 * @return menu item
	 */
	private @NonNull MenuItem createMenuItem(Runnable method, @NonNull String text) {
		final MenuItem action = new MenuItem(menu, SWT.PUSH);
		action.setText(text);
		action.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				method.run();
			}
		});
		return action;
	}
}
