/**
 * Copyright 2017-2019 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.model.config.ArrayConfig;
import com.nxp.swtools.periphs.model.data.SettingOptions;

/**
 * Factory of the {@link AArrayControlInternal} implementations.
 * @author Juraj Ondruska
 */
public class ArrayControlInternalFactory {

	/**
	 * Create proper {@link AArrayControlInternal} for the array configuration given its options.
	 * {@link SettingOptions#UI_ARRAY_LAYOUT_HORIZONTAL} and {@link SettingOptions#SHOW_CONTENT_AS} are considered.
	 * @param arrayConfig for which to create the GUI
	 * @param controllerWrapper containing the generic controller
	 * @return implementation of an {@link AArrayControlInternal}
	 */
	public static @NonNull AArrayControlInternal create(@NonNull ArrayConfig arrayConfig, @NonNull IControllerWrapper controllerWrapper) {
		if (shouldShowAsTable(arrayConfig)) {
			if (shouldLayoutHorizontally(arrayConfig)) {
				return new ArrayControlTabularHorizontal(arrayConfig, controllerWrapper);
			} else {
				return new ArrayControlTabularVertical(arrayConfig, controllerWrapper);
			}
		} else if (shouldShowAsMasterDetail(arrayConfig)) {
			return new ArrayControlMasterDetail(arrayConfig, controllerWrapper);
		} else if (shouldShowAsRadioButtons(arrayConfig)) {
			return new ArrayControlRadioHorizontal(arrayConfig, controllerWrapper);
		} else {
			if (shouldLayoutHorizontally(arrayConfig)) {
				return new ArrayControlStandardHorizontal(arrayConfig, controllerWrapper);
			} else {
				return new ArrayControlStandardVertical(arrayConfig, controllerWrapper);
			}
		}
	}

	/**
	 * Gets ShowAs option from given array config
	 * @param arrayConfig to look in
	 * @return content of ShowAs option or empty string if this option is not set
	 */
	private static @NonNull String getShowAsOption(@NonNull ArrayConfig arrayConfig) {
		Object optionValue = arrayConfig.getModelData().getOptionValue(SettingOptions.SHOW_CONTENT_AS, arrayConfig.getExpressionContext());
		if (optionValue instanceof String) {
			String showContentAs = (String) optionValue;
			return showContentAs;
		}
		return UtilsText.EMPTY_STRING;
	}

	/**
	 * Checks whether the array should be laid out as a table.
	 * @param arrayConfig to check
	 * @return {@code true} if the array should be laid out as a table, {@code false} otherwise
	 */
	private static boolean shouldShowAsTable(@NonNull ArrayConfig arrayConfig) {
		return SettingOptions.VALUE_SHOW_CONTENT_AS_TABLE.equals(getShowAsOption(arrayConfig));
	}

	/**
	 * Checks whether the array should be laid out as a master-detail.
	 * @param arrayConfig to check
	 * @return {@code true} if the array should be laid out as a master-detail, {@code false} otherwise
	 */
	private static boolean shouldShowAsMasterDetail(@NonNull ArrayConfig arrayConfig) {
		return SettingOptions.VALUE_SHOW_CONTENT_AS_MASTER_DETAIL.equals(getShowAsOption(arrayConfig));
	}

	/**
	 * Checks whether the array should be laid out as a radio buttons.
	 * @param arrayConfig to check
	 * @return {@code true} if the array should be laid out as a radio buttons, {@code false} otherwise
	 */
	private static boolean shouldShowAsRadioButtons(@NonNull ArrayConfig arrayConfig) {
		return SettingOptions.VALUE_SHOW_CONTENT_AS_RADIO_GROUP.equals(getShowAsOption(arrayConfig));
	}

	/**
	 * Checks whether the array should be laid out horizontally.
	 * @param arrayConfig  to check
	 * @return whether the array should be laid out horizontally
	 */
	private static boolean shouldLayoutHorizontally(@NonNull ArrayConfig arrayConfig) {
		return arrayConfig.isOptionSet(SettingOptions.UI_ARRAY_LAYOUT_HORIZONTAL);
	}
}
