/**
 * Copyright 2017-2019 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.logging.Logger;

import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

import com.nxp.swtools.common.ui.utils.swt.FontFactory;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.os.OSDetect;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.APeriphsViewBase;
import com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlFactory;
import com.nxp.swtools.periphs.gui.view.componentsettings.ControlOptions;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl;
import com.nxp.swtools.periphs.model.config.ArrayConfig;
import com.nxp.swtools.periphs.model.config.IChild;
import com.nxp.swtools.periphs.model.config.IChildProvidable;
import com.nxp.swtools.periphs.model.config.ISettingConfig;
import com.nxp.swtools.periphs.model.config.ScalarConfig;
import com.nxp.swtools.periphs.model.config.ScalarConfig.Type;
import com.nxp.swtools.periphs.model.config.SetConfig.SetPresence;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.progress.ProgressUtils;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsColors;
import com.nxp.swtools.utils.resources.ToolsImages;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;

/**
 * Class represents common base for each of the tabular array GUI representations (vertical/horizontal).
 * @author Juraj Ondruska
 * @author Marek Ciz
 */
public abstract class AArrayControlTabular extends AArrayControlGroup {
	/** Logger of the class */
	protected static final @NonNull Logger LOGGER = LogManager.getLogger(AArrayControlTabular.class);
	/** Specifies the minimum width of table column */
	protected static final int INDICES_COLUMN_MIN_WIDTH = 50;
	/** Specifies the maximum width of table column */
	protected static final int COLUMN_MAX_WIDTH = 180;
	/** string for cell value */
	protected static final @NonNull String NOT_APPLICABLE = "N/A"; //$NON-NLS-1$
	/** Icon for array menu */
	static final @Nullable Image ICON_ARRAY_MENU = ToolsImages.getImage(IToolsImages.ICON_ARRAY_MENU);
	/** Foreground color of uneditable cells */
	static final @Nullable Color DISABLED_TEXT_COLOR = ToolsColors.SwToolsColors.DISABLED_FG.getColor();
	/** Background color of disabled cells */
	static final @Nullable Color DISABLED_COLOR = ToolsColors.SwToolsColors.DISABLED_BG.getColor();
	/** Error background decorator color */
	static final @Nullable Color ERROR_COLOR = ToolsColors.SwToolsColors.ERROR_BG.getColor();
	/** Warning background decorator color */
	static final @Nullable Color WARNING_COLOR = ToolsColors.SwToolsColors.WARNING_FG.getColor();
	/** Header cell font */
	protected @Nullable Font boldFont;
	/** "Not available" cell font */
	protected @Nullable Font italicFont;
	/** Table viewer showing settings table */
	protected @Nullable TableViewer tableViewer;
	/** Composite for containing whole tables */
	protected @Nullable Composite contentContainer = null;
	/** The scrolled composite in which the table is located */
	protected @Nullable ScrolledComposite scrolledComposite;

	/**
	 * Constructor.
	 * @param arrayConfig for which to create the GUI
	 * @param controllerWrapper containing the generic controller
	 */
	protected AArrayControlTabular(@NonNull ArrayConfig arrayConfig, @NonNull IControllerWrapper controllerWrapper) {
		super(arrayConfig, controllerWrapper);
	}

	/**
	 * Get all non-structured setting configs from the child (if the child is non-structured then the child itself is returned).
	 * This method works recursively.
	 * @param rootChild to obtain non-structured setting configs from
	 * @return all non-structured setting configs
	 */
	public static @NonNull List<@NonNull IChild> getSettingsFlat(@NonNull IChild rootChild) {
		List<@NonNull IChild> result = CollectionsUtils.emptyList();
		if (rootChild instanceof IChildProvidable) {
			IChildProvidable childProvidable = (IChildProvidable) rootChild;
			if (childProvidable instanceof ArrayConfig) {
				LOGGER.severe("Nested arrays are not allowed for displaying array in a table."); //$NON-NLS-1$
			} else {
				result = new ArrayList<>();
				for (IChild subChild : childProvidable.getChildren()) {
					result.addAll(getSettingsFlat(subChild));
				}
			}
		} else {
			// do not add Variable type ScalarConfigs
			if (!(rootChild instanceof ScalarConfig) || (((ScalarConfig) rootChild).getType() != Type.VARIABLE)) {
				result = CollectionsUtils.asList(rootChild);
			}
		}
		return result;
	}
	
	/**
	 * Get setting at the given position.
	 * @param itemNum numer of item
	 * @param idx number of setting in the item
	 * @return the setting or {@code null} if any setting was found at the given position
	 */
	public @Nullable IChild getSettingAt(int itemNum, int idx) {
		if ((itemNum >= 0) && (itemNum < children.size())) {
			List<@NonNull IChild> itemSettings = getSettingsFlat(children.get(itemNum).getChild());
			if ((idx >= 0) && (idx < itemSettings.size())) {
				return itemSettings.get(idx);
			}
		}
		 return null;
	}
	
	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#setAllSettingsTo(com.nxp.swtools.periphs.gui.view.componentsettings.internal.ArrayControlItemMenu, com.nxp.swtools.periphs.gui.view.componentsettings.internal.IArrayControlItemMenuControl)
	 */
	@Override
	public void setAllSettingsTo(@NonNull ArrayControlItemMenu caller, @NonNull IArrayControlItemMenuControl control) {
		IChild selectedSetting = control.getSelectedSettingHint();
		int idx = -1;
		List<@NonNull List<@NonNull IChild>> flattenedChildren = new ArrayList<>();
		for (IChildControl childControl : getChildren()) {
			List<@NonNull IChild> settingsFlat = getSettingsFlat(childControl.getChild());
			int idxOfSetting = settingsFlat.indexOf(selectedSetting);
			if (idxOfSetting >= 0) {
				idx = idxOfSetting;
			}
			flattenedChildren.add(settingsFlat);
		}
		List<@NonNull IChild> settings = new ArrayList<>();
		for (@NonNull List<@NonNull IChild> childrenOfItem : flattenedChildren) {
			IChild setting = childrenOfItem.get(idx);
			if (setting.isAvailable() && setting.isEnabled()) {
				settings.add(setting);
			}
		}
		if (selectedSetting != null) {
			ProgressUtils.run((m) -> {
				controllerWrapper.getController().setAllSettingsTo(UtilsText.safeToString(selectedSetting.getValue()), settings, this);
			});
		}
	}

	/**
	 * Creates column of Indices numbers
	 * @param tableViewer table viewer of the table
	 */
	protected static void createIndicesTableViewerColumn(@NonNull TableViewer tableViewer) {
		// first column in the table with left align
		final TableViewerColumn indicesViewerColumn = new TableViewerColumn(tableViewer, SWT.LEFT, 0);
		final TableColumn indicesColumn = indicesViewerColumn.getColumn();
		indicesColumn.setWidth(INDICES_COLUMN_MIN_WIDTH);
		indicesColumn.setAlignment(SWT.LEFT);
		indicesColumn.setText("#"); //$NON-NLS-1$
		indicesViewerColumn.setLabelProvider(new ColumnLabelProvider() {
			/*
			 * (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				String indices = UtilsText.EMPTY_STRING;
				if (element instanceof IChildControl) {
					indices = ((IChildControl) element).getChild().getUiName();
				}
				return indices;
			}
			
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getForeground(java.lang.Object)
			 */
			@Override
			public Color getForeground(Object element) {
				if (element instanceof IChildControl) {
					if (!((IChildControl) element).getChild().isEnabled()) {
						return DISABLED_TEXT_COLOR;
					}
				}
				return super.getForeground(element);
			}
			
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getBackground(java.lang.Object)
			 */
			@Override
			public Color getBackground(Object element) {
				if (element instanceof IChildControl) {
					return AArrayControlTabular.getBackgroundColor(((IChildControl) element).getChild());
				}
				return super.getBackground(element);
			}
			
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getImage(java.lang.Object)
			 */
			@Override
			public Image getImage(Object element) {
				if (element instanceof IChildControl) {
					return AArrayControlTabular.getImage(((IChildControl) element).getChild());
				}
				return super.getImage(element);
			}
			
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getToolTipText(java.lang.Object)
			 */
			@Override
			public String getToolTipText(Object element) {
				String tooltip = UtilsText.EMPTY_STRING;
				if (element instanceof IChildControl) {
					tooltip = ToolTipableFormatter.getToolTipText(new ToolTipableMarkdownDescriptionDecorator(((IChildControl) element).getChild()));
				}
				return tooltip;
			}
		});	
	}
	
	/**
	 * @param currentChild current child with settings
	 * @return {@code true}, if the table cell (element) is enabled 
	 */
	static boolean isCellEnabled(@NonNull IChild currentChild) {
		return currentChild.isAvailable() && currentChild.isEnabled();
	}

	/**
	 * Paints cells background according to errors, warnings info and availability of a child.
	 * @param currentChild current setting in the current cell
	 * @return background color
	 */
	protected static @Nullable Color getBackgroundColor(@NonNull IChild currentChild) {
		// yellow cell background if there is a warning
		if (currentChild.getWarning() != null) {
			return WARNING_COLOR;
		}
		// red cell background if there is an error
		if (currentChild.getError() != null) {
			return ERROR_COLOR;
		}		
		return null;
	}

	/**
	 * set Layout of table viewer's table
	 * @param table table from table viewer
	 * @param arrayConfigId Test ID for the table
	 * @param isVerticalTable flag, informs us if the table is in vertical table or not
	 */
	static void setTableLayout(@NonNull Table table, @NonNull String arrayConfigId, boolean isVerticalTable) {
		SWTFactoryProxy.INSTANCE.setTestId(table, TestIDs.PERIPHS_COMPONENT_SETTINGS_TABLE + UtilsText.UNDERSCORE + arrayConfigId);
		table.setLayoutData(new GridDataComponents(SWT.FILL, SWT.FILL, true, true));
		if (isVerticalTable) {
			table.setHeaderVisible(true);
		}
		table.setLinesVisible(true);
	}	
	
	/**
	 * Creates checked or unchecked check box decorated icon with error, warning, and info
	 * add no decorator, if there are no errors, warnings, or info
	 * @param currentChild current setting of the table cell
	 * @param isChecked flag is check box is checked
	 * @param isEnabled flag if check box is enabled
	 * @return decorated check box icon
	 */
	static @Nullable Image getDecoratedCheckBoxIcon(@NonNull IChild currentChild, boolean isChecked, boolean isEnabled) {
		int problemLevel = ErrorLevels.MIN_LEVEL - 1;
		String iconName = null;
		if (isChecked) {
			iconName = isEnabled ? IToolsImages.ICON_CHECKED_LEFT_TRANSPARENT_PIXEL : IToolsImages.ICON_CHECKED_DISABLED_LEFT_TRANSPARENT_PIXEL;
		} else {
			iconName = isEnabled ? IToolsImages.ICON_UNCHECKED_LEFT_TRANSPARENT_PIXEL : IToolsImages.ICON_UNCHECKED_DISABLED_LEFT_TRANSPARENT_PIXEL;
		}
		if (currentChild.getError() != null) {
			problemLevel = ErrorLevels.LEVEL_ERROR;
		} else if (currentChild.getWarning() != null) {
			problemLevel = ErrorLevels.LEVEL_WARNING;
		} else if (currentChild.getInfo() != null) {
			problemLevel = ErrorLevels.LEVEL_INFORMATION;
		}
		return APeriphsViewBase.getDecoratedImage(iconName, problemLevel);
	}
	
	/**
	 * Return image instead of check-box widget
	 * @param currentChild current child in table
	 * @return result icon which will be displayed instead of check box
	 */
	protected static @Nullable Image getImage(@NonNull IChild currentChild) {
		@Nullable Image shownIcon = null; 
		if (currentChild instanceof SetPresence) {
			SetPresence setPresence = (SetPresence) currentChild;
			shownIcon = AArrayControlTabular.getDecoratedCheckBoxIcon(currentChild, setPresence.getBoolValue(), setPresence.isEnabled());
		} else if ((currentChild instanceof ScalarConfig) && ((ScalarConfig) currentChild).getType().equals(Type.BOOL)) {
			boolean isChecked = currentChild.getValue().equals(Boolean.TRUE) ? Boolean.TRUE.booleanValue() : Boolean.FALSE.booleanValue();
			shownIcon = AArrayControlTabular.getDecoratedCheckBoxIcon(currentChild, isChecked, currentChild.isEnabled());
		} else {
			if (currentChild.getError() != null) {
				shownIcon = APeriphsViewBase.getDecoratedImage(IToolsImages.ICON_EMPTY_21P, ErrorLevels.LEVEL_ERROR);
			} else if (currentChild.getWarning() != null) {
				shownIcon = APeriphsViewBase.getDecoratedImage(IToolsImages.ICON_EMPTY_21P, ErrorLevels.LEVEL_WARNING);
			} else if (currentChild.getInfo() != null) {
				shownIcon = APeriphsViewBase.getDecoratedImage(IToolsImages.ICON_EMPTY_21P, ErrorLevels.LEVEL_INFORMATION);
			}
		}
		return shownIcon;
	}

	/**
	 * @param tableViewer tableViewer in which the new column will be created
	 * @param style of new font
	 * @return new font with requested style
	 */
	protected static Font createFont(@NonNull TableViewer tableViewer, int style) {
		final Table tableLoc = tableViewer.getTable();
		final Font fontLoc = tableLoc.getFont();
		assert fontLoc != null;
		final Font newFont = FontFactory.changeStyle(fontLoc, style);
		tableLoc.addDisposeListener(d -> newFont.dispose());
		return newFont;
	}
	
	/**
	 * Get value text to be displayed.
	 * @param scalarConfig the configuration containing the value
	 * @return textual representation of a value
	 */
	protected static @Nullable String getText(@NonNull ScalarConfig scalarConfig) {
		switch (scalarConfig.getType()) {
		case FLOAT:
		case INTEGER:
		case STRING:
			return scalarConfig.getStringValue();
		case INFO:
			return scalarConfig.getValue().toString();
		case ENUM:
			return scalarConfig.getEnumItemUiName();
		default:
			return UtilsText.EMPTY_STRING;
		}
	}

	/**
	 * Sets widths of table viewer columns as specified by {@link #getControlOptions} 
	 * @param column in the table of settings
	 */
	protected void setColumnWidth(@NonNull TableColumn column) {
		final Table table = Objects.requireNonNull(column.getParent());
		final int unitSize = computeColumnWidthUnitSize(table);
		final int index = table.indexOf(column);
		final List<@NonNull Integer> widths = getControlOptions().getTableColumnWidths();
		if ((widths != null) && (index < widths.size())) {
			final int columnWidthInUnits = widths.get(index).intValue();
			if (columnWidthInUnits >= 0) {
				column.setWidth(columnWidthInUnits * unitSize);
			}
		}
	}

	@Override
	public @NonNull Control createMainControl(@NonNull Composite composite) {
		// create composite with the content (1 column is enough for the table and + button under it)
		Composite contentComposite = createComposite(composite, 1);
		contentContainer = contentComposite;
		createLabelWithControls(contentComposite);
		Composite tableComposite = contentComposite;
		int tableStyle = SWT.SINGLE | SWT.FULL_SELECTION | SWT.H_SCROLL | SWT.BORDER;
		boolean scrolledCompositeRequired = OSDetect.isUnix(); // table scroll does not work properly with GTK3
		// Create scroll area
		if (scrolledCompositeRequired) {
			ScrolledComposite scrolledCompositeLoc = new ScrolledComposite(contentComposite, SWT.BORDER | SWT.H_SCROLL);
			scrolledComposite = scrolledCompositeLoc;
			scrolledCompositeLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.FILL, true, false));
			tableComposite = scrolledComposite;
			tableStyle = SWT.SINGLE | SWT.FULL_SELECTION;
		}
		// create a table viewer
		TableViewer tableViewerLoc = new TableViewer(tableComposite, tableStyle);
		tableViewerLoc.setContentProvider(ArrayContentProvider.getInstance());
		italicFont = createFont(tableViewerLoc, SWT.ITALIC);
		boldFont = createFont(tableViewerLoc, SWT.BOLD);
		tableViewer = tableViewerLoc;
		SWTFactoryProxy.INSTANCE.enableHtmlTooltipFor(tableViewerLoc);
		tableViewerHandling(tableViewerLoc);
		// create table context menu
		if (isUiArrayReorderSpecified() || !isUiArrayFixedSpecified()) {
			registerControl(getMenuContext(tableViewerLoc));
		}
		final Table table = tableViewerLoc.getTable();
		addKeyboardListener(table);
		// initialize the scroll properties
		ScrolledComposite scrolledCompositeLoc = scrolledComposite;
		if (scrolledCompositeLoc != null) {
			scrolledCompositeLoc.setContent(table);
			scrolledCompositeLoc.setExpandVertical(true);
			scrolledCompositeLoc.setExpandHorizontal(true);
			updateScrollSize();
			contentComposite.addListener(SWT.Resize, e -> {
				updateScrollSize();
			});
		}
		Composite contentContainerLoc = contentContainer;
		if (contentContainerLoc != null) {
			addScrollListener(table, contentContainerLoc);
		}
		return contentComposite;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#updateMainContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	public void updateMainContent(@NonNull Control mainControlLoc, @NonNull UpdateType updateType) {
		TableViewer viewerLoc = tableViewer;
		if (updateType == UpdateType.PROBLEM_DECORATION) {
			if (viewerLoc != null) {
				final Object input = viewerLoc.getInput();
				if (input instanceof List<?>) {
					viewerLoc.update(((List<?>) input).toArray(), null);
				} else {
					viewerLoc.refresh();
				}
			}
			return;
		}
		updateLabelWithControls(updateType);
		Iterator<IChildControl> childrenIterator = children.iterator();
		Iterator<ISettingConfig> arrayItemIterator = arrayConfig.getChildren().iterator();
		ISettingConfig arrayItem = arrayItemIterator.hasNext() ? arrayItemIterator.next() : null;
		// dispose controls of all removed items
		while (childrenIterator.hasNext()) {
			IChildControl childControl = childrenIterator.next();
			if (!childControl.getChild().equals(arrayItem)) {
				// Remove child
				childrenIterator.remove();
				childControl.dispose();
			} else { // match
				arrayItem = arrayItemIterator.hasNext() ? arrayItemIterator.next() : null;
			}
		}
		// update all the children
		for (IChildControl childControl : children) {
			childControl.update(updateType);
		}
		// add newly added items to the GUI
		while (arrayItem != null) {
			IChildControl currentChildControl = ChildControlFactory.create(arrayItem, new ControlOptions().labelHidden(getControlOptions().isArrayIndicesHidden()), controllerWrapper);
			if (currentChildControl != null) {
				children.add(currentChildControl);
				currentChildControl.update(updateType);
			}
			arrayItem = arrayItemIterator.hasNext() ? arrayItemIterator.next() : null;
		}
		if (viewerLoc != null) {
			updateTableViewer(viewerLoc);
		}
		IChildControl childToSelect = getChildToSelect();
		if (childToSelect != null) {
			setSelectedChild(childToSelect);
		}
	}

	/**
	 * Updates content of table viewer
	 * @param viewer to be updated
	 */
	protected abstract void updateTableViewer(@NonNull TableViewer viewer);

	/**
	 * Update size of the scroll area
	 */
	protected void updateScrollSize() {
		ScrolledComposite scrolledCompositeLoc = scrolledComposite;
		TableViewer tableViewerLoc = tableViewer;
		if ((scrolledCompositeLoc != null) && (tableViewerLoc != null)) {
			scrolledCompositeLoc.setMinSize(tableViewerLoc.getTable().computeSize(SWT.DEFAULT, SWT.DEFAULT));
		}
	}

	/**
	 * Returns menu context
	 * @param viewer for which the menu is created
	 * @return menu context
	 */
	protected abstract @NonNull ArrayControlItemMenuContext getMenuContext(@NonNull TableViewer viewer);

	/**
	 * Custom handling with table viewer
	 * @param viewer to be handled
	 */
	protected abstract void tableViewerHandling(@NonNull TableViewer viewer);

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#selectLastItem()
	 */
	@Override
	public void selectLastItem() {
		TableViewer tableViewerLoc = tableViewer;
		if ((tableViewerLoc != null) && !getChildren().isEmpty()) {
			IChildControl lastControl = getChildren().get(getChildren().size() - 1);
			tableViewerLoc.setSelection(new StructuredSelection(lastControl));
		}
	}
}
