/**
 * Copyright 2017-2019 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

import com.nxp.swtools.common.ui.utils.swt.FontFactory;
import com.nxp.swtools.common.ui.utils.swt.ScrolledCompositeHelper;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlFactory;
import com.nxp.swtools.periphs.gui.view.componentsettings.ControlOptions;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl;
import com.nxp.swtools.periphs.model.config.ArrayConfig;
import com.nxp.swtools.periphs.model.config.ISettingConfig;

/**
 * Common base for array settings independent in the GUI and not sharing controls.
 * @author Juraj Ondruska
 * @author Viktar Paklonski
 */
public abstract class AArrayControlStandard extends AArrayControlInternal {
	/** Map of all created menu controls (key: child control, value: menu control of child control) */
	private final @NonNull Map<IChildControl, IArrayControlItemMenuControl> menuControls = new HashMap<>();

	/**
	 * Constructor.
	 * @param arrayConfig for which to create the GUI
	 * @param controllerWrapper containing the generic controller
	 */
	protected AArrayControlStandard(@NonNull ArrayConfig arrayConfig, @NonNull IControllerWrapper controllerWrapper) {
		super(arrayConfig, controllerWrapper);
	}
	
	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#createLabelControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @Nullable Control createLabelControl(@NonNull Composite composite) {
		final Control label = super.createLabelControl(composite);
		if (label != null) {
			FontFactory.changeStyle(label, SWT.BOLD);
			createErrorDecoration(label, SWT.LEFT | SWT.TOP);
			registerCopyPasteMenu(label);
		}
		return label;
	}
	
	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#updateLabelContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	protected void updateLabelContent(@NonNull Control label, @NonNull UpdateType updateType) {
		super.updateLabelContent(label, updateType);
		updateErrorDecoration(label);
	}

	/**
	 * Create content of an array item. This method also creates menu button for the item.
	 * @param childControl to create content of
	 * @param contentComposite to create content in
	 * @param colSpan column span of the newly added item
	 */
	protected void createItemGui(@NonNull IChildControl childControl, @NonNull Composite contentComposite, int colSpan) {
		childControl.create(contentComposite, colSpan);
		if (!isUiArrayFixedSpecified()) {
			registerMenuControl(childControl, new ArrayControlItemMenuButton(this, contentComposite, childControl));
		}
	}

	/**
	 * Associates child setting controls with the specified menu control. 
	 * @param childControl to create content of
	 * @param menu control instance
	 */
	protected void registerMenuControl(@NonNull IChildControl childControl, @NonNull ArrayControlItemMenuButton menu) {
		menuControls.put(childControl, menu);
	}

	/**
	 * Destroys a previously associated setting menu control. 
	 * @param setting control instance
	 */
	protected void removeMenuControl(@NonNull IChildControl setting) {
		IArrayControlItemMenuControl control = menuControls.get(setting); 
		if (control != null) {
			control.dispose();
			menuControls.remove(setting);
		}
	}

	@Override
	protected void updateButtons(@Nullable UpdateType updateType) {
		menuControls.forEach((x, y) -> y.update(updateType));
		updateAddButton(updateType);
		updateRemoveButton(updateType);
		updateUpButton(updateType);
		updateDownButton(updateType);
	}

	@Override
	public void setAllSettingsTo(@NonNull ArrayControlItemMenu caller, @NonNull IArrayControlItemMenuControl control) {
		// not supported for standard array
	}

	@Override
	public void dispose() {
		menuControls.values().forEach(x -> x.dispose());
		menuControls.clear();
		super.dispose();
	}

	/**
	 * Create controls in composite for newly added setting if visible in the UI
	 * @param updateType Type of update
	 * @param childControl Newly added setting
	 * @param composite Composite in which the controls should be created
	 */
	protected void addItemToComposite(@NonNull UpdateType updateType, @NonNull IChildControl childControl, @NonNull Composite composite) {
		createItem(composite, childControl);
		childControl.update(updateType);
		ScrolledCompositeHelper.scrollToLastControlOfComposite(composite);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#createMainControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @NonNull Control createMainControl(@NonNull Composite contentComposite) {
		for (IChildControl childControl : children) {
			createItem(contentComposite, childControl);
		}
		if (!isUiArrayFixedSpecified()) {
			recreateAddButton(contentComposite);
		}
		return contentComposite;
	}

	/**
	 * Common implementation of {@link updateMainContent}
	 * @param updateType type of update
	 * @param compositeForItems composite to which new items will be added
	 * @return {@code true} if layout was changed, otherwise returns {@code false}
	 */
	public boolean updateMainCommonImpl(@NonNull UpdateType updateType, @NonNull Composite compositeForItems) {
		boolean layoutChange = false;
		if (updateType != UpdateType.PROBLEM_DECORATION) {
			Iterator<IChildControl> childrenIterator = children.iterator();
			Iterator<ISettingConfig> arrayItemIterator = arrayConfig.getChildren().iterator();
			ISettingConfig arrayItem = arrayItemIterator.hasNext() ? arrayItemIterator.next() : null;
			// dispose controls of all removed items
			while (childrenIterator.hasNext()) {
				IChildControl childControl = childrenIterator.next();
				if (!childControl.getChild().equals(arrayItem)) {
					// Remove child
					childrenIterator.remove();
					// Remove menu button
					removeMenuControl(childControl);
					disposeChild(childControl);
					layoutChange = true;
				} else { // match
					arrayItem = arrayItemIterator.hasNext() ? arrayItemIterator.next() : null;
				}
			}
			// update all the children
			for (IChildControl childControl : children) {
				childControl.update(updateType);
			}
			// add newly added items to GUI
			while (arrayItem != null) {
				layoutChange = true;
				IChildControl childControl = ChildControlFactory.create(arrayItem, new ControlOptions().labelHidden(getControlOptions().isArrayIndicesHidden()), controllerWrapper);
				if (childControl != null) {
					destroyAddButton();
					children.add(childControl);
					addItemToComposite(updateType, childControl, compositeForItems);
					if (!isUiArrayFixedSpecified()) {
						recreateAddButton(compositeForItems);
					}
				}
				arrayItem = arrayItemIterator.hasNext() ? arrayItemIterator.next() : null;
			}
			updateButtons(updateType);
		}
		return layoutChange;
	}

	/**
	 * Common handler for dispose of child control
	 * @param childControl to be disposed
	 */
	protected abstract void disposeChild(IChildControl childControl);

	/**
	 * Common handler for recreating add buttons
	 * @param composite in which the new button will be created
	 */
	protected abstract void recreateAddButton(@NonNull Composite composite);

	/**
	 * Common handler for creating GUI of item
	 * @param compositeForItem in which composite the GUI of new item will be created
	 * @param childControl for which childControl the GUI should be created
	 */
	protected abstract void createItem(@NonNull Composite compositeForItem, @NonNull IChildControl childControl);
}
