/**
 * Copyright 2017-2018 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;

import com.nxp.swtools.common.ui.utils.services.Rap;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.runtime.DelayedActionExecutorAdapter;
import com.nxp.swtools.common.utils.runtime.DelayedExecution;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.utils.resources.ToolsColors.SwToolsColors;

/**
 * Class providing help for working with text boxes of {@link Text} type.
 * @author Juraj Ondruska
 */
public class TextBoxHelper {
	/** Delay of value confirmation used when typing, in milliseconds */
	public static final int TYPING_DELAY = 500;

	/**
	 * Status of a setting.
	 * @author David Danaj
	 */
	public static enum Status {
		/** Value in text box is ok */
		OK,
		/** Value in text box is in error state. E.g. value out of range */
		VALUE_ERROR,
		/** Value in text box is invalid and can't be parsed. E.g. invalid character in number  */
		INVALID
	}

	/**
	 * Attach listener used for listening to modify events. The listener checks for erroneous states and updates the text's error
	 * indication.
	 * @param defaultValue supplier providing value to be used in case the user finished modification but the value was incorrect
	 * @param text the text box with the value and error indication
	 * @param checkFunction function used for validness checks
	 */
	public static void attachModifyErrorListener(@NonNull Supplier<@NonNull String> defaultValue, @NonNull Text text,
			@NonNull Function<@NonNull String, @NonNull Status> checkFunction) {
		attachModifyErrorListener(defaultValue, text, checkFunction, null);
	}

	/**
	 * Attach listener used for listening to modify events. The listener checks for erroneous states and updates the text's error
	 * indication. 
	 * @param defaultValue supplier providing value to be used in case the user finished modification but the value was incorrect
	 * @param text the text box with the value and error indication
	 * @param checkFunction function used for validness checks
	 * @param postUpdateCallBack function for additional actions during update
	 */
	public static void attachModifyErrorListener(@NonNull Supplier<@NonNull String> defaultValue, @NonNull Text text,
			@NonNull Function<@NonNull String, @NonNull Status> checkFunction, @Nullable Runnable postUpdateCallBack) {
		updateErrorIndication(text, checkFunction, postUpdateCallBack);
		text.addListener(SWT.Modify, e -> updateErrorIndication(text, checkFunction, postUpdateCallBack));
		text.addListener(SWT.Deactivate, e -> {
			Status status = isValueOk(text, checkFunction);
			if (status == Status.OK) {
				return;
			} else if (status == Status.INVALID) {
				text.setText(defaultValue.get());
			} else if (status == Status.VALUE_ERROR) {
				// can be extended here
			}
			updateErrorIndication(text, checkFunction, postUpdateCallBack);
		});
	}

	/**
	 * Check whether the current value of the text-box is correct given the check function.
	 * @param text the text box
	 * @param checkFunction the checking function
	 * @return whether the current value of the text-box is correct given the check function
	 */
	private static @NonNull Status isValueOk(@NonNull Text text, @NonNull Function<@NonNull String, @NonNull Status> checkFunction) {
		String textBoxValue = UtilsText.safeString(text.getText());
		Status checkResult = checkFunction.apply(textBoxValue);
		assert checkResult != null;
		return checkResult;
	}

	/**
	 * Update error indication of the text box given it's current state and the check function.
	 * @param text the text box with the value and error indication
	 * @param checkFunction function used for validness checks
	 * @param postUpdateCallBack function for additional actions during update
	 */
	private static void updateErrorIndication(@NonNull Text text, @NonNull Function<@NonNull String, @NonNull Status> checkFunction, @Nullable Runnable postUpdateCallBack) {
		if (isValueOk(text, checkFunction) == Status.OK) {
			text.setBackground(text.getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND));
			text.setForeground(text.getDisplay().getSystemColor(SWT.COLOR_LIST_FOREGROUND));
		} else {
			text.setBackground(SwToolsColors.ERROR_BG.getColor());
			text.setForeground(SwToolsColors.ERROR_FG.getColor());
		}
		if (postUpdateCallBack != null) {
			postUpdateCallBack.run();
		}
	}

	/**
	 * Attach listeners used for listening to more event types. These listeners should provide ability to detect text changes when
	 * an user stops typing.
	 * @param text the text box to listen for changes of
	 * @param consumer invoked when the text was modified
	 */
	public static void attachModifyListeners(@NonNull Text text, @NonNull final Consumer<@NonNull String> consumer) {
		DelayedExecution delayedExecution = new DelayedExecution("Apply value", new DelayedActionExecutorAdapter() { //$NON-NLS-1$
			@Override
			public void runAction() {
				if (!text.isDisposed()) {
					text.getDisplay().asyncExec(() ->
					{
						if (!text.isDisposed()) {
							consumer.accept(UtilsText.safeString(text.getText()));
						}
					});
				}
			}
		}, TYPING_DELAY);
		text.addFocusListener(new ModifyTextFocusListenerAdapter() {
			@Override
			protected void textModified() {
				consumer.accept(UtilsText.safeString(text.getText()));
			}
		});
		text.addListener(SWT.DefaultSelection, new Listener() {
			@Override
			public void handleEvent(Event event) {
				consumer.accept(UtilsText.safeString(text.getText()));
			}
		});
		text.addListener(SWT.Deactivate, new Listener() {
			@Override
			public void handleEvent(Event event) {
				if (text.isFocusControl()) {
					Objects.requireNonNull(text.getParent()).forceFocus();
				}
			}
		});
		if (!Rap.isActive()) {
			// this could cause huge traffic in the online version
			text.addKeyListener(new KeyListener() {
				@Override
				public void keyReleased(KeyEvent e) {
					// intentionally empty
				}
				@Override
				public void keyPressed(KeyEvent e) {
					delayedExecution.requestExecution();
				}
			});
		}
	}

	/**
	 * Removes all listeners of specific type from text
	 * @param text to remove from
	 * @param type of event
	 */
	public static void removeListenersOfType(@NonNull Text text, int type) {
		Listener[] listeners = text.getListeners(type);
		for (Listener listener: listeners) {
			if (listener != null) {
				text.removeListener(type, listener);
			}
		}
	}
}
