/**
 * Copyright 2017-2019 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.util.Objects;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Text;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.expression.Expression;
import com.nxp.swtools.common.utils.expression.ExpressionException;
import com.nxp.swtools.common.utils.expression.IFunction;
import com.nxp.swtools.common.utils.expression.IValue;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.TextBoxHelper.Status;
import com.nxp.swtools.periphs.model.config.ScalarConfig;
import com.nxp.swtools.periphs.model.config.ScalarUtils;
import com.nxp.swtools.utils.resources.ToolsColors.SwToolsColors;

/**
 * Class representing control of a text setting configuration.
 * @author Juraj Ondruska
 */
public class ScalarTextControl extends ScalarControl {
	/**
	 * @param child to create control for
	 * @param controllerWrapper containing the generic controller
	 */
	public ScalarTextControl(@NonNull ScalarConfig child, @NonNull IControllerWrapper controllerWrapper) {
		super(child, controllerWrapper);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.ScalarControl#createMainControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @NonNull Control createMainControl(@NonNull Composite composite) {
		Text text = new Text(composite, getSwtStyle());
		TextBoxHelper.attachModifyListeners(text, (value) -> {
			String childValue = getChild().getStringValue();
			if (!Objects.equals(value, childValue)) {
				Expression stringValidationExpr = getChild().getStringValidationExpr();
				if (stringValidationExpr != null) {
					IValue resolvedExpression;
					try {
						resolvedExpression = stringValidationExpr.resolve(getChild().getExpressionContext());
					} catch (Exception e) {
						LOGGER.severe("Expression resolving caused an error with message: " + e.getMessage()); //$NON-NLS-1$
						return;
					}
					IFunction functionReference;
					try {
						functionReference = resolvedExpression.getFunctionReference();
					} catch (ExpressionException e) {
						e.log();
						return;
					}
					Object result = functionReference.invokeOn(getChild().getExpressionContext(), value).getValue();
					if ((result instanceof Boolean) && !((Boolean) result).booleanValue()) {
						return;
					}
				}
				changeModelValue(value);
			}
		});
		TextBoxHelper.attachModifyErrorListener(() -> getChild().getStringValue(), text,
				value -> getStatus(value));
		createErrorDecoration(text, SWT.LEFT | SWT.TOP);
		return text;
	}

	/**
	 * Get status based on value.
	 * @param value String value that is being checked
	 * @return {@link Status#INVALID} if the text can't be parsed to value according to setting's type
	 * {@link Status#VALUE_ERROR} if value of the text is invalid for the setting (e.g. out of range)
	 * {@link Status#OK} otherwise
	 */
	private @NonNull Status getStatus(@NonNull String value) {
		if (child.isEnabled()) {
			if (!getChild().isParseable(value)) {
				return Status.INVALID;
			}
			if (!(ScalarUtils.isInRange(getChild(), value))) {
				return Status.VALUE_ERROR;
			}
			Expression stringValidationExpr = getChild().getStringValidationExpr();
			if (stringValidationExpr != null) {
				try {
					IFunction functionReference;
					try {
						functionReference = stringValidationExpr.resolve(getChild().getExpressionContext()).getFunctionReference();
					} catch (ExpressionException e) {
						String description = "Lambda function is required in string valdation expression"; //$NON-NLS-1$
						LOGGER.severe(description);
						throw e;
					}
					Object result = functionReference.invokeOn(getChild().getExpressionContext(), value).getValue();
					if ((result instanceof Boolean) && !((Boolean) result).booleanValue()) {
						return Status.INVALID;
					}
				} catch (@SuppressWarnings("unused") IllegalArgumentException e) {
					return Status.INVALID;
				}
			}
		}
		return Status.OK;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarControl#updateMainContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	protected void updateMainContent(@NonNull Control contentControl, UpdateType updateType) {
		if (updateType != UpdateType.PROBLEM_DECORATION) {
			Text textControl = (Text) contentControl;
			String stringValue = getChild().getStringValue();
			
			if (!Objects.equals(stringValue, textControl.getText())) {
				// update content only if it is changed
				int caretPos = textControl.getCaretPosition();
				textControl.setText(stringValue);
				textControl.setSelection(caretPos);
			}
			Color background = textControl.getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND);
			Color foreground = textControl.getDisplay().getSystemColor(SWT.COLOR_WIDGET_FOREGROUND);
			if (getChild().getWarning() != null) {
				background = textControl.getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND);
				foreground = SwToolsColors.WARNING_FG.getColor();
			}
			if (getChild().getError() != null) {
				background = SwToolsColors.ERROR_BG.getColor();
				foreground = textControl.getDisplay().getSystemColor(SWT.COLOR_WIDGET_FOREGROUND);
			}
			textControl.setBackground(background);
			textControl.setForeground(foreground);
		}
		updateErrorDecoration(contentControl);
	}

}
