/**
 * Copyright 2018-2019 NXP
 * Created: Jul 11, 2018
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.model.config.ScalarConfig;
import com.nxp.swtools.periphs.model.data.setting.InfoSetting;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;

/**
 * Class representing the control of a scalar {@link InfoSetting} configuration that can be copied into the clipboard.
 * @author Viktar Paklonski
 */
public class ScalarCopyControl extends ScalarControl {

	/** number of columns in the setting control */
	public static final int LAYOUT_COLS = 1;

	/**
	 * Containing composite for sub-elements.
	 */
	private class Container extends Composite {
		/** button to copy the {@link InfoSetting} value */
		private @NonNull Button copyButton;

		/**
		 * Creates the composite with sub-elements.
		 * @param parent a widget which will be the parent of the new instance
		 */
		public Container(@NonNull Composite parent) {
			super(parent, SWT.NONE);
			final GridLayoutComponents grid = new GridLayoutComponents(LAYOUT_COLS, false);
			grid.marginWidth = 0;
			grid.marginHeight = 0;
			this.setLayout(grid);
			copyButton = new Button(this, SWT.NONE);
			copyButton.setText(Messages.get().ScalarCopyControl_Button_Text);
			copyButton.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false));
			copyButton.addSelectionListener(new SelectionAdapter() {
				/* (non-Javadoc)
				 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
				 */
				@Override
				public void widgetSelected(SelectionEvent e) {
					copyToClipboard(UtilsText.safeString(child.getValue().toString()));
				}
			});
		}

		/**
		 * @return {@link #copyButton}
		 */
		public @NonNull Button getCopyButton() {
			return copyButton;
		}

		/**
		 * Place data of the text type on the system clipboard.
		 * @param text data
		 */
		protected void copyToClipboard(@NonNull String text) {
			final StringSelection selection = new StringSelection(text);
			final Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
			clipboard.setContents(selection, selection);
		}
	}

	/**
	 * @param child to create control for
	 * @param controllerWrapper containing the generic controller
	 */
	public ScalarCopyControl(@NonNull ScalarConfig child, @NonNull IControllerWrapper controllerWrapper) {
		super(child, controllerWrapper);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarControl#createMainControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @NonNull Control createMainControl(@NonNull Composite composite) {
		final Container container = new Container(composite);
		createErrorDecoration(container, SWT.LEFT | SWT.TOP);
		return container;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarControl#updateMainContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	protected void updateMainContent(@NonNull Control contentControl, @NonNull UpdateType updateType) {
		if (updateType != UpdateType.PROBLEM_DECORATION) {
			final Container container = (Container) contentControl;
			final boolean enabled = (updateType != UpdateType.FORCE_DISABLE) && child.isEnabled();
			container.getCopyButton().setEnabled(enabled);
			final String toolTipText = ToolTipableFormatter.getToolTipText(new ToolTipableMarkdownDescriptionDecorator(child));
			SWTFactoryProxy.INSTANCE.setHtmlTooltip(container.getCopyButton(), toolTipText);
		}
		updateErrorDecoration(contentControl);
	}

}
