/**
 * Copyright 2017-2019 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import java.util.logging.Logger;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewSite;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.ui.utils.swt.widgets.InstantSearchList;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.configuration.properties.SWToolsProperties;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.controller.UiValueIdHolder;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.AddComponentDialog;
import com.nxp.swtools.periphs.gui.view.componentsettings.TextBoxHelper.Status;
import com.nxp.swtools.periphs.model.config.IChildProvidable;
import com.nxp.swtools.periphs.model.config.ScalarConfig;
import com.nxp.swtools.periphs.model.data.ConfigurationComponent;
import com.nxp.swtools.periphs.model.data.ConfigurationComponentTypeId;
import com.nxp.swtools.periphs.model.data.setting.DynamicEnumSetting.CustomValueSupport;
import com.nxp.swtools.periphs.model.data.setting.SetSetting.CustomItem;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.progress.ProgressUtils;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsColors.SwToolsColors;
import com.nxp.swtools.utils.resources.ToolsImages;

/**
 * Class representing control of a combo-box setting configuration.
 * @author Juraj Ondruska
 */
public class ScalarComboControl extends ScalarControl {
	/** Logger of the class */
	private static final @NonNull Logger LOGGER = LogManager.getLogger(ScalarComboControl.class);
	/** Number of columns used when FEATURE_COMPONENT_ADD is defined for a setting */
	final static int FEATURE_COMPONENT_ADD_NUM_COLUMNS = 2;
	/** Main combo-box */
	@Nullable InstantSearchList combo;

	/** Check function for custom value */
	private @NonNull Function<@NonNull String, @NonNull Status> checkFunction = (newVal) -> {
		if (getChild().getValueName().startsWith(CustomItem.PREFIX)) { // Only check if current value is custom
			final InstantSearchList comboLoc = combo;
			boolean isPredefined = (comboLoc != null) && !comboLoc.isDisposed() && comboLoc.getItems().contains(newVal); // allow to enter predefined value (MCUXCON-3666)
			if (!getChild().isCustomValueValid(newVal) && !isPredefined) {
				return Status.INVALID;
			}
		}
		return Status.OK;
	};

	/**
	 * @param child to create control for
	 * @param controllerWrapper containing the generic controller
	 */
	public ScalarComboControl(@NonNull ScalarConfig child, @NonNull IControllerWrapper controllerWrapper) {
		super(child, controllerWrapper);
	}
	
	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.ScalarControl#createMainControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @Nullable Control createMainControl(@NonNull Composite composite) {
		Composite contentComposite = composite;
		
		if (getChild().isFeatureComponentAddDefined()) {
			// create composite only if feature is defined
			contentComposite = new Composite(composite, SWT.NONE);
			GridLayoutComponents layout = new GridLayoutComponents(FEATURE_COMPONENT_ADD_NUM_COLUMNS, false);
			layout.marginWidth = layout.marginHeight = 0;
			contentComposite.setLayout(layout);
		}
		
		InstantSearchList comboLoc = new InstantSearchList(contentComposite, getSwtStyle());
		combo = comboLoc;
		
		if (getChild().isFeatureComponentAddDefined()) {
			// if feature is defined, set layout data for combo-box -> will not be set by parent as whole composite is now main control, not only combo itself
			comboLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false));
			SWTFactoryProxy.INSTANCE.setTestId(comboLoc, TestIDs.PERIPHS_SETTING_CONTROL + child.getId());
		}
		
		comboLoc.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
				widgetSelected(e);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			@Override
			public void widgetSelected(SelectionEvent e) {
				String text = comboLoc.getText();
				@SuppressWarnings("unchecked")
				List<@NonNull UiValueIdHolder> items = (List<@NonNull UiValueIdHolder>) Objects.requireNonNull(comboLoc.getData(KEY_INPUT));
				CustomItem customItem = getChild().getCustomItem();
				for (UiValueIdHolder item : items) {
					if ((item.getId().equals(CustomItem.CUSTOM_ID)) && (text.equals(CustomItem.CUSTOM_LABEL))) { // Newly selected custom option
						if (!comboLoc.isAllowCustomValue()) { // Do not remove already given value
							comboLoc.setAllowCustomValue(true);
							CustomValueSupport customValueSupport = getChild().getCustomValueSupport();
							if (customValueSupport != null) {
								text = customValueSupport.getDefaultValue();
							}
							comboLoc.setText(text);
						}
						changeModelValue(CustomItem.getStorableFormat(text));
						comboLoc.setSelection(new Point(0, text.length()));
						updateErrorDecoration(comboLoc);
						// SELECTION OF CUSTOM VALUE (Custom...) IN COMBO ENDS HERE!!!
						return;
					}
					if ((item.getUiValue().equals(text)) && (!item.getId().equals(CustomItem.CUSTOM_ID))) { // When any other item than custom item
						comboLoc.setAllowCustomValue(false);
						if (customItem != null) {
							customItem.setValue(UtilsText.safeString(CustomItem.CUSTOM_LABEL)); // Roll-back to default value for custom item
						}
						if (!Objects.equals(getChild().getValue(), (item.getId()))) {
							changeModelValue(item.getId());
						}
						break;
					}
				}
				if (comboLoc.isAllowCustomValue()) {
					final String storableFormat = CustomItem.getStorableFormat(UtilsText.safeString(text));
					if (!getChild().getStringValue().equals(storableFormat)) {
						changeModelValue(storableFormat); // Change to flush cache
					}
				}
				updateErrorDecoration(comboLoc);
			}
		});
		comboLoc.setData(KEY_INPUT, CollectionsUtils.emptyList());
		createErrorDecoration(comboLoc, SWT.LEFT | SWT.TOP);
		
		if (getChild().isFeatureComponentAddDefined()) {
			// create button if feature is defined
			Button button = new Button(contentComposite, SWT.PUSH);
			SWTFactoryProxy.INSTANCE.setTestId(button, TestIDs.PERIPHS_ADD_COMPONENT_INSTANCE_IN_COMPONENT_SETTING_VIEW_BUTTON + child.getId());
			SWTFactoryProxy.INSTANCE.setHtmlTooltip(button, Messages.get().ComponentsView_AddButtonTooltip);
			button.setImage(ToolsImages.getImage(IToolsImages.ICON_ADD_CIRCLE));
			button.setLayoutData(new GridDataComponents(SWT.RIGHT, SWT.FILL, false, false));
			button.addSelectionListener(new SelectionAdapter() {
				/* (non-Javadoc)
				 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
				 */
				@Override
				public void widgetSelected(SelectionEvent e) {
					IViewPart viewPart = ComponentSettingViewHelper.getViewPart();
					if (viewPart != null) {
						IViewSite viewSite = viewPart.getViewSite();
						if (viewSite != null) {
							controllerWrapper.getController().runTransaction(new Runnable() {
								@Override
								public void run() {
									//FIXME Rares replace this code with guiController call  
									List<@NonNull String> optionValues = getChild().getFeatureComponentAddValues();
									if (optionValues != null) {
										List<@NonNull ConfigurationComponentTypeId> componentsTypeIds = CollectionsUtils.getInstancesOf(optionValues.stream()
											.map(type -> controllerWrapper.getController().getConfigurationComponentTypeId(type)), ConfigurationComponentTypeId.class)
											.collect(CollectorsUtils.toList());
										AddComponentDialog openForComponents = AddComponentDialog.openForComponents(viewSite, componentsTypeIds, controllerWrapper);
										IChildProvidable result = openForComponents.getResult();
										if (result != null) {
											// newly created component/component instance should be selected in a combo
											ProgressUtils.run((m) -> {
												controllerWrapper.getController().setValue(getChild(), result.getUiName(), this);
											});
										}
									}
								}
							});
							
						}
					}
				}
			});
			if (SWToolsProperties.isVerificationOn()) {
				List<@NonNull String> componentsIds = getChild().getFeatureComponentAddValues();
				if (componentsIds != null) {
					CollectionsUtils.getInstancesOf(componentsIds.stream()
						.map(id -> controllerWrapper.getController().getConfigurationComponent(id)), ConfigurationComponent.class)
						.filter(c -> c.isGlobalOnly())
						.forEach(c -> LOGGER.warning("Component type \"" + c.getId() + "\" referenced in enum with id \"" + child.getId() + "\" is not instantiable.")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				}
			}
		}
		// if feature is defined, return whole composite, otherwise return only combo -> parent later sets layoutData for returned value to have proper look
		return (composite == contentComposite) ? comboLoc : contentComposite;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarControl#updateMainContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	protected void updateMainContent(@NonNull Control contentControl, UpdateType updateType) {
		InstantSearchList comboLoc = combo;
		if (comboLoc != null) {
			if (updateType != UpdateType.PROBLEM_DECORATION) {
				Text textBox = comboLoc.getTextBox();
				if ((textBox != null) && (comboLoc.isAllowCustomValue())) {
					TextBoxHelper.removeListenersOfType(textBox, SWT.Modify);
					TextBoxHelper.removeListenersOfType(textBox, SWT.Deactivate);
				}
				List<@NonNull UiValueIdHolder> items = controllerWrapper.getController().getEnumItems(getChild());
				String[] arrayOfItems = items.stream().map(x -> x.getUiValue()).toArray(String[]::new);
				if (!Arrays.equals(comboLoc.getItems().toArray(), arrayOfItems))  {
					comboLoc.setData(KEY_INPUT, items);
					comboLoc.setItems(arrayOfItems);
					for (UiValueIdHolder uiValueIdHolder : items) {
						comboLoc.addItemToolTip(uiValueIdHolder.getUiValue(), uiValueIdHolder.getDescription());
					}
				}
				String valueName = getChild().getValueName();
				if (valueName.startsWith(CustomItem.PREFIX)) {
					valueName = CustomItem.CUSTOM_ID;
					comboLoc.setAllowCustomValue(true); // If given value is custom value, then allow to edit it
				}
				String value = controllerWrapper.getController().getEnumItemById(getChild(), valueName).getUiValue();
				if (!Objects.equals(comboLoc.getText(), value)) {
					comboLoc.select(value);
				}
				if ((textBox != null) && (comboLoc.isAllowCustomValue())) {
					TextBoxHelper.attachModifyErrorListener(() -> CustomItem.parseValueFromStorableString(
							getChild().getStringValue()), textBox, checkFunction, () -> updateErrorDecoration(comboLoc));
				}
			}
			updateErrorDecoration(comboLoc);
		}
	}

	/**
	 * Update background of the control when error or warning occurred.
	 * @param controlWithDecoration instant search list to be decorated
	 */
	@Override
	protected void updateErrorDecoration(@NonNull Control controlWithDecoration) {
		super.updateErrorDecoration(controlWithDecoration);
		if (controlWithDecoration instanceof InstantSearchList) {
			InstantSearchList searchList = (InstantSearchList) controlWithDecoration;
			String text = UtilsText.safeString(searchList.getText());
			if ((child.getError() != null) || (checkFunction.apply(text) == Status.INVALID)) {
				searchList.setBackground(SwToolsColors.ERROR_BG.getColor());
				searchList.setForeground(searchList.getDisplay().getSystemColor(SWT.COLOR_WIDGET_FOREGROUND));
			} else if (child.getWarning() != null) {
				searchList.setBackground(searchList.getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND));
				searchList.setForeground(SwToolsColors.WARNING_FG.getColor());
			} else {
				searchList.setBackground(searchList.getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND));
				searchList.setForeground(searchList.getDisplay().getSystemColor(SWT.COLOR_WIDGET_FOREGROUND));
			}
		}
	}

}
