/**
 * Copyright 2018-2019 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.text.MessageFormat;
import java.util.Objects;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.ui.utils.swt.widgets.InstantSearchList;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType;
import com.nxp.swtools.periphs.model.config.IPresettable;
import com.nxp.swtools.periphs.model.config.QuickSelectionHelper;
import com.nxp.swtools.periphs.model.data.QuickSelection;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.progress.ProgressUtils;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;
import com.nxp.swtools.utils.tooltip.ToolTipableImplementation;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;

/**
 * Class used for displaying quick selections.
 * @author Juraj Ondruska
 */
public class QuickSelectionControl {
	/** The quick selection configuration */
	protected @NonNull IPresettable presettable;
	/** The composite */
	private @Nullable Composite quickSelectionComposite;
	/** The label */
	private @Nullable Label label;
	/** The quick selection combo box */
	private @Nullable InstantSearchList instantSearchList;
	/** The wrapper containing the generic controller */
	protected @NonNull IControllerWrapper controllerWrapper;

	/**
	 * Constructor.
	 * @param presetable the quick selection configuration
	 * @param controllerWrapper containing the generic controller
	 */
	public QuickSelectionControl(@NonNull IPresettable presetable, @NonNull IControllerWrapper controllerWrapper) {
		this.presettable = presetable;
		this.controllerWrapper = controllerWrapper;
	}

	/**
	 * Create the quick selection combo box.
	 * @param composite to fill
	 * @param colSpan col span to use in the grid data of the combo box
	 */
	public void create(@NonNull Composite composite, int colSpan) {
		if (!isEmpty()) {
			final Composite qsComposite = new Composite(composite, SWT.NONE);
			qsComposite.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, colSpan, 1));
			final GridLayoutComponents layout = new GridLayoutComponents(2, false);
			layout.marginWidth = layout.marginHeight = 0;
			qsComposite.setLayout(layout);
			quickSelectionComposite = qsComposite;
			final Label labelLoc = new Label(qsComposite, SWT.NONE);
			labelLoc.setLayoutData(new GridDataComponents(SWT.RIGHT, SWT.CENTER, true, false));
			labelLoc.setText(Messages.get().ConfigSetControl_ConfigSetComboTitle);
			label = labelLoc;
			final InstantSearchList instantSearchListLoc = new InstantSearchList(qsComposite, SWT.BORDER);
			instantSearchListLoc.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false));
			instantSearchListLoc.addSelectionListener(new SelectionAdapter() {
				/* (non-Javadoc)
				 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
				 */
				@Override
				public void widgetSelected(SelectionEvent e) {
					String selectedQuickSelection = UtilsText.safeString(instantSearchListLoc.getText());
					QuickSelection currentQuickSelection = presettable.getQuickSelection();
					if ((currentQuickSelection == null) || !selectedQuickSelection.equals(currentQuickSelection.getUIName(presettable.getExpressionContext()))) {
						ProgressUtils.run((m) -> {
							controllerWrapper.getController().applyQuickSelection(selectedQuickSelection, presettable, instantSearchListLoc);
						});
					}
				}
			});
			SWTFactoryProxy.INSTANCE.setTestId(instantSearchListLoc, TestIDs.PERIPHS_QUICK_SELECTION_COMBO + presettable.getId());
			instantSearchList = instantSearchListLoc;
		}
	}

	/**
	 * Dispose GUI.
	 */
	public void dispose() {
		if (instantSearchList != null) {
			instantSearchList.dispose();
			instantSearchList = null;
		}
		if (label != null) {
			label.dispose();
			label = null;
		}
		if (quickSelectionComposite != null) {
			quickSelectionComposite.dispose();
			quickSelectionComposite = null;
		}
	}

	/**
	 * @return {@code true} if the quick selection configuration is empty, {@code false} otherwise
	 */
	public boolean isEmpty() {
		return QuickSelectionHelper.getAvailableQuickSelections(presettable).isEmpty();
	}

	/**
	 * Update state of the quick selection combo.
	 * @param updateType type of the update to perform
	 */
	public void update(@NonNull UpdateType updateType) {
		if (updateType == UpdateType.PROBLEM_DECORATION) {
			return;
		}
		// update visibility of quick selection
		Composite quickSelectionCompositeLoc = quickSelectionComposite;
		if (quickSelectionCompositeLoc != null) {
			final boolean visible = presettable.isAvailable();
			boolean layoutChange = false;
			layoutChange = quickSelectionCompositeLoc.isVisible() != visible;
			quickSelectionCompositeLoc.setVisible(visible);
			((GridDataComponents) Objects.requireNonNull(quickSelectionCompositeLoc.getLayoutData())).exclude = !visible;
			if (layoutChange) {
				Composite parent = quickSelectionCompositeLoc.getParent();
				if (parent != null) {
					parent.requestLayout();
				}
			}
		}
		// update content
		InstantSearchList combo = instantSearchList;
		Label labelLoc = label;
		if ((labelLoc != null) || (combo != null)) {
			ToolTipableImplementation toolTipableImplementation = new ToolTipableImplementation();
			toolTipableImplementation.setUiName(Messages.get().ConfigSetControl_ConfigSetComboTitle);
			toolTipableImplementation.setDescription(MessageFormat.format(UtilsText.safeString(Messages.get().ConfigSetControl_ConfigSetComboDescription), presettable.getUiName()));
			QuickSelection quickSelection = presettable.getQuickSelection();
			if (quickSelection != null) {
				toolTipableImplementation.setValueName(quickSelection.getUIName(presettable.getExpressionContext()));
				toolTipableImplementation.setValueDescription(quickSelection.getDescription());
			} else {
				toolTipableImplementation.setDescription(Messages.get().ConfigSetControl_QuickSelection_Custom);
			}
			String toolTipText = ToolTipableFormatter.getToolTipText(new ToolTipableMarkdownDescriptionDecorator(toolTipableImplementation));
			if (labelLoc != null) {
				SWTFactoryProxy.INSTANCE.setHtmlTooltip(labelLoc, toolTipText);
				labelLoc.setEnabled((updateType != UpdateType.FORCE_DISABLE) && presettable.isEnabled());
			}
			if (combo != null) {
				@NonNull String[] quickSelections = QuickSelectionHelper.getAvailableQuickSelections(presettable).stream()
						.map(x -> x.getUIName(presettable.getExpressionContext()))
						.toArray(l -> new @NonNull String[l]);
				SWTFactoryProxy.INSTANCE.setTestId(combo, TestIDs.PERIPHS_QUICK_SELECTION_COMBO + presettable.getId());
				combo.setItems(quickSelections);
				if (quickSelection != null) {
					combo.select(quickSelection.getUIName(presettable.getExpressionContext()));
				} else {
					combo.select(UtilsText.safeString(Messages.get().ConfigSetControl_QuickSelection_Custom));
				}
				SWTFactoryProxy.INSTANCE.setHtmlTooltip(combo, toolTipText);
				combo.setEnabled((updateType != UpdateType.FORCE_DISABLE) && presettable.isEnabled());
			}
		}
	}
}
