/**
 * Copyright 2018 NXP
 * Created: Jun 13, 2018
 */

package com.nxp.swtools.periphs.gui.view.componentsettings;

import static com.nxp.swtools.periphs.model.data.SettingOptions.*;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.model.config.IChild;
import com.nxp.swtools.periphs.model.data.Mode;
import com.nxp.swtools.periphs.model.data.setting.InfoSetting;

/**
 * UI options for specifying how controls should be displayed
 * @author David Danaj (b57899/nxa30572)
 */
public final class ControlOptions {
	/** Logger of the class */
	private static final @NonNull Logger LOGGER = LogManager.getLogger(ControlOptions.class);
	
	/**
	 * Enumeration of possible types how setting can be displayed
	 * @author David Danaj (b57899/nxa30572)
	 */
	public enum ShowContentAs {
		/** Show as FORM */
		@NonNull FORM(VALUE_SHOW_CONTENT_AS_FORM),
		/** Show as TABLE */
		@NonNull TABLE(VALUE_SHOW_CONTENT_AS_TABLE),
		/** Show as TABS */
		@NonNull TABS(VALUE_SHOW_CONTENT_AS_TABS),
		/** Show as MASTER_DETAIL */
		@NonNull MASTER_DETAIL(VALUE_SHOW_CONTENT_AS_MASTER_DETAIL);
		
		/** String representation of ShowContentAs */
		@NonNull private String xmlName;
		
		/**
		 * Constructor.
		 * @param xmlName String representation of ShowContentAs
		 */
		private ShowContentAs(@NonNull String xmlName) {
			this.xmlName = xmlName; 
		}

		/* (non-Javadoc)
		 * @see java.lang.Enum#toString()
		 */
		@Override
		public String toString() {
			return xmlName;
		}
		
		/**
		 * Get ShowContentAs value based on SettingOption string value.
		 * @param settingOption string to parse
		 * @return parsed ShowContentAs or {@code null} if unit was not recognized
		 */
		public static @Nullable ShowContentAs parse(@Nullable String settingOption) {
			if (settingOption != null) {
				for (ShowContentAs showContentAs : values()) {
					if (showContentAs.toString().equals(settingOption)) {
						return showContentAs;
					}
				}
			}
			return null;
		}
	}
	
	/**
	 * Enumeration of possible types how setting can be printed
	 * @author David Danaj (b57899/nxa30572)
	 */
	public enum PrintAs {
		/** Print as PLAIN */
		PLAIN(VALUE_PRINT_THIS_AS_PLAIN),
		/** Print as LINK */
		LINK(VALUE_PRINT_THIS_AS_LINK);
		
		/** String representation of PrintAs */
		@NonNull String xmlName;
		
		/**
		 * Constructor.
		 * @param xmlName String representation of PrintAs
		 */
		private PrintAs(@NonNull String xmlName) {
			this.xmlName = xmlName; 
		}
		
		/* (non-Javadoc)
		 * @see java.lang.Enum#toString()
		 */
		@Override
		public String toString() {
			return xmlName;
		}
		
		/**
		 * Get PrintAs value based on SettingOption string value.
		 * @param settingOption string to parse
		 * @return parsed PrintAs or {@code null} if unit was not recognized
		 */
		public static @Nullable PrintAs parse(@Nullable String settingOption) {
			if (settingOption != null) {
				for (PrintAs printAs : values()) {
					if (printAs.toString().equals(settingOption)) {
						return printAs;
					}
				}
			}
			return null;
		}
	}
	
	// null in fields is used as "not defined" - to avoid overwriting existing settings with default state when merging options
	/** ShowContentAs option */
	private @Nullable ShowContentAs showContentAs;
	/** PrintAs option */
	private @Nullable PrintAs printAs;
	/** should array be fixed */
	private @Nullable Boolean arrayFixed;
	/** should array allow reorder of items */
	private @Nullable Boolean arrayReorder;
	/** should array indices be hidden */
	private @Nullable Boolean arrayIndicesHidden;
	/** should array be displayed horizontally  */
	private @Nullable Boolean arrayHorizontal;
	/** should label be hidden */
	private @Nullable Boolean labelHidden;
	/** should border be hidden */
	private @Nullable Boolean borderHidden;
	/** should provide a button to copy {@link InfoSetting} value */
	private @Nullable Boolean infoCopyButton;
	/** should {@link Mode} be hidden */
	private @Nullable Boolean componentModeHidden;
	/** widths of the table columns */
	private @Nullable List<@NonNull Integer> tableColumnWidths;

	/**
	 * Empty constructor.
	 */
	public ControlOptions() {
		// empty on purpose
	}
	
	/**
	 * Create new instance by retrieving options from model. If options are not defined, set fields to default state.
	 * @param child setting for which control options are created
	 */
	public ControlOptions(@NonNull final IChild child) {
		Object optionValue = child.getOptionValue(SHOW_CONTENT_AS);
		if ((optionValue != null) && (optionValue instanceof String)) {
			showContentAs = ShowContentAs.parse((String) optionValue);
		} else {
			showContentAs = ShowContentAs.FORM; // set default
		}
		optionValue = child.getOptionValue(PRINT_THIS_AS);
		if ((optionValue != null) && (optionValue instanceof String)) {
			printAs = PrintAs.parse((String) optionValue);
		} else {
			printAs = PrintAs.PLAIN; // set default
		}
		// if option is not set - it is set to default state (false)
		arrayFixed = Boolean.valueOf(child.isOptionSet(UI_ARRAY_FIXED));
		arrayReorder = Boolean.valueOf(child.isOptionSet(UI_ARRAY_REORDER));
		arrayIndicesHidden = Boolean.valueOf(child.isOptionSet(UI_ARRAY_INDICES_HIDDEN));
		arrayHorizontal = Boolean.valueOf(child.isOptionSet(UI_ARRAY_LAYOUT_HORIZONTAL));
		labelHidden = Boolean.valueOf(child.isOptionSet(UI_LABEL_HIDDEN));
		borderHidden = Boolean.valueOf(child.isOptionSet(UI_BORDER_HIDDEN));
		infoCopyButton = Boolean.valueOf(child.isOptionSet(UI_INFO_COPY_BUTTON));
		componentModeHidden = Boolean.valueOf(child.isOptionSet(UI_COMPONENT_MODE_HIDDEN));
		optionValue = child.getOptionValue(UI_TABLE_COLUMN_WIDTHS);
		if ((optionValue != null) && (optionValue instanceof String)) {
			tableColumnWidths = parseIntArray((String) optionValue);
		}
	}
	
	/**
	 * @return the arrayFixed
	 */
	public boolean isArrayFixed() {
		return Boolean.TRUE.equals(arrayFixed);
	}

	/**
	 * @return the arrayReorder
	 */
	public boolean isArrayReorder() {
		return Boolean.TRUE.equals(arrayReorder);
	}

	/**
	 * @return the arrayIndicesHidden
	 */
	public boolean isArrayIndicesHidden() {
		return Boolean.TRUE.equals(arrayIndicesHidden);
	}

	/**
	 * @return the arrayHorizontal
	 */
	public boolean isArrayHorizontal() {
		return Boolean.TRUE.equals(arrayHorizontal);
	}

	/**
	 * @return the labelHidden
	 */
	public boolean isLabelHidden() {
		return Boolean.TRUE.equals(labelHidden);
	}

	/**
	 * @return the borderHidden
	 */
	public boolean isBorderHidden() {
		return Boolean.TRUE.equals(borderHidden);
	}

	/**
	 * @return whether {@link #infoCopyButton} equals {@code true}
	 */
	public boolean isInfoCopyButtonRequested() {
		return Boolean.TRUE.equals(infoCopyButton);
	}

	/**
	 * @return whether {@link #componentModeHidden} equals {@code true}
	 */
	public boolean isComponentModeHidden() {
		return Boolean.TRUE.equals(componentModeHidden);
	}

    /**
	 * @return the arrayFixed
	 */
	public @Nullable Boolean getArrayFixed() {
		return arrayFixed;
	}

	/**
	 * @return the arrayReorder
	 */
	public @Nullable Boolean getArrayReorder() {
		return arrayReorder;
	}

	/**
	 * @return the arrayIndicesHidden
	 */
	public @Nullable Boolean getArrayIndicesHidden() {
		return arrayIndicesHidden;
	}

	/**
	 * @return the arrayHorizontal
	 */
	public @Nullable Boolean getArrayHorizontal() {
		return arrayHorizontal;
	}

	/**
	 * @return the labelHidden
	 */
	public @Nullable Boolean getLabelHidden() {
		return labelHidden;
	}

	/**
	 * @return the borderHidden
	 */
	public @Nullable Boolean getBorderHidden() {
		return borderHidden;
	}

	/**
	 * @return {@link #componentModeHidden}
	 */
	public @Nullable Boolean getComponentModeHidden() {
		return componentModeHidden;
	}
	
	/**
	 * @return the showContentAs
	 */
	public @Nullable ShowContentAs getShowContentAs() {
		return showContentAs;
	}

	/**
	 * @return the printAs
	 */
	public @Nullable PrintAs getPrintAs() {
		return printAs;
	}

	/**
	 * @return {@link #infoCopyButton}
	 */
	public @Nullable Boolean getInfoCopyButton() {
		return infoCopyButton;
	}

	/**
	 * @return {@link #tableColumnWidths}
	 */
	public @Nullable List<@NonNull Integer> getTableColumnWidths() {
		return tableColumnWidths;
	}
	
	/**
	 * @param setShowContentAs set how content should be displayed
	 * @return this
	 */
	public @NonNull ControlOptions showContentAs(@NonNull final ShowContentAs setShowContentAs) {
		this.showContentAs = setShowContentAs;
		return this;
	}
	
	/**
	 * @param setPrintAs set how content should be printed
	 * @return this
	 */
	public @NonNull ControlOptions printAs(@NonNull final PrintAs setPrintAs) {
		this.printAs = setPrintAs;
		return this;
	}
	
	/**
	 * @param isArrayFixed set arrayFixed
	 * @return this
	 */
	public @NonNull ControlOptions arrayFixed(final boolean isArrayFixed) {
		this.arrayFixed = Boolean.valueOf(isArrayFixed);
		return this;
	}

	/**
	 * @param isArrayReorder set arrayReorder
	 * @return this
	 */
	public @NonNull ControlOptions arrayReorder(final boolean isArrayReorder) {
		this.arrayReorder = Boolean.valueOf(isArrayReorder);
		return this;
	}

	/**
	 * @param isArrayIndicesHidden set arrayIndicesHidden
	 * @return this
	 */
	public @NonNull ControlOptions arrayIndicesHidden(final boolean isArrayIndicesHidden) {
    	this.arrayIndicesHidden = Boolean.valueOf(isArrayIndicesHidden);
        return this;
    }
    
	/**
	 * @param isArrayHorizontal set arrayHorizontal
	 * @return this
	 */
    public @NonNull ControlOptions arrayHorizontal(final boolean isArrayHorizontal) {
    	this.arrayHorizontal = Boolean.valueOf(isArrayHorizontal);
        return this;
    }
    
    /**
	 * @param isLabelHidden set labelHidden
	 * @return this
	 */
    public @NonNull ControlOptions labelHidden(final boolean isLabelHidden) {
    	this.labelHidden = Boolean.valueOf(isLabelHidden);
        return this;
    }
    
    /**
	 * @param isBorderHidden set borderHidden
	 * @return this
	 */
    public @NonNull ControlOptions borderHidden(final boolean isBorderHidden) {
    	this.borderHidden = Boolean.valueOf(isBorderHidden);
        return this;
    }

    /**
     * @param setTableColumnWidths set tableColumnWidths
     * @return this
     */
    public @NonNull ControlOptions tableColumnWidths(final @Nullable List<@NonNull Integer> setTableColumnWidths) {
    	this.tableColumnWidths = setTableColumnWidths;
    	return this;
    }
    	
    /**
     * Merge options into existing options. Defined options (not null) in argument overrides options in this object. 
     * @param overrideOptions ControlOptions to be merged into existing options
     * @return this
     */
    public @NonNull ControlOptions merge(@NonNull ControlOptions overrideOptions) {
    	Field[] fields = this.getClass().getDeclaredFields();
    	for (Field field : fields) {
    		if ((field.getModifiers() & (Modifier.FINAL | Modifier.STATIC)) == 0) { // update only non-static, non-final fields
	    		// get value
	    		Object value;
	    		try {
	    			value = field.get(overrideOptions);
	    			// if value is defined, set it into this options
	    			if (value != null) {
	    				field.set(this, value);
	    			}
	    		} catch (IllegalArgumentException | IllegalAccessException e) {
	    			LOGGER.warning("Error in merging control options: " + e.getMessage()); //$NON-NLS-1$
	    		}
    		}
    	}
    	return this;
    }

	/**
	 * Extracts integer array content from string 
	 * @param values comma-separated signed integers
	 * @return array of integer values
	 */
	private static @NonNull List<@NonNull Integer> parseIntArray(@NonNull String values) {
		final List<@NonNull Integer> result = new ArrayList<>();
		final String[] integers = values.split(UtilsText.COMMA);
		for (String integer : integers) {
			try {
				result.add(Integer.valueOf(integer.trim()));
			} catch (NumberFormatException e) {
				LOGGER.log(Level.SEVERE, String.format("Invalid array of integers '%1s': %2s", values, e.getMessage())); //$NON-NLS-1$
				break;
			}
		}
		return result;
	}
}
