/**
 * Copyright 2017-2019 NXP
 * Created: Aug 28, 2017
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.Collection;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.Map.Entry;
import java.util.logging.Logger;

import org.eclipse.core.runtime.IPath;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.WorkbenchException;
import org.eclipse.ui.XMLMemento;

import com.nxp.swtools.common.ui.utils.perspectives.PerspectivesHelper;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionMap;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.lang.CollectionsUtils.IdentitySet;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.kex.MfactConstants;
import com.nxp.swtools.kex.selector.IMcuIdentification;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.controller.events.EventTypes;
import com.nxp.swtools.periphs.gui.perspective.PeripheralsPerspective;
import com.nxp.swtools.periphs.model.config.ComponentConfig;
import com.nxp.swtools.periphs.model.config.ComponentInstanceConfig;
import com.nxp.swtools.periphs.model.config.FunctionalGroup;
import com.nxp.swtools.provider.configuration.SharedConfigurationFactory;
import com.nxp.swtools.provider.configuration.storage.periphs.AStoragePeriphsComponent;
import com.nxp.swtools.provider.configuration.storage.periphs.StoragePeriphsComponent;
import com.nxp.swtools.provider.configuration.storage.periphs.StoragePeriphsComponentInstance;
import com.nxp.swtools.provider.configuration.storage.periphs.StoragePeriphsFuncGroup;
import com.nxp.swtools.utils.events.IEventListener;
import com.nxp.swtools.utils.events.ToolEvent;
import com.nxp.swtools.utils.storage.StorageHelper;

/**
 * Helper class for ComponentSettingView class
 * @author Tomas Rudolf - nxf31690
 */
public class ComponentSettingViewHelper {
	/** Logger of class */
	private static final @NonNull Logger LOGGER = LogManager.getLogger(ComponentSettingViewHelper.class);
	/**
	 *  SetMap of opened editors. HashSet<FuncGroup, Component>
	 *  Each functional group contains set of components.
	 *  Each component is either instance config or global config
	 */
	private @NonNull CollectionMap<@NonNull String, @NonNull AStoragePeriphsComponent> editors = new CollectionMap<>(IdentityHashMap.class, IdentitySet.class);
	/** Key for type */
	public static final @NonNull String TYPE = "type"; //$NON-NLS-1$
	/** Key for name */
	public static final @NonNull String NAME = "name"; //$NON-NLS-1$
	/** Key for global */
	public static final @NonNull String GLOBAL = "global"; //$NON-NLS-1$
	/** Key for group */
	public static final @NonNull String GROUP = "group"; //$NON-NLS-1$
	/** Name of editor node in memento */
	public static final @NonNull String EDITOR = "editor"; //$NON-NLS-1$
	/** editors contant */
	public static final @NonNull String EDITORS = "editors"; //$NON-NLS-1$
	/** Storage ID prefix */
	public static final @NonNull String EDITORS_UNDERSCORE = EDITORS + UtilsText.UNDERSCORE;
	/** ID of empty editor */
	public static final @NonNull String EMPTY_EDITOR_ID = "org.eclipse.ui.internal.emptyEditorTab"; //$NON-NLS-1$
	/** Disable removing editors form list of opened editors */
	public static final boolean REMOVE_DISABLED = true;
	/** Enable removing editors form list of opened editors */
	public static final boolean REMOVE_ENABLED = false;
	/** Storage for XML memento */
	@NonNull StorageHelper storageHelper = new StorageHelper(MfactConstants.TOOL_PERIPHERALS_ID);
	/** Current MCU identification */
	@NonNull IMcuIdentification mcuIdentification = Controller.getInstance().getMcu().getMcuIdentification();
	/** Reference to current functional group */
	@NonNull StoragePeriphsFuncGroup functionalGroupReference = Controller.getInstance().getFunctionalGroup().getStorageFuncGroup();
	/** Last hash of functional group reference */
	int functionalGroupReferenceHash = functionalGroupReference.hashCode();
	/** Current state of removing editors from list of opened editors */
	boolean removeState = REMOVE_ENABLED;

	/**
	 * Get singleton
	 * @return singleton of this class
	 */
	public static @NonNull ComponentSettingViewHelper getInstance() {
		try {
			ComponentSettingViewHelper helper = SWTFactoryProxy.INSTANCE.getSingletonInstance(ComponentSettingViewHelper.class);
			return helper;
		} catch (InstantiationException | IllegalAccessException e) {
			throw new IllegalStateException("Cannot obtain instance of a component editor helper", e); //$NON-NLS-1$
		}
	}

	/**
	 * Switches remove state from disabled to enabled and back
	 */
	public void switchRemoveState() {
		removeState = !removeState;
	}

	/**
	 * Sets given remove state as active
	 * @param newRemoveState new state
	 */
	public void setRemoveState(boolean newRemoveState) {
		removeState = newRemoveState;
	}

	/**
	 * Constructor
	 */
	public ComponentSettingViewHelper() {
		// Reset temporary location before saving to mex
		storageHelper.saveString(EDITORS_UNDERSCORE + UtilsText.EMPTY_STRING, UtilsText.EMPTY_STRING);
		Controller.getInstance().addListener(EventTypes.CHANGE, new IEventListener() {
			@Override
			public void handle(@NonNull ToolEvent event) {
				String currentPerspectiveId = PerspectivesHelper.getActivePerspectiveId();
				if (currentPerspectiveId == null) {
					// Do nothing if no perspective is active
					return;
				}
				if (!currentPerspectiveId.equals(PeripheralsPerspective.ID)) {
					// Do nothing if current perspective is not Peripherals
					return;
				}
				IViewPart view = getViewPart();
				// No view
				if (view == null) {
					return;
				}
				IViewSite viewSite = view.getViewSite();
				// No viewSite
				if (viewSite == null) {
					// Should not happen
					return;
				}
				IMcuIdentification mcuIdentificationNew = Controller.getInstance().getMcu().getMcuIdentification();
				Display display = Display.getCurrent();
				assert display != null;
				if (!mcuIdentification.equals(mcuIdentificationNew)) {
					// Common config has changed
					mcuIdentification = mcuIdentificationNew;
					display.asyncExec(new Runnable() {
						@Override
						public void run() {
							// Save functional group after loading of MEX/newMCU
							functionalGroupReference = Controller.getInstance().getFunctionalGroup().getStorageFuncGroup();
							functionalGroupReferenceHash = functionalGroupReference.hashCode();
							// Save and erase everything from old config from memory
							closeOpenedEditors();
							clearEditors();
							// Run restoring
							restoreEditors();
							showOpenedEditors(viewSite);
						}
					});
				} else {
					StoragePeriphsFuncGroup functionalGroupNew = Controller.getInstance().getFunctionalGroup().getStorageFuncGroup();
					// Functional group changed
					if (!functionalGroupReference.getUUID().equals(functionalGroupNew.getUUID())) {
						functionalGroupReference = functionalGroupNew;
						functionalGroupReferenceHash = functionalGroupReference.hashCode();
						display.asyncExec(new Runnable() {
							@Override
							public void run() {
								removeState = REMOVE_DISABLED;
								closeOpenedEditors();
								showOpenedEditors(viewSite);
								removeState = REMOVE_ENABLED;
							}
						});
					} else {
						// React to changes of name, prefix and selected core
						if (functionalGroupReferenceHash != functionalGroupReference.hashCode()) {
							saveEditors();
						}
					}
				}
			}
		});
	}

	/**
	 * Returns first non null view part. If none is found then returns null.
	 * @return first non null view part. If none is found then returns null.
	 */
	public static @Nullable IViewPart getViewPart() {
		IViewReference[] viewReferences = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().getViewReferences();
		for (IViewReference reference : viewReferences) {
			IViewPart view = reference.getView(false);
			if (view != null) {
				return view;
			}
		}
		return null;
	}

	/**
	 * Clears list of opened editors
	 */
	public void clearEditors() {
		editors.clear();
	}

	/**
	 * Removes editor from list of opened ones
	 * @param input for comparing which editor need to be closed
	 * @param funcGroupUuid for using as key
	 */
	public void removeEditor(@NonNull ComponentSettingViewInput input, @NonNull String funcGroupUuid) {
		// Do not remove anything in switching state. Otherwise you will close editor and delete it from persistent storage
		if (removeState == REMOVE_DISABLED) {
			return;
		}
		IdentitySet<@NonNull AStoragePeriphsComponent> originalSet = (IdentitySet<@NonNull AStoragePeriphsComponent>) editors.get(funcGroupUuid);
		IdentitySet<@NonNull AStoragePeriphsComponent> funcGroupEditors;
		if (originalSet == null) {
			funcGroupEditors = new IdentitySet<>();
		} else {
			funcGroupEditors = new IdentitySet<>(originalSet);
		}
		if (input.isGlobalConfig()) {
			for (@NonNull Entry<String, AStoragePeriphsComponent> entry : editors.flatEntrySet()) {
				AStoragePeriphsComponent component = entry.getValue();
				if (component instanceof StoragePeriphsComponent) {
					if (component.getName().equals(input.getComponentType())) {
						funcGroupUuid = UtilsText.safeString(entry.getKey());
						funcGroupEditors.add(component);
					}
				}
			}
		}
		if (funcGroupEditors.isEmpty()) {
			return;
		}
		@Nullable AStoragePeriphsComponent editorToBeRemoved =
			CollectionsUtils.nullableOptionalGet(funcGroupEditors.stream()
		.filter(x -> {
			if (input.isGlobalConfig()) {
				if (x instanceof StoragePeriphsComponent) {
					return x.getName().equals(input.getComponentType());
				}
			} else {
				if (x instanceof StoragePeriphsComponentInstance) {
					return x.getName().equals(input.getComponent());
				}
			}
			return false;
		})
		.findAny());
		if (editorToBeRemoved == null) {
			return;
		}
		if (editors.removeItem(funcGroupUuid, editorToBeRemoved)) {
			saveEditors();
		}
	}

	/**
	 * Adds editor to set of opened ones
	 * @param funcGroup name of functional group to add editor to
	 * @param component to be stored as opened editor
	 */
	public void addEditor(@NonNull StoragePeriphsFuncGroup funcGroup, @NonNull AStoragePeriphsComponent component) {
		if (component instanceof StoragePeriphsComponent) {
			// Skip when already in opened editors no matter in which group
			final Object comp = component;
			if (editors.containsValue(comp)) {
				return;
			}
		}
		Object found = CollectionsUtils.findAny(editors.flatEntrySet(), entry -> funcGroup.getUUID().equals(entry.getKey()) && component.equals(entry.getValue()));
		// Do not add something that is already there
		if (found == null) {
			editors.add(funcGroup.getUUID(), component);
		}
		saveEditors();
	}

	/**
	 * Looks up for MEX location
	 * @return path to MEX file
	 */
	private static String getMexFilePath() {
		@Nullable IPath locationPath = SharedConfigurationFactory.getSharedConfigurationSingleton().getLocationPath();
		String projectLocation;
		if (locationPath == null) {
			projectLocation = UtilsText.EMPTY_STRING;
		} else {
			projectLocation = locationPath.toOSString();
		}
		return projectLocation;
	}

	/**
	 * Shows currently opened editors from editors
	 * @param viewSite to create editors in
	 */
	public void showOpenedEditors(@NonNull IViewSite viewSite) {
		// Open new ones
		Iterator<Entry<@NonNull String, @NonNull AStoragePeriphsComponent>> iterator = editors.flatEntrySet().stream()
				.filter(e -> {
						Controller controller = Controller.getInstance();
						@SuppressWarnings("null") //to avoid false-positive warning
						String fgUuid = e.getKey();
						// filter only components which belong to the currently selected functional group, or are global
						return (fgUuid.equals(controller.getFunctionalGroup().getStorageFuncGroup().getUUID()))
								|| (e.getValue() instanceof StoragePeriphsComponent);
				}).iterator();
		while (iterator.hasNext()) {
			Entry<@NonNull String, @NonNull AStoragePeriphsComponent> entry = iterator.next();
			@SuppressWarnings("null") //to avoid false-positive warning
			AStoragePeriphsComponent config = entry.getValue();
			boolean global = false;
			String name = UtilsText.EMPTY_STRING;
			String type = UtilsText.EMPTY_STRING;
			if (config instanceof StoragePeriphsComponentInstance) {
				name = ((StoragePeriphsComponentInstance)config).getName();
				type = ((StoragePeriphsComponentInstance)config).getType();
			} else if (config instanceof StoragePeriphsComponent) {
				name = ((StoragePeriphsComponent)config).getName();
				type = ((StoragePeriphsComponent)config).getName();
				global = true;
			}
			ComponentSettingView.open(viewSite, type, name, global, false);
		}
	}

	/**
	 * Restores editors from XML memento
	 * @return {@code true} if nothing fails, otherwise returns {@code false}
	 */
	public boolean restoreEditors() {
		boolean result = false;
		try {
			String mexFilePath = getMexFilePath();
			if (UtilsText.isEmpty(mexFilePath)) {
				LOGGER.info("Empty MEX file path, not restoring editors"); //$NON-NLS-1$
				return false;
			}
			String mementoStrContent = storageHelper.loadString(EDITORS_UNDERSCORE + mexFilePath, UtilsText.EMPTY_STRING);
			if (UtilsText.EMPTY_STRING.equals(mementoStrContent)) {
				LOGGER.info("No editors saved for MEX file, not restoring editors"); //$NON-NLS-1$
				return false;
			}
			StringReader reader = new StringReader(mementoStrContent);
			// Prepare memento from storage
			XMLMemento xmlMemento = XMLMemento.createReadRoot(reader);
			// Read all children
			IMemento[] children = xmlMemento.getChildren();
			for (IMemento child : children) {
				String funcGroupUuid = UtilsText.safeString(child.getString(GROUP));
				FunctionalGroup functionalGroup = Controller.getInstance().getProfile().getFunctionalGroupWithUUID(funcGroupUuid);
				if (functionalGroup == null) {
					LOGGER.info("Trying to restore editors: Functional group with given id was not found in current profile"); //$NON-NLS-1$
					continue;
				}
				String type = UtilsText.safeString(child.getString(TYPE));
				String name = UtilsText.safeString(child.getString(NAME));
				Boolean globalBoolean = child.getBoolean(GLOBAL);
				if (globalBoolean == null) {
					LOGGER.warning("Global is not set in XML memento"); //$NON-NLS-1$
					continue;
				}
				boolean global = globalBoolean.booleanValue();
				AStoragePeriphsComponent configToSave;
				if (global) {
					ComponentConfig configuredComponent = Controller.getInstance().getConfiguredComponent(type);
					if (configuredComponent == null) {
						LOGGER.info("Trying to restore component setting view of component config that is not present in current functional group"); //$NON-NLS-1$
						continue;
					}
					configToSave = configuredComponent.getStorageComponent();
				} else {
					ComponentInstanceConfig instance = functionalGroup.getInstance(name);
					if (instance == null) {
						LOGGER.info("Trying to restore component setting view of component instance config that is not present in current functional group"); //$NON-NLS-1$
						continue;
					}
					configToSave = instance.getStorageComponent();
				}
				addEditor(functionalGroup.getStorageFuncGroup(), configToSave);
			}
			result = true;
		} catch (WorkbenchException e) {
			LOGGER.severe("XMLMemento exception occured. Detail: " + e.getMessage()); //$NON-NLS-1$
		}
		return result;
	}

	/**
	 * Saves current list of opened editors to XML memento
	 * @return {@code true} if nothing fails, otherwise returns {@code false}
	 */
	public boolean saveEditors() {
		boolean result = false;
		StringWriter writer= new StringWriter();
		try {
			XMLMemento xmlMemento = XMLMemento.createWriteRoot("editors"); //$NON-NLS-1$
			// Each entry needs to be stored
			for (Entry<@NonNull String, @NonNull AStoragePeriphsComponent> editorSettings : editors.flatEntrySet()) {
				@SuppressWarnings("null") //to avoid false-positive warning
				String funcGroupUuid = editorSettings.getKey();
				@SuppressWarnings("null") //to avoid false-positive warning
				AStoragePeriphsComponent component = editorSettings.getValue();
				// Create node and set values
				IMemento node = xmlMemento.createChild(EDITOR);
				node.putString(NAME, component.getName());
				String global = UtilsText.EMPTY_STRING;
				if (component instanceof StoragePeriphsComponentInstance) {
					node.putString(TYPE, ((StoragePeriphsComponentInstance)component).getType());
					global = String.valueOf(false);
				} else if(component instanceof StoragePeriphsComponent) {
					node.putString(TYPE, ((StoragePeriphsComponent)component).getName());
					global = String.valueOf(true);
				} else {
					LOGGER.severe("Editor given for save is not instance or component config editor"); //$NON-NLS-1$
					continue;
				}
				node.putString(GLOBAL, global);
				node.putString(GROUP, funcGroupUuid);
			}
			// Save and write out
			xmlMemento.save(writer);
			storageHelper.saveString(EDITORS_UNDERSCORE + getMexFilePath(), UtilsText.safeString(writer.toString()));
			result = true;
		} catch (IOException e) {
			LOGGER.info("File exception occured. Detail: " + e.getMessage()); //$NON-NLS-1$
		}
		return result;
	}

	/**
	 * Close all editors opened in UI
	 */
	public void closeOpenedEditors() {
		IWorkbench workbench = PlatformUI.getWorkbench();
		if (workbench == null) return;
		IWorkbenchWindow window = workbench.getActiveWorkbenchWindow();
		if (window == null) return;
		IWorkbenchPage activePage = workbench.getActiveWorkbenchWindow().getActivePage();
		if (activePage == null) return;
		IViewReference[] viewReferences = activePage.getViewReferences();
		for (int index = 0; index < viewReferences.length; index++) {
			IViewReference viewRef = viewReferences[index];
			String id = viewRef.getId();
			if (id.equals(ComponentSettingView.ID)) {
				activePage.hideView(viewRef);
			}
		}
	}

	/**
	 * Removes editors of given group from opened ones
	 * @param group in which to close all opened editors
	 */
	public void removeEditorsOfGroup(@NonNull StoragePeriphsFuncGroup group) {
		Collection<@NonNull AStoragePeriphsComponent> editorsOfGroup = editors.get(group.getUUID()); 
		if (editorsOfGroup != null) {
			editorsOfGroup.clear();
			editors.remove(group.getUUID());
		}
	}

	/**
	 * Reopens views where their secondary ID does not correspond to their type or name
	 */
	public void reopenInvalidViews() {
		IWorkbench workbench = PlatformUI.getWorkbench();
		IWorkbenchPage activePage = workbench.getActiveWorkbenchWindow().getActivePage();
		IViewReference[] viewReferences = activePage.getViewReferences();
		for (int index = 0; index < viewReferences.length; index++) {
			String primaryID = viewReferences[index].getId();
			String secondaryID = UtilsText.safeString(viewReferences[index].getSecondaryId());
			if (primaryID.equals(ComponentSettingView.ID)) {
				ComponentSettingView view = (ComponentSettingView) viewReferences[index].getView(false);
				if (view == null) {
					continue;
				}
				ComponentSettingViewInput componentInput = view.componentInput;
				if (componentInput == null) {
					continue;
				}
				if (!secondaryID.equals(createSecondaryId(componentInput))) {
					// Close and open again
					activePage.hideView(view);
					ComponentSettingView.open(activePage, componentInput.getComponentType(), componentInput.getComponent(), componentInput.isGlobalConfig(), false);
				}
			}
		}
	}

	/**
	 * Creates secondary ID for view
	 * @param componentInput to get information from
	 * @return secondary ID string
	 */
	public static @NonNull String createSecondaryId(@NonNull ComponentSettingViewInput componentInput) {
		return componentInput.getComponentType() + ComponentSettingView.SECONDARY_ID_NAME_TYPE_SEPARATOR
				+ componentInput.getComponent();
	}
}
