/**
 * Copyright 2017-2019 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;
import java.util.logging.Logger;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Link;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.ToolItem;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.IWorkbenchPartSite;
import org.eclipse.ui.PartInitException;

import com.nxp.swtools.common.ui.utils.swt.ControlDecorationUtils;
import com.nxp.swtools.common.ui.utils.swt.FontFactory;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.ui.utils.swt.widgets.InstantSearchList;
import com.nxp.swtools.common.ui.utils.swt.widgets.ToggleButton;
import com.nxp.swtools.common.ui.utils.views.PartListener2Adapter;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.os.OSDetect;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.controller.ValidationHelper;
import com.nxp.swtools.periphs.controller.events.EventTypes;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.controller.PeriphControllerWrapper;
import com.nxp.swtools.periphs.gui.utils.GcUtils;
import com.nxp.swtools.periphs.gui.view.APeriphsViewBase;
import com.nxp.swtools.periphs.gui.view.DocumentationView;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType;
import com.nxp.swtools.periphs.gui.view.componentsettings.TextBoxHelper.Status;
import com.nxp.swtools.periphs.model.config.ComponentConfig;
import com.nxp.swtools.periphs.model.config.ComponentInstanceConfig;
import com.nxp.swtools.periphs.model.config.ConfigSetConfig;
import com.nxp.swtools.periphs.model.config.IChild;
import com.nxp.swtools.periphs.model.data.ConfigurationComponentTypeId;
import com.nxp.swtools.periphs.model.data.MasterPeripheral;
import com.nxp.swtools.periphs.model.data.SettingOptions;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.provider.configuration.storage.periphs.AStoragePeriphsComponent;
import com.nxp.swtools.utils.Limits;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.events.IEventListener;
import com.nxp.swtools.utils.events.ToolEvent;
import com.nxp.swtools.utils.progress.ProgressUtils;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsColors.SwToolsColors;
import com.nxp.swtools.utils.resources.ToolsImages;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;
import com.nxp.swtools.utils.tooltip.ToolTipableImplementation;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;
import com.nxp.swtools.utils.view.ToolView;
import com.nxp.swtools.validation.engine.IValidationProblem;
import com.nxp.swtools.validation.engine.ValidationEngineFactory;
import com.nxp.swtools.validation.engine.ValidationProblemListenerAdapter;

/**
 * Editor used for editing component instance settings.
 * @author Juraj Ondruska
 * @author Tomas Rudolf - nxf31690
 */
public class ComponentSettingView extends ToolView {
	/** Logger of class */
	static final @NonNull Logger LOGGER = LogManager.getLogger(ComponentSettingView.class);
	/** Separator used to divide component type and component name in the secondary ID of the view */
	static final @NonNull String SECONDARY_ID_NAME_TYPE_SEPARATOR = "/"; //$NON-NLS-1$
	/** Number of columns in the component view title (Title, documentation link, enable button) */
	public static final int TITLE_ROW_COLS = 3;
	/** Number of columns in the component settings (name, peripheral, etc.) */
	public static final int COMPONENT_COLS = 2;
	/** Number of columns in the config set settings */
	public static final int CONFIG_SET_COLS = 3;
	/** Number of columns in the set settings */
	public static final int SET_COLS = 2;
	/** Number of columns in label section of configSet */
	public static final int NUM_OF_CONFIGSET_LABEL_COLUMNS = 2;
	/** ID of the editor */
	public static final @NonNull String ID = "com.nxp.swtools.periphs.gui.view.componentsettings"; //$NON-NLS-1$
	/** Key for error decoration */
	public static final @NonNull String ERROR_DECORATION_KEY = "error_decoration"; //$NON-NLS-1$
	/** Minimal width of the editor in ems */
	private static final int MIN_WIDTH_EMS = 45;
	/** The "top-most" composite with scroll ability */
	protected ScrolledComposite scrolledComposite;
	/** Content of the editor */
	protected Composite contentComposite;
	/** Controller associated with the editor */
	final Controller controller = Controller.getInstance();
	/** Component editor input */
	ComponentSettingViewInput componentInput;
	/** {@link ComponentInstanceControl} associated with the editor */
	private @Nullable IChildControl componentSettingControl = null;
	/** Listeners which were registered and need to be unregistered during dispose */
	final @NonNull Collection<@NonNull IEventListener> registeredListeners = new ArrayList<>();
	/** Controls of all displayed config sets */
	final @NonNull Map<@NonNull ConfigSetConfig, ConfigSetControl> configSetControls = new HashMap<>();
	/** Combo used for configuration of component mode */
	private @Nullable InstantSearchList comboBoxMode;
	/** Resize listener instance */
	private @Nullable Listener resizeListener;
	/** Label for name textbox */
	private @Nullable Label labelName;
	/** Textbox for name */
	private @Nullable Text textBoxName;
	/** Label for mode combo */
	private @Nullable Label labelMode;
	/** Label for peripherals combo */
	private @Nullable Label labelPeripheral;
	/** Combobox for peripheral */
	private @Nullable InstantSearchList comboBoxPeripheral;
	/** Stores validity of current name of instance */
	boolean nameIsValid = true;
	
	/** Associated controller wrapper */
	protected final @NonNull IControllerWrapper controllerWrapper = PeriphControllerWrapper.getInstance();
	/** State of the view - uninitialized by default */
	@NonNull State state = State.UNINITIALIZED;
	/** Action which needs to be performed after the view becomes visible */
	@NonNull ActionRequired requiredAction = ActionRequired.NO_ACTION;
	/** Listener for validations changes */
	private @NonNull ValidationProblemListenerAdapter validationsListener = new ValidationProblemListenerAdapter() {
		@Override
		public void validationProblemsChanged(@NonNull Collection<@NonNull IValidationProblem> problems) {
			if ((contentComposite != null) && !contentComposite.isDisposed()) {
				refreshIfVisible(UpdateType.PROBLEM_DECORATION);
			}
		}
	};

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.ViewPart#init(org.eclipse.ui.IViewSite)
	 */
	@Override
	public void init(IViewSite site) throws PartInitException {
		setSite(site);
		String secondaryId = site.getSecondaryId();
		if (secondaryId == null) {
			return;
		}
		String [] split = secondaryId.split(SECONDARY_ID_NAME_TYPE_SEPARATOR);
		String componentType = UtilsText.safeString(split[0]);
		String componentName = split.length > 1 ? UtilsText.safeString(split[1]) : UtilsText.EMPTY_STRING;
		this.componentInput = new ComponentSettingViewInput(componentType, componentName, UtilsText.isEmpty(componentName));
		IChild inputSetting = getComponent();
		setPartName((inputSetting == null) ? null : inputSetting.getUiName());
		getSiteSafe().getPage().addPartListener(new IPartListener() {
			@Override
			public void partOpened(IWorkbenchPart part) {
				if (part.equals(ComponentSettingView.this)) {
					ComponentConfig component = controller.getConfiguredComponent(componentInput.getComponentType());
					ComponentInstanceConfig componentInstance = controller.getComponentSetting(componentInput.getComponentType(),
							componentInput.getComponent());
					AStoragePeriphsComponent configToSave = null;
					// Use component instance if it is not null, otherwise use component (can be null)
					if (componentInstance != null) {
						configToSave = componentInstance.getStorageComponent();
					} else if (component != null) {
						 configToSave = component.getStorageComponent();
					} else {
						// probably residue from the previous run, will be closed later
						return;
					}
					ComponentSettingViewHelper.getInstance()
							.addEditor(Controller.getInstance().getFunctionalGroup().getStorageFuncGroup(), configToSave);
				}
			}
			@Override
			public void partDeactivated(IWorkbenchPart part) {
				// Do nothing
			}
			@Override
			public void partClosed(IWorkbenchPart part) {
				if (part.equals(ComponentSettingView.this)) {
					ComponentSettingViewInput inputOfClosedEditor = componentInput;
					if (inputOfClosedEditor != null) {
						ComponentSettingViewHelper.getInstance().removeEditor(inputOfClosedEditor,
								Controller.getInstance().getFunctionalGroup().getStorageFuncGroup().getUUID());
					}
					getSiteSafe().getPage().removePartListener(this);
				}
			}
			@Override
			public void partBroughtToTop(IWorkbenchPart part) {
				// Do nothing
			}
			@Override
			public void partActivated(IWorkbenchPart part) {
				// Do nothing
			}
		});
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#dispose()
	 */
	@Override
	public void dispose() {
		ValidationEngineFactory.removeListener(validationsListener);
		for (IEventListener listener : registeredListeners) {
			controller.removeListener(listener);
		}
		super.dispose();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createPartControl(Composite parent) {
		if (componentInput == null) {
			return;
		}
		scrolledComposite = new ScrolledComposite(parent, SWT.V_SCROLL | SWT.H_SCROLL);
		SWTFactoryProxy.INSTANCE.setTestId(scrolledComposite, TestIDs.PERIPHS_COMPONENT_SETTING_VIEW_CONTENT);
		contentComposite = createDefaultComposite(scrolledComposite);
		GridLayoutComponents layout = new GridLayoutComponents(COMPONENT_COLS, false);
		layout.marginWidth = 8;
		layout.horizontalSpacing = 8;
		contentComposite.setLayout(layout);
		scrolledComposite.setContent(contentComposite);
		scrolledComposite.setExpandHorizontal(true);
		scrolledComposite.setExpandVertical(true);
		getSiteSafe().getPage().addPartListener(new PartListener2Adapter() {
			@Override
			public void partVisible(IWorkbenchPartReference partRef) {
				if (getViewSiteNonNull().getPart() == partRef.getPart(false)) {
					if (state == State.CREATED_HIDDEN) {
						// the view was hidden before, make it visible and refresh its content if required
						state = State.CREATED_VISIBLE;
						if (requiredAction == ActionRequired.RECREATE) {
							recreate();
						} else if (requiredAction == ActionRequired.REFRESH) {
							refreshSettings();
						}
						requiredAction = ActionRequired.NO_ACTION;
					} else if (state == State.READY_TO_CREATE) {
						// the view is ready to be created, create it
						state = State.CREATED_VISIBLE;
						recreate();
						requiredAction = ActionRequired.NO_ACTION;
					}
				}
			}
			@Override
			public void partHidden(IWorkbenchPartReference partRef) {
				if (getViewSiteNonNull().getPart() == partRef.getPart(false)) {
					if (state == State.CREATED_VISIBLE) {
						// the view was visible before, make it hidden and clear the required action flag
						state = State.CREATED_HIDDEN;
						requiredAction = ActionRequired.NO_ACTION;
					}
				}
			}
			@Override
			public void partClosed(IWorkbenchPartReference partRef) {
				if (getViewSiteNonNull().getPart() == partRef.getPart(false)) {
					getSiteSafe().getPage().removePartListener(this);
					state = State.UNINITIALIZED;
					requiredAction = ActionRequired.NO_ACTION;
				}
			}
		});
		// create content of the view asynchronously as it is possible that the view becomes hidden meanwhile, postpone creation in that case
		parent.getDisplay().asyncExec(() -> {
			if (state == State.UNINITIALIZED) {
				// the view is not created yet
				state = State.READY_TO_CREATE;
				if (getSiteSafe().getPage().isPartVisible(getSiteSafe().getPart())) {
					// the view is still visible, create its content
					recreate();
					contentComposite.forceFocus();
					state = State.CREATED_VISIBLE;
				}
			}
		});
		registerEventListener();
		ValidationEngineFactory.addListener(validationsListener);
		updateErrorIndicators();
	}

	/**
	 * @return nonnull IViewSite guearded by assert
	 */
	protected @NonNull IViewSite getViewSiteNonNull() {
		IViewSite site = getViewSite();
		assert site != null;
		return site;
	}

	/**
	 * Register listener responsible for closing the editor if it is not needed anymore or refreshing the content.
	 */
	private void registerEventListener() {
		// register close listener
		IEventListener changesListener = new IEventListener() {
			/* (non-Javadoc)
			 * @see com.nxp.swtools.utils.events.IEventListener#handle(com.nxp.swtools.utils.events.ToolEvent)
			 */
			@Override
			public void handle(ToolEvent event) {
				boolean shouldCloseView = false;
				boolean shouldRemoveViewFromOpened = false;
				String componentName = componentInput.getComponent();
				String componentType = componentInput.getComponentType();
				if (componentInput.isGlobalConfig()) {
					shouldCloseView = controller.getProfile().getConfiguredComponents().values().stream()
							.noneMatch(config -> config.getName().equals(componentType));
					// Global configuration does not exist -> remove it from opened views
					shouldRemoveViewFromOpened = shouldCloseView;
				} else {
					shouldCloseView = controller.getFunctionalGroup().getInstances().values().stream()
							.noneMatch(instance -> instance.getUiName().equals(componentName));
					shouldRemoveViewFromOpened = controller.getProfile().getAllInstances().stream()
							.noneMatch(instance -> instance.getUiName().equals(componentName));
				}
				// Remove from opened views in helper when no configuration instance for this view exists
				if (shouldRemoveViewFromOpened){
					ComponentSettingViewInput componentInputReference = componentInput;
					if (componentInputReference != null) {
						ComponentSettingViewHelper.getInstance().removeEditor(componentInputReference,
								controller.getFunctionalGroup().getStorageFuncGroup().getUUID());
					}
				}
				// If no instance with required type and name exists then hide this view
				if (shouldCloseView) {
					Display display = Display.getCurrent();
					if (display != null) {
						display.asyncExec(new Runnable() {
							@Override
							public void run() {
								getViewSiteNonNull().getWorkbenchWindow().getActivePage().hideView(ComponentSettingView.this);
							}
						});
					}
				}
				// refresh the content asynchronously (avoid concurrent modification exception)
				if (event.originator != ComponentSettingView.this) {
					if (state == State.CREATED_VISIBLE) {
						// the view is visible, recreate it
						Display display = Display.getCurrent();
						if (display != null) {
							display.asyncExec(() -> recreate());
						}
					} else if (state == State.CREATED_HIDDEN) {
						// the view is not visible, indicate that the view needs to be recreated
						requiredAction = ActionRequired.RECREATE;
					}
				}
				updateErrorIndicators();
			}
		};
		controller.addListener(EventTypes.CHANGE, changesListener);
		registeredListeners.add(changesListener);
		// register setting value change listener
		IEventListener settingListener = new IEventListener() {
			/* (non-Javadoc)
			 * @see com.nxp.swtools.utils.events.IEventListener#handle(com.nxp.swtools.utils.events.ToolEvent)
			 */
			@Override
			public void handle(ToolEvent event) {
				refreshIfVisible(UpdateType.NORMAL);
			}
		};
		controller.addListener(EventTypes.SETTING_CHANGE, settingListener);
		registeredListeners.add(settingListener);
	}

	/**
	 * Refresh the View in case it is visible (update title in case the View is not visible).
	 * @param updateType type of the update
	 */
	protected void refreshIfVisible(@NonNull UpdateType updateType) {
		if (state == State.CREATED_HIDDEN) {
			if (requiredAction != ActionRequired.RECREATE) {
				// the refresh should be scheduled only if the recreate is not scheduled yet
				requiredAction = ActionRequired.REFRESH;
			}
			updateErrorIndicators();
		} else {
			refreshSettings(updateType);
		}
	}

	/**
	 * Update error indicator(s) of the editor's title given the current state.
	 */
	protected void updateErrorIndicators() {
		IChild component = getComponent();
		Image image = null;
		if (component != null) {
			final int problemLevel = ValidationHelper.getHighestSeverityComponentValidationProblemLevel(component);
			if (componentInput.isGlobalConfig()) {
				image = APeriphsViewBase.getDecoratedImage(IToolsImages.IMAGE_EDITOR_COMPONENT_GLOBAL, problemLevel);
			} else {
				image = APeriphsViewBase.getDecoratedImage(IToolsImages.IMAGE_EDITOR_COMPONENT_LOCAL, problemLevel);
			}
		}
		setTitleImage(image);
	}

	/**
	 * Update error decorators of the non-config set related editors (peripheral selection, mode selection, name selection).
	 */
	protected void updateErrorDecorators() {
		ComponentInstanceConfig compInstance = controller.getComponentSetting(componentInput.getComponentType(), componentInput.getComponent());
		if (compInstance != null) {
			Control editors[] = {textBoxName, comboBoxMode, comboBoxPeripheral};
			Supplier<?> suppliers[] = {() -> compInstance.getNotUniqueNameError(), null, () -> getPeripheralError(compInstance)};
			assert (editors.length == suppliers.length);
			for (int i = 0; i < editors.length; i++) {
				Control editor = editors[i];
				@SuppressWarnings("unchecked")
				Supplier<@Nullable String> supplier = (Supplier<@Nullable String>) suppliers[i];
				if ((editor != null) && (supplier != null)) {
					ControlDecorationUtils.updateControlDecoration(editor, supplier);
				}
			}
		}
	}

	/**
	 * Creates first row in the view.
	 * Contains Title, documentation link and enable/disable button if applicable.
	 * @param parent composite in which row should be created
	 */
	void createTitleRow(@NonNull Composite parent) {
		boolean isDocumentationPresent = false;
		ConfigurationComponentTypeId configCompTypeId = null;
		String titleString = UtilsText.EMPTY_STRING;
		ComponentInstanceConfig componentInstanceConfig = controller.getComponentSetting(componentInput.getComponentType(), componentInput.getComponent());
		ComponentConfig componentConfig = controller.getConfiguredComponent(componentInput.getComponentType());
		int titleGridDataAdvancedColumns = 1; // default num columns

		Composite titleRow = new Composite(parent, SWT.NONE);
		titleRow.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, COMPONENT_COLS, 1));
		// set layout to TITLE_ROW_COLS columns
		final GridLayoutComponents titleRowLayout = new GridLayoutComponents(TITLE_ROW_COLS, false);
		titleRowLayout.marginWidth = titleRowLayout.marginHeight = 0;
		titleRow.setLayout(titleRowLayout);

		// create title
		Label title = new Label(titleRow, SWT.WRAP);
		FontFactory.changeStyle(title, SWT.BOLD);
		FontFactory.scaleFontSize(title, 1.5);

		// get info for creation of link to documentation
		if (componentInput.isGlobalConfig() && (componentConfig != null)) {
			titleString = componentConfig.getComponent().getResolvedDescription(componentConfig.getExpressionContext());
			if (titleString == null) {
				titleString = componentConfig.getComponent().getLabel(componentConfig.getExpressionContext());
			}
			configCompTypeId = componentConfig.getConfigCompTypeId();
			isDocumentationPresent = configCompTypeId.isDocumentationPresent();
		}
		if (!componentInput.isGlobalConfig() && (componentInstanceConfig != null)) {
			titleString = componentInstanceConfig.getComponent().getResolvedDescription(componentInstanceConfig.getExpressionContext());
			if (titleString == null) {
				titleString = componentInstanceConfig.getComponent().getLabel(componentInstanceConfig.getExpressionContext());
			}
			configCompTypeId = componentInstanceConfig.getConfigCompTypeId();
			isDocumentationPresent = componentInstanceConfig.getConfigCompTypeId().isDocumentationPresent();
		}

		// create documentation link
		if ((isDocumentationPresent) && (configCompTypeId != null)) {
			createLinkToDocumentation(titleRow, configCompTypeId);
		} else {
			titleGridDataAdvancedColumns++; // documentation button not needed -> take one more column
		}

		// if input is component instance config - create also enable/disable button
		if (!componentInput.isGlobalConfig() && (componentInstanceConfig != null)) {
			createEnableButton(titleRow, componentInstanceConfig);
		} else {
			titleGridDataAdvancedColumns++; // no enable button -> take one more column
		}

		// set title
		title.setText(titleString);
		title.setLayoutData(new GridDataComponents(SWT.FILL, SWT.FILL, true, false, titleGridDataAdvancedColumns, 1));
	}

	/**
	 * Create GUI elements for the selected component.
	 */
	void createContent() {
		Composite contentCompositeLoc = contentComposite;
		if (contentCompositeLoc != null) {
			createTitleRow(contentCompositeLoc);
			if (componentInput.isGlobalConfig()) {
				ComponentConfig componentConfig = controller.getConfiguredComponent(componentInput.getComponentType());
				if (componentConfig != null) {
					Label label = new Label(contentCompositeLoc, SWT.NONE);
					label.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, true, false, COMPONENT_COLS, 1));
					FontFactory.changeStyle(label, SWT.BOLD);
					label.setText(MessageFormat.format(UtilsText.safeString(Messages.get().ComponentEditor_GlobalConfigurationOf), componentConfig.getUiName()));
					if (!UtilsText.isEmpty(componentConfig.getDescription())) {
						label = new Label(contentCompositeLoc, SWT.SEPARATOR | SWT.HORIZONTAL);
						label.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, COMPONENT_COLS, 1));
						label = new Label(contentCompositeLoc, SWT.WRAP);
						label.setLayoutData(new GridDataComponents(SWT.FILL, SWT.TOP, true, false, COMPONENT_COLS, 1));
						label.setText(UtilsText.safeString(componentConfig.getDescription()));
					}
					ConfigSetConfig configSetConfig = componentConfig.getGlobalConfigSet();
					if (configSetConfig != null) {
						updateConfigSetsContent(CollectionsUtils.asList(configSetConfig), contentCompositeLoc);
					} else {
						label = new Label(contentCompositeLoc, SWT.NONE);
						label.setText(Messages.get().ComponentSettingView_ComponentHasNoGlobalConfigSet);
						FontFactory.changeStyle(label, SWT.BOLD);
						FontFactory.scaleFontSize(label, 1.5);
					}
				}
			} else {
				ComponentInstanceConfig componentSetting = controller.getComponentSetting(componentInput.getComponentType(), componentInput.getComponent());
				if (componentSetting != null) {
					Label label = new Label(contentCompositeLoc, SWT.NONE);
					labelName = label;
					label.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false));
					label.setText(Messages.get().ComponentEditor_Name);
					Text text = new Text(contentCompositeLoc, SWT.BORDER);
					textBoxName = text;
					ControlDecorationUtils.createErrorDecoration(text, SWT.NONE, ToolsImages.getStatusDecoratorImg(ErrorLevels.LEVEL_ERROR));
					SWTFactoryProxy.INSTANCE.setTestId(text, TestIDs.PERIPHS_COMPONENT_NAME_EDITOR);
					text.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, false, false, COMPONENT_COLS - 1, 1));
					text.setTextLimit(Limits.INPUT_LENGTH_C_COMPATIBLE);
					text.setText(componentInput.getComponent());
					TextBoxHelper.attachModifyListeners(text, (newName) -> {
						// change, use new name
						controller.runTransaction(() -> {
							// this must run as a transaction - listeners should be fired after the component input is updated,
							// the editor is closed otherwise because there is conflict in component name and name in component input
							if (controller.renameComponentInstance(componentSetting, newName, ComponentSettingView.this)) {
								componentInput.setComponent(newName);
								setPartName(newName);
								updateControls();
							}
						});
					});
					TextBoxHelper.attachModifyErrorListener(() -> componentSetting.getUiName(), text, (newName) -> {
						boolean isSameText = componentSetting.getUiName().equalsIgnoreCase(newName);
						boolean isNameUsable = controller.isNameUsable(newName);
						String errorText = null;
						if (!isNameUsable) {
							errorText = Messages.get().ComponentEditor_NameNotUniqueError;
						}
						// if newName is the same as previous name -> check for uniqueness error
						// (can be caused by setting same prefix for fn groups which have comp. instances with same name)
						if (isSameText) {
							errorText = componentSetting.getNotUniqueNameError();
						}
						final String error = errorText;
						if (errorText != null) {
							nameIsValid = false;
						} else {
							nameIsValid = true;
						}
						ControlDecorationUtils.updateControlDecoration(text, () -> error);
						updateTooltips();
						if (errorText != null) {
							return Status.INVALID;
						} else {
							return Status.OK;
						}
					});
					if (componentSetting.getNotUniqueNameError() != null) {
						nameIsValid = false;
						text.setBackground(SwToolsColors.ERROR_BG.getColor());
						text.setForeground(SwToolsColors.ERROR_FG.getColor());
					}
					if (!componentSetting.isOptionSet(SettingOptions.UI_COMPONENT_MODE_HIDDEN)) {
						label = new Label(contentCompositeLoc, SWT.NONE);
						labelMode = label;
						label.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false));
						label.setText(Messages.get().ComponentEditor_Mode);
						InstantSearchList modeComboLoc = new InstantSearchList(contentCompositeLoc, SWT.BORDER);
						this.comboBoxMode = modeComboLoc;
						SWTFactoryProxy.INSTANCE.setTestId(modeComboLoc, TestIDs.PERIPHS_COMPONENT_MODE_COMBO);
						modeComboLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.FILL, false, false, COMPONENT_COLS - 1, 1));
						modeComboLoc.setItems(componentSetting.getComponent().getScenarios().stream()
								.filter(y -> y.isAvailable(componentSetting.getExpressionContext()))
								.map(y -> y.getUIName(componentSetting.getExpressionContext()))
								.toArray(String[]::new));
						modeComboLoc.select(componentSetting.getMode().getUIName(componentSetting.getExpressionContext()));
						modeComboLoc.addSelectionListener(new SelectionAdapter() {
							/* (non-Javadoc)
							 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
							 */
							@Override
							public void widgetSelected(SelectionEvent e) {
								IWorkbenchPartSite site = getSite();
								assert site != null;
								if (!site.getPage().isPartVisible(ComponentSettingView.this)) {
									return;
								}
								String modeSelection = UtilsText.safeString(((InstantSearchList) e.widget).getText());
								String type = componentInput.getComponentType();
								String name = componentInput.getComponent();
								ProgressUtils.run((m) -> {
									controller.changeMode(type, name, modeSelection, ComponentSettingView.this);
								});
								recreate();
							}
						});
					}
					MasterPeripheral masterPeripheral = componentSetting.getMode().getMasterPeripheral();
					if (masterPeripheral != null) {
						label = new Label(contentCompositeLoc, SWT.NONE);
						labelPeripheral = label;
						label.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false));
						label.setText(Messages.get().ComponentEditor_Peripheral);
						InstantSearchList combo = new InstantSearchList(contentCompositeLoc, SWT.BORDER);
						comboBoxPeripheral = combo;
						SWTFactoryProxy.INSTANCE.setTestId(combo, TestIDs.PERIPHS_COMPONENT_PERIPHERAL_COMBO);
						GridDataComponents layoutData = new GridDataComponents(SWT.FILL, SWT.FILL, false, false, COMPONENT_COLS - 1, 1);
						combo.setLayoutData(layoutData);
						combo.setItems(masterPeripheral.getPeripherals().stream()
								.map(x -> controller.getPeripherals(x))
								.flatMap(x -> x.stream())
								.toArray(String[]::new));

						ControlDecorationUtils.createErrorDecoration(combo, SWT.TOP | SWT.LEFT, ToolsImages.getStatusDecoratorImg(ErrorLevels.LEVEL_ERROR));

						String selectedPeripheral = componentSetting.getPeripheral();
						if (selectedPeripheral != null) {
							combo.setSelection(selectedPeripheral);
						}
						combo.addSelectionListener(new SelectionAdapter() {
							/* (non-Javadoc)
							 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
							 */
							@Override
							public void widgetSelected(SelectionEvent e) {
								IWorkbenchPartSite partSite = getSite();
								assert partSite != null;
								if (!partSite.getPage().isPartVisible(ComponentSettingView.this)) {
									return;
								}
								String peripheralToSet = UtilsText.safeString(((InstantSearchList)e.widget).getText());
								String componentType = componentInput.getComponentType();
								String componentName = componentInput.getComponent();
								ProgressUtils.run((m) -> {
									controller.setPeripheral(componentType, componentName, peripheralToSet, ComponentSettingView.this);
								});
								recreate();
							}
						});
						Color background = combo.getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND);
						if (componentSetting.getMasterPeripheralError() != null) {
							background = SwToolsColors.ERROR_BG.getColor();
						}
						combo.setBackground(background);
					}
					// Link to component global config set
					if (controller.hasGlobalConfigSet(componentInput.getComponentType())) {
						createLinkToGlobalConfig(componentInput.getComponentType());
					}
					IChildControl componentSettingControlLoc = componentSettingControl;
					if ((componentSettingControlLoc == null) || !componentSettingControlLoc.getChild().equals(componentSetting)) {
						if (componentSettingControlLoc != null) {
							componentSettingControlLoc.dispose();
						}
						componentSettingControlLoc = ChildControlFactory.create(componentSetting, controllerWrapper);
						componentSettingControl = componentSettingControlLoc;
						if (componentSettingControlLoc != null) {
							componentSettingControlLoc.getControlOptions().labelHidden(true);
							componentSettingControlLoc.getControlOptions().borderHidden(true);
						}
					}
					if (componentSettingControlLoc != null) {
						componentSettingControlLoc.create(contentCompositeLoc, COMPONENT_COLS);
					}
					updateControls();
				}
			}
		}
		updateErrorIndicators();
		updateErrorDecorators();
		scrolledComposite.setMinWidth(GcUtils.getEmWidth(scrolledComposite, true) * MIN_WIDTH_EMS);
		scrolledComposite.setMinHeight(contentComposite.computeSize(scrolledComposite.getClientArea().width, SWT.DEFAULT).y);
		resizeListener = new Listener() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.widgets.Listener#handleEvent(org.eclipse.swt.widgets.Event)
			 */
			@Override
			public void handleEvent(Event event) {
				if (!getSiteSafe().getPage().isPartVisible(ComponentSettingView.this)) {
					return;
				}
				scrolledComposite.setMinHeight(contentComposite.computeSize(scrolledComposite.getClientArea().width, SWT.DEFAULT).y);
			}
		};
		contentComposite.addListener(SWT.Resize, resizeListener);
	}

	/**
	 * Returns error of this instance
	 * @param config of the component instance
	 * @return Error message
	 */
	private static @Nullable String getPeripheralError(@NonNull ComponentInstanceConfig config) {
		return config.getMasterPeripheralError();
	}

	/**
	 * Add link to component global config set
	 * @param componentType type of component that is being linked to
	 */
	private void createLinkToGlobalConfig(@NonNull String componentType) {
		// draw line
		Label label = new Label(contentComposite, SWT.SEPARATOR | SWT.HORIZONTAL);
		label.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, COMPONENT_COLS, 1));

		// add space after line - consistent with configSetControl
		label = new Label(contentComposite, SWT.NONE);
		label.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.TOP, false, false, COMPONENT_COLS, 1));

		// Title
		label = new Label(contentComposite, SWT.NONE);
		label.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.TOP, false, false, 1, 1));
		label.setText(Messages.get().ComponentEditor_ComponentGlobalSettings);
		FontFactory.changeStyle(label, SWT.BOLD);

		// link to global config set editor
		Link link = new Link(contentComposite, SWT.NONE);
		link.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false, COMPONENT_COLS - 1, 1));
		link.setText("<A>" + componentType + "</A>"); //$NON-NLS-1$ //$NON-NLS-2$
		SWTFactoryProxy.INSTANCE.setTestId(link, TestIDs.PERIPHS_COMPONENT_GLOBAL_CONFIG_SET_LINK);

		link.addListener(SWT.Selection, new Listener() {
			@Override
			public void handleEvent(Event event) {
				// open component global settings editor
				open(getSiteSafe().getPage(), componentType, componentType, true, true);
			}
		});
	}

	/**
	 * Creates button into parent composite that opens documentation view with documentation for given ConfigurationComponentTypeId.
	 * @param parent composite into which button should be created
	 * @param confCompTypeId ConfigurationComponentTypeId that contains documentation to show
	 */
	private static void createLinkToDocumentation(@NonNull Composite parent, @NonNull ConfigurationComponentTypeId confCompTypeId) {
		final Button btn = new Button(parent, SWT.NONE);
		final Image img = ToolsImages.getImage(IToolsImages.ICON_DOCUMENTATION);
		if (img == null) {
			assert false : "image not found: " + IToolsImages.ICON_DOCUMENTATION; //$NON-NLS-1$
		} else {
			btn.setImage(img);
		}
		SWTFactoryProxy.INSTANCE.setTestId(btn, TestIDs.PERIPHS_COMPONENT_SETTING_VIEW_DOCUMENTATION);
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(btn, UtilsText.safeString(Messages.get().ComponentSettingView_Documentation));
		btn.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false));
		btn.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				IViewPart viewPart = ComponentSettingViewHelper.getViewPart();
				if (viewPart != null) {
					IViewSite viewSite = viewPart.getViewSite();
					if (viewSite != null) {
						DocumentationView.open(viewSite, confCompTypeId.getTypeId(), true);
					}
				}
			}
		});
	}

	/**
	 * Creates button for enabling/disabling component instance.
	 * @param parent composite into which button should be created
	 * @param componentInstanceConfig ComponentInstanceConfig that state is being changed by button
	 */
	private void createEnableButton(@NonNull Composite parent, @NonNull ComponentInstanceConfig componentInstanceConfig) {
		Runnable action = new Runnable() {
			@Override
			public void run() {
				ProgressUtils.run((m) -> {
					controller.setComponentInstancesEnabled(CollectionsUtils.asList(componentInstanceConfig), !componentInstanceConfig.isEnabled(), this);
				});
				recreate();
			}
		};
		ToggleButton toggleButton = new ToggleButton(parent, componentInstanceConfig.isEnabled(), TestIDs.PERIPHS_ENABLE_INSTANCE_BUTTON + componentInput.getComponent(), action);
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(toggleButton, Messages.get().ComponentSettingView_EnableDisableComponentInstance);
		toggleButton.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false));
		toggleButton.setData(ToolView.IS_FORM_DISABLING_FORBIDDEN, Boolean.TRUE);
	}

	/**
	 * Assign action to widget, that is executed, if widget is selected
	 * @param widget either Button or ToolItem (button on toolbar)
	 * @param action to be executed if button pressed
	 */
	static void assignActionToWidget(@NonNull Widget widget, @NonNull Runnable action) {
		final SelectionAdapter actionAdapter = new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent event) {
				action.run();
			}
		};
		if (widget instanceof Button) {
			((Button) widget).addSelectionListener(actionAdapter);
		} else if (widget instanceof ToolItem) {
			((ToolItem) widget).addSelectionListener(actionAdapter);
		} else {
			assert false : "widget type not supported"; //$NON-NLS-1$
		}
	}

	/**
	 * Create controls for given config sets (or update if already exist).
	 * @param configSets for which to create GUI
	 * @param composite in which to create the GUI
	 */
	private void updateConfigSetsContent(@NonNull List<@NonNull ConfigSetConfig> configSets, @NonNull Composite composite) {
		// drop the unused controls
		configSetControls.keySet().retainAll(configSets);
		for (ConfigSetConfig configSet : configSets) {
			ConfigSetControl configSetControl = configSetControls.get(configSet);
			if (configSetControl == null) { // control for the config set does not exist, create one
				configSetControl = new ConfigSetControl(configSet, PeriphControllerWrapper.getInstance());
			}
			// create GUI
			configSetControl.create(composite, CONFIG_SET_COLS);
			configSetControl.update(UpdateType.NORMAL);
			configSetControls.put(configSet, configSetControl);
		}
	}

	/**
	 * Refresh content of the editor.
	 */
	void recreate() {
		if (contentComposite.isDisposed()) {
			return;
		}
		contentComposite.setRedraw(false);
		try {
			if (resizeListener != null) {
				contentComposite.removeListener(SWT.Resize, resizeListener);
			}
			if (componentSettingControl != null) {
				componentSettingControl.dispose();
			}
			for (ConfigSetControl configSetControl : configSetControls.values()) {
				configSetControl.dispose();
			}
			for (Control ctrl : contentComposite.getChildren()) {
				ctrl.dispose();
			}
			createContent();
		} finally {
			contentComposite.setRedraw(true);
		}
		contentComposite.layout();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#setFocus()
	 */
	@Override
	public void setFocus() {
		Composite composite = contentComposite;
		if ((composite != null) && !composite.isDisposed() && (state == State.CREATED_VISIBLE)) {
			Control focusedControl = composite.getDisplay().getFocusControl();
			if (!(OSDetect.isUnix() && (focusedControl != null) && ToolView.isControlAncestor(focusedControl, composite))) {
				// MCUXCON-2650 On Linux with GTK2 setFocus call, even when select value from combo inside view
				composite.setFocus();
			}
		}
	}

	/**
	 * Open component editor.
	 * @param viewSite with binding to an editor
	 * @param componentType type of the component to edit
	 * @param componentName name of component to edit (component type in case global is set to {@code true})
	 * @param global whether editor for global configSet or for component instance should be opened
	 * @param activate whether to activate the view after creating
	 * @return {@code true} if an editor was successfully opened, {@code false} otherwise
	 */
	public static boolean open(@NonNull IViewSite viewSite, @NonNull String componentType, @NonNull String componentName,
			boolean global, boolean activate) {
		IWorkbenchPage iWorkbenchPage = viewSite.getPage();
		if (iWorkbenchPage != null) {
			return open(iWorkbenchPage, componentType, componentName, global, activate);
		}
		return false;
	}

	/**
	 * Open component editor.
	 * @param workbenchPage with binding to an editor
	 * @param componentType type of the component to edit
	 * @param componentName name of component to edit (component type in case global is set to {@code true})
	 * @param global whether editor for global configSet or for component instance should be opened
	 * @param activate whether to activate the view after creating
	 * @return {@code true} if an editor was successfully opened, {@code false} otherwise
	 */
	public static boolean open(@Nullable IWorkbenchPage workbenchPage, @NonNull String componentType,
			@NonNull String componentName, boolean global, boolean activate) {
		if (workbenchPage != null) {
			// Reopen views that have their secondary ID based on old config name
			ComponentSettingViewHelper.getInstance().reopenInvalidViews();
			try {
				if (global) {
					componentName = UtilsText.EMPTY_STRING;
				}
				workbenchPage.showView(ComponentSettingView.ID, componentType + SECONDARY_ID_NAME_TYPE_SEPARATOR + componentName,
						activate ? IWorkbenchPage.VIEW_ACTIVATE : IWorkbenchPage.VIEW_VISIBLE);
				return true;
			} catch (@SuppressWarnings("unused") PartInitException e) {
				return false;
			}
		}
		return false;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.utils.view.ToolView#refreshStatus()
	 */
	@Override
	protected void refreshStatus() {
		refreshStatusBar();
	}

	/**
	 * Refresh settings, errors indicators, available modes.
	 * @deprecated use {@link #refreshSettings(UpdateType)} instead
	 */
	@Deprecated
	void refreshSettings() {
		refreshSettings(UpdateType.NORMAL);
	}

	/**
	 * Refresh settings, errors indicators, available modes.
	 * @param updateType type of the update
	 */
	void refreshSettings(@NonNull UpdateType updateType) {
		// update error indicator
		updateErrorIndicators();
		// update controls
		updateControls(updateType);
		// update error decorations of the non-config set related editors
		updateErrorDecorators();
		if (updateType != UpdateType.PROBLEM_DECORATION) {
			// update available modes
			ComponentInstanceConfig componentSetting = controller.getComponentSetting(componentInput.getComponentType(),
					componentInput.getComponent());
			InstantSearchList combo = comboBoxMode;
			if ((componentSetting != null) && (combo != null) && !combo.isDisposed()) {
				combo.setItems(componentSetting.getComponent().getScenarios().stream()
						.filter(y -> y.isAvailable(componentSetting.getExpressionContext()))
						.map(y -> y.getUIName(componentSetting.getExpressionContext()))
						.toArray(String[]::new));
				combo.select(componentSetting.getMode().getUIName(componentSetting.getExpressionContext()));
			}
			// update dimensions of scrollable pane
			scrolledComposite.setMinHeight(contentComposite.computeSize(scrolledComposite.getClientArea().width, SWT.DEFAULT).y);
		}
	}

	/**
	 * Update all controls in config sets.
	 * @deprecated use {@link #updateControls(UpdateType)} instead
	 */
	@Deprecated
	void updateControls() {
		updateControls(UpdateType.NORMAL);
	}

	/**
	 * Update all controls in config sets.
	 * @param updateType type of the update
	 */
	void updateControls(@NonNull UpdateType updateType) {
		contentComposite.setRedraw(false);
		try {
			if (componentSettingControl != null) {
				componentSettingControl.update(updateType);
			}
			configSetControls.values().forEach(x -> x.update(updateType));
			if (updateType != UpdateType.PROBLEM_DECORATION) {
				final IChild component = getComponent();
				final boolean enabled = (component != null) && component.isEnabled();
				@Nullable Control[] controls = new @Nullable Control[] {textBoxName, comboBoxMode, comboBoxPeripheral, labelName, labelMode, labelPeripheral};
				for (Control control : controls) {
					if ((control != null) && !control.isDisposed()) {
						control.setEnabled(enabled);
					}
				}
				updateTooltips();
			}
		} finally {
			contentComposite.setRedraw(true);
		}
	}

	/**
	 * Updates texts of tooltips and sets them on labels and editors
	 */
	void updateTooltips() {
		ToolTipableImplementation tooltipName = new ToolTipableImplementation();
		ToolTipableImplementation tooltipMode = new ToolTipableImplementation();
		ToolTipableImplementation tooltipPeripheral = new ToolTipableImplementation();
		// Update texts
		ComponentInstanceConfig componentSetting = controller.getComponentSetting(componentInput.getComponentType(), componentInput.getComponent());
		if (componentSetting == null) {
			return;
		}
		// Name
		Text textbox = textBoxName;
		if (textbox != null) {
			if (!textbox.isDisposed()) {
				String nameValue = UtilsText.safeString(textbox.getText());
				String error = getNameErrorMessageTooltip(componentSetting, nameValue);
				tooltipName.setError(error);
				tooltipName.setValueName(nameValue);
			}
			tooltipName.setUiName(Messages.get().ComponentEditor_Name);
			tooltipName.setDescription(Messages.get().ComponentEditor_NameDescription);
		}
		// Mode
		InstantSearchList combobox = comboBoxMode;
		if (combobox != null) {
			if (!combobox.isDisposed()) {
				tooltipMode.setValueName(combobox.getText());
			}
			tooltipMode.setValueDescription(componentSetting.getMode().getDescription());
			tooltipMode.setUiName(Messages.get().ComponentEditor_Mode);
			tooltipMode.setDescription(Messages.get().ComponentEditor_ModeDescription);
		}
		// Peripheral
		combobox = comboBoxPeripheral;
		if (combobox != null) {
			if (!combobox.isDisposed()) {
				tooltipPeripheral.setValueName(combobox.getText());
			}
			tooltipPeripheral.setUiName(Messages.get().ComponentEditor_Peripheral);
			tooltipPeripheral.setDescription(Messages.get().ComponentEditor_PeripheralDescription);
			tooltipPeripheral.setError(getPeripheralError(componentSetting));
		}
		// Update controls with tooltips
		@NonNull ToolTipableImplementation tooltips @NonNull [] = new @NonNull ToolTipableImplementation[]
				{tooltipName, tooltipMode, tooltipPeripheral};
		Label labels[] = {labelName, labelMode, labelPeripheral};
		Control editors[] = {textBoxName, comboBoxMode, comboBoxPeripheral};
		// Iterate over tooltips, labels and controls
		for (int index = 0; index < tooltips.length; index++) {
			@NonNull ToolTipableImplementation tooltip = tooltips[index];
			Label label = labels[index];
			Control editor = editors[index];
			String toolTipText = ToolTipableFormatter.getToolTipText(new ToolTipableMarkdownDescriptionDecorator(tooltip));
			Label nonNullLabel = label;
			if (nonNullLabel != null) {
				if (!nonNullLabel.isDisposed()) {
					SWTFactoryProxy.INSTANCE.setHtmlTooltip(nonNullLabel, toolTipText);
				}
			}
			Control nonNullEditor = editor;
			if (nonNullEditor != null) {
				if (!nonNullEditor.isDisposed()) {
					SWTFactoryProxy.INSTANCE.setHtmlTooltip(nonNullEditor, toolTipText);
				}
			}
		}
	}

	/**
	 * Returns error message when some errors are found and {@code null} when no errors are found
	 * @param instance to get functional group and profile from
	 * @param newName new name of instance
	 * @return {@code null} if there is no conflict and error message when there is a name conflict
	 */
	private @Nullable String getNameErrorMessageTooltip(@NonNull ComponentInstanceConfig instance, @NonNull String newName) {
		boolean nameIsIdentifier = Controller.isNameValid(newName);
		String error = null;
		if (!nameIsValid) {
			if (nameIsIdentifier) {
				error = ComponentInstanceConfig.getNameConflictError(instance, newName);
			} else {
				error = Messages.get().ComponentEditor_NameNotIdentifierError;
			}
		}
		return error;
	}

	/**
	 * Get component from the view's input.
	 * @return component from the view's input or {@code null} in case the specified component does not exist
	 */
	private @Nullable IChild getComponent() {
		ComponentSettingViewInput componentInputLoc = componentInput;
		if (componentInputLoc != null) {
			if (componentInputLoc.isGlobalConfig()) {
				return controller.getConfiguredComponent(componentInputLoc.getComponentType());
			}
			return controller.getComponentSetting(componentInputLoc.getComponentType(), componentInputLoc.getComponent());
		}
		return null;
	}

	/**
	 * Enumeration containing possible states of the component setting view.
	 * @author Juraj Ondruska
	 */
	private enum State {
		/** the view is unitialized (its content has not been created yet) */
		UNINITIALIZED,
		/** the view is unitialized but is ready to be created when it becomes visible */
		READY_TO_CREATE,
		/** the view is initialized but hidden */
		CREATED_HIDDEN,
		/** the view is initialized and visible */
		CREATED_VISIBLE
	}

	/**
	 * Enumeration containging required actions of the component setting view.
	 * @author Juraj Ondruska
	 */
	private enum ActionRequired {
		/** No action required when the view becomes visible */
		NO_ACTION,
		/** Refresh of the view is required when the view becomes visible */
		REFRESH,
		/** Recreate of the view is required when the view becomes visible */
		RECREATE
	}

	/**
	 * @return non null IWorkbenchPartSite - calls {@link #getSite()}e
	 */
	public @NonNull IWorkbenchPartSite getSiteSafe() {
		IWorkbenchPartSite site = getSite();
		assert site != null;
		return site;
	}
}
