/**
 * Copyright 2018-2019 NXP
 * Created: Jul 18, 2018
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.model.config.ComponentInstanceConfig;
import com.nxp.swtools.periphs.model.config.ConfigSetConfig;
import com.nxp.swtools.periphs.model.config.IChild;

/**
 * Class representing the GUI elements of component instance configuration.
 * @author Viktar Paklonski
 */
public class ComponentInstanceControl extends ChildProvidableControlBase {

	/** controls of all displayed config sets */
	private final @NonNull Map<@NonNull ConfigSetConfig, ConfigSetControl> configSetControls = new HashMap<>();
	/** content of the editor */
	private Composite contentComposite;

	/**
	 * @param instanceConfig to create control for
	 * @param controllerWrapper containing the generic controller
	 */
	public ComponentInstanceControl(@NonNull ComponentInstanceConfig instanceConfig, @NonNull IControllerWrapper controllerWrapper) {
		super(instanceConfig, controllerWrapper);
		onCreateConfigSetsContent();
	}

	/**
	 * @return either {@link ChildControlBase#mainControl} or {@link #contentComposite}
	 */
	private @NonNull Composite getMainComposite() {
		final Control mainControlLoc = mainControl;
		if (mainControlLoc != null) {
			return (Composite) mainControlLoc;
		}
		final Composite contentCompositeLoc = contentComposite;
		assert contentCompositeLoc != null;
		return contentCompositeLoc;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#create(org.eclipse.swt.widgets.Composite, int)
	 */
	@Override
	public void create(@NonNull Composite composite, int colSpan) {
		contentComposite = composite;
		if (!Objects.equals(ControlOptions.ShowContentAs.TABS, getControlOptions().getShowContentAs())) {
			// draw line
			final Label label = new Label(composite, SWT.SEPARATOR | SWT.HORIZONTAL);
			label.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, ComponentSettingView.CONFIG_SET_COLS, 1));
		}
		super.create(composite, colSpan);
		final Control labelControlLoc = labelControl;
		if (labelControlLoc != null) {
			labelControlLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, colSpan, 1));
		}
		Control mainControlLoc = mainControl;
		if (mainControlLoc != null) {
			mainControlLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.TOP, true, false, colSpan, 1));
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#update(com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	public void update(@NonNull UpdateType updateType) {
		if (updateType != UpdateType.PROBLEM_DECORATION) {
			onUpdateConfigSetsContent();
		}
		super.update(updateType);
	}

	/**
	 * Update controls map for the given config sets.
	 */
	void onUpdateConfigSetsContent() {
		final List<@NonNull ConfigSetConfig> newChildren = ((ComponentInstanceConfig) child).getChildren();
		// dispose of old controls
		for (Map.Entry<@NonNull ConfigSetConfig, ConfigSetControl> entry : configSetControls.entrySet()) {
			if (!newChildren.contains(entry.getKey())) {
				final ConfigSetControl value = Objects.requireNonNull(entry.getValue());
				if (!value.isDisposed()) {
					value.dispose();
				}
			}
		}
		final List<@NonNull IChildControl> newControls = new ArrayList<>();
		// re-create GUI for new and re-used controls
		for (ConfigSetConfig key : newChildren) {
			IChildControl control = configSetControls.get(key);
			if (control == null) {
				final ControlOptions overrideOptions = createControlOptionsForChild(key);
				control = ChildControlFactory.create(key, overrideOptions, controllerWrapper);
				assert control != null;
				control.create(getMainComposite(), ComponentSettingView.CONFIG_SET_COLS);
				configSetControls.put(key, (ConfigSetControl) control);
			} else if (control.isDisposed()) {
				control.create(getMainComposite(), ComponentSettingView.CONFIG_SET_COLS);
			}
			newControls.add(control);
		}
		children.clear();
		children.addAll(newControls);
	}
	
	/**
	 * Create controls map for the given config sets.
	 */
	private void onCreateConfigSetsContent() {
		// associate config sets with controls
		for (IChildControl control : children) {
			final IChild controlChild = control.getChild();
			if (controlChild instanceof ConfigSetConfig) {
				configSetControls.put((ConfigSetConfig) controlChild, (ConfigSetControl) control);
			}
		}
	}
	
}
