/**
 * Copyright 2017-2019 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;

import com.nxp.swtools.common.ui.utils.swt.ITooltipProvider2;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.ControlOptions.ShowContentAs;
import com.nxp.swtools.periphs.model.config.IChild;
import com.nxp.swtools.periphs.model.config.IChildProvidable;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.utils.resources.ToolsImages;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;

/**
 * Base class representing control of a structured setting configuration.
 * @author Juraj Ondruska
 */
public class ChildProvidableControlBase extends ChildControlBase {
	/** The children */
	protected final @NonNull List<@NonNull IChildControl> children = new ArrayList<>();		
	/** Key for setting data (IChildControl) into CTabItems */
	public static final @NonNull String KEY_TAB_CHILD_CONTROL = ChildProvidableControlBase.class.getCanonicalName() + ".key_tab_child_control"; //$NON-NLS-1$
	/** The currently selected child */
	protected @Nullable IChildControl selectedChild;
	/** Index of the currently selected child */
	protected int selectedChildIdx = -1;
	/** Tab folder with the children */
	protected @Nullable CTabFolder tabFolder;
	/** String used as flag to check whether the tab has been initialized */
	private static final @NonNull String READY_STRING = "ready"; //$NON-NLS-1$
	
	/**
	 * @param childProvidable to create control for
	 * @param controllerWrapper containing the generic controller
	 */
	public ChildProvidableControlBase(@NonNull IChildProvidable childProvidable, @NonNull IControllerWrapper controllerWrapper) {
		super(childProvidable, controllerWrapper);
		
		for (IChild subChild : childProvidable.getChildren()) {
			ControlOptions overrideOptions = createControlOptionsForChild(subChild);
			
			IChildControl subChildControl = ChildControlFactory.create(subChild, overrideOptions, controllerWrapper);
			if (subChildControl != null) {
				children.add(subChildControl);
			}
		}
	}

	/**
	 * Create Control Options for child based on options of its parent (calling object)
	 * @param subChild child for which control options are created
	 * @return ControlOptions for a child
	 */
	protected @NonNull ControlOptions createControlOptionsForChild(@NonNull IChild subChild) {
		ControlOptions overrideOptions = new ControlOptions();
		// if indices should be hidden, hide label of child
		if (getControlOptions().isArrayIndicesHidden()) {
			overrideOptions.labelHidden(true);
		}
		if (ShowContentAs.TABS.equals(getControlOptions().getShowContentAs())) {
			// if child should be shown as Tab and has own children
			if (subChild instanceof IChildProvidable) {
				overrideOptions.borderHidden(true); // hide its border (no reason to have border around whole tab content)
				overrideOptions.labelHidden(true); // hide its label (label is put into tab name)
			}
		} else if (ShowContentAs.TABLE.equals(getControlOptions().getShowContentAs())) {
			// hide border in table
			overrideOptions = overrideOptions.borderHidden(true);
		} else if (ShowContentAs.MASTER_DETAIL.equals(getControlOptions().getShowContentAs())) {
			// hide indices/labels in detail 
			overrideOptions = overrideOptions.labelHidden(true);
		}
		return overrideOptions;
	}
	
	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.ChildControlBase#update(com.nxp.swtools.periphs.gui.editor.IChildControl.UpdateType)
	 */
	@Override
	public void update(@NonNull UpdateType updateType) {		
		UpdateType updateChildrenType = child.isEnabled() ? updateType : UpdateType.FORCE_DISABLE;
		if ((tabFolder != null) && (updateType != UpdateType.PROBLEM_DECORATION)) {
			 updateTabs(children);
		}
		super.update(updateType);
		for (IChildControl childControl : children) {
			childControl.update(updateChildrenType);
		}
		// remove html tooltip from mainControl (tooltip was visible on empty places in composite) - MCUXCON-1778
		Control mainControlLoc = mainControl;
		if ((mainControlLoc != null) && (!mainControlLoc.isDisposed())) {
			SWTFactoryProxy.INSTANCE.setHtmlTooltip(mainControlLoc, null);
		}
	}

	/**
	 * Update CTabFolder to match information from childrenControls. Creates CTabItems on correct indexes if they were not displayed
	 * and should be displayed. If CTabItems were displayed and should be hidden now, they are disposed.
	 * @param childrenControls List of IChildControls based on which CTabFolder is being updated
	 */
	public void updateTabs(@NonNull List<@NonNull IChildControl> childrenControls) {
		final CTabFolder tabFolderLoc = tabFolder;
		if (tabFolderLoc != null) {
			int tabIndex = 0;
			IChildControl childControl;
			IChild childLoc;
			boolean shouldBeVisible;
			for (int i = 0; i < childrenControls.size(); i++) {
				childControl = childrenControls.get(i);
				childLoc = childControl.getChild();
				shouldBeVisible = childLoc.isAvailable();
				if (shouldBeVisible) {
					if (tabIndex >= tabFolderLoc.getItemCount()) {
						createTabItem(childControl, ComponentSettingView.CONFIG_SET_COLS, tabIndex);
					} else {
						CTabItem tabItem = tabFolderLoc.getItem(tabIndex);
						if (!tabItem.isDisposed() && (tabItem.getData(KEY_TAB_CHILD_CONTROL) != childControl)) {
							createTabItem(childControl, ComponentSettingView.CONFIG_SET_COLS, tabIndex);
						}
					}
					tabIndex++;	
				} else {
					if (tabIndex < tabFolderLoc.getItemCount()) {
						CTabItem tabItem = tabFolderLoc.getItem(tabIndex);
						if (tabItem.getData(KEY_TAB_CHILD_CONTROL) == childControl) {
							tabItem.dispose();
						}
					}
				}
			}
			for (int i = 0; i < tabFolderLoc.getItemCount(); i++) {
				CTabItem tabItem = tabFolderLoc.getItem(i);
				if (!tabItem.isDisposed() && !childrenControls.contains(tabItem.getData(KEY_TAB_CHILD_CONTROL))) {
					tabItem.dispose();
				}
			}
			CTabItem tabToSelect = getTabToSelect();
			if (tabToSelect == null) {
				if (tabFolderLoc.getItemCount() > 0) {
					tabToSelect = tabFolderLoc.getItem(0);
				}
			}
			if (tabToSelect != null && !tabToSelect.isDisposed()) {
				tabFolderLoc.setSelection(tabToSelect);
				updateChildSelection(ComponentSettingView.CONFIG_SET_COLS);
			}
			if (!tabFolderLoc.isDisposed()) {
				updateErrorDecoration(tabFolderLoc);
				tabFolderLoc.requestLayout();
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.ChildControlBase#updateErrorDecoration
	 */
	@Override
	protected void updateErrorDecoration(@NonNull Control controlWithDecoration) {
		super.updateErrorDecoration(controlWithDecoration);
		if (controlWithDecoration instanceof CTabFolder) {
			for (CTabItem tabItem : ((CTabFolder) controlWithDecoration).getItems()) {
				Object data = tabItem.getData(KEY_TAB_CHILD_CONTROL);
				if (data instanceof IChildControl) {
					IChildControl childControl = (IChildControl) data;
					IChild tabChild = childControl.getChild();
					int highestSeverityProblemLevel = tabChild.getHighestSeverityProblemLevel();
					if ((highestSeverityProblemLevel >= ErrorLevels.MIN_LEVEL) 
							&& (highestSeverityProblemLevel <= ErrorLevels.MAX_LEVEL)) {
						tabItem.setImage(ToolsImages.getStatusDecoratorImg(highestSeverityProblemLevel));
					} else {
						tabItem.setImage(null);
					}
				}
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.AChildControl#dispose()
	 */
	@Override
	public void dispose() {
		super.dispose();
		for (IChildControl childControl : children) {
			childControl.dispose();
		}
		tabFolder = null;
	}
	
	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.AChildControl#getChild()
	 */
	@Override
	public @NonNull IChildProvidable getChild() {
		return ((IChildProvidable) child);
	}

	/**
	 * @return unmodifiable {@link #children} 
	 */
	public @NonNull List<@NonNull IChildControl> getChildren() {
		return CollectionsUtils.safeList(Collections.unmodifiableList(children));
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#createMainControl
	 */
	@Override
	public @Nullable Control createMainControl(@NonNull Composite composite) {
		Composite contentComposite;
		if (ShowContentAs.TABS.equals(getControlOptions().getShowContentAs())) {
			contentComposite = createControlTabs(composite, ComponentSettingView.CONFIG_SET_COLS);
		} else {
			contentComposite = createControlStandard(composite, ComponentSettingView.CONFIG_SET_COLS);			
		}
		return (contentComposite == composite) ? null : contentComposite;
	}

	/**
	 * Helper method to create correct composite according to options set.
	 * @param parentComposite parent composite in which new composite should be created
	 * @param numColumns number of columns set to newly created grid layout for new composite
	 * @return created composite
	 */
	public @NonNull Composite createComposite(@NonNull Composite parentComposite, int numColumns) {
		return createComposite(parentComposite, numColumns, this);	
	}
	
	/**
	 * Helper method to create correct composite according to options set.
	 * @param parentComposite parent composite in which new composite should be created
	 * @param numColumns number of columns set to newly created grid layout for new composite
	 * @param childControl child for which composite should be created
	 * @return created composite
	 */
	public @NonNull Composite createComposite(@NonNull Composite parentComposite, int numColumns, @NonNull IChildControl childControl) {
		GridLayoutComponents layout = new GridLayoutComponents(numColumns, false);
		int style = SWT.NONE;
		// new composite has to be created for settings that should be shown as tabs
		if ((childControl.getControlOptions().isBorderHidden()) && (!ShowContentAs.TABS.equals(getControlOptions().getShowContentAs()))) {
			return parentComposite;
		} else {
			layout.marginWidth = 8;		
		}
		style = getSwtStyle(childControl);
		layout.horizontalSpacing = 8;
		Composite composite = new Composite(parentComposite, style);	
		composite.setLayout(layout);
		return composite;	
	}
	
	/**
	 * Create standard control. Composite and children in it
	 * @param parentComposite parent composite in which new composite should be created
	 * @param numColumns number of columns set to newly created grid layout for new composite
	 * @return created composite
	 */
	public @NonNull Composite createControlStandard(@NonNull Composite parentComposite, int numColumns) {
		Composite contentComposite = createComposite(parentComposite, numColumns);
		fillTopOfMainControl(contentComposite, numColumns);
		for (IChildControl childControl : children) {
			childControl.create(contentComposite, numColumns);
		}
		return contentComposite;
	}
	
	/**
	 * Fill top part of the main control.
	 * @param composite to fill
	 * @param numColumns number of columns of the composite
	 */
	protected void fillTopOfMainControl(@SuppressWarnings("unused") @NonNull Composite composite, @SuppressWarnings("unused") int numColumns) {
		// nothing to do
	}
	
	/**
	 * Create control as a CTabFolder. Children are created as individual CTabItems with their own composite inside the CTabFolder
	 * @param parentComposite parent composite in which new composite should be created
	 * @param numColumns number of columns set to newly created grid layout for new composite
	 * @return created composite
	 */
	public @NonNull Composite createControlTabs(@NonNull Composite parentComposite, int numColumns) {
		fillTopOfMainControl(parentComposite, numColumns);
		CTabFolder tabFolderLoc = new CTabFolder(parentComposite, SWT.BORDER);
		tabFolder = tabFolderLoc;
		createHtmlTooltipFolder(tabFolderLoc);
		tabFolderLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.FILL, true, true));
		for (IChildControl childControl : children) {
			createTabItem(childControl, numColumns);
		}
		CTabItem tabToSelect = getTabToSelect();
		if (tabToSelect != null) {
			tabFolderLoc.setSelection(tabToSelect);
			updateChildSelection(numColumns);
		}
		tabFolderLoc.addSelectionListener(new SelectionAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			@Override
			public void widgetSelected(SelectionEvent e) {
				updateChildSelection(numColumns);
			}
		});
		return tabFolderLoc;
	}
	
	/**
	 * Set selected child based on the tab selection.
	 * @param colSpan number of columns set to newly created grid layout for new composite
	 */
	void updateChildSelection(int colSpan) {
		if (tabFolder != null) {
			final CTabItem selection = tabFolder.getSelection();
			if (selection != null) {
				Object itemData = selection.getData(KEY_TAB_CHILD_CONTROL);
				if (itemData instanceof IChildControl) {
					setSelectedChild((IChildControl) itemData);
					if (selection.getData(READY_STRING) == null) {
						Control selectionControl = selection.getControl();
						if ((selectionControl != null) && (selectionControl instanceof Composite)) {
							((IChildControl) itemData).create((Composite) selectionControl, colSpan);
							((IChildControl) itemData).update(UpdateType.NORMAL);
							selection.setData(READY_STRING, "true"); //$NON-NLS-1$
						}
					}
	
					// these two lines must be placed here in this order. otherwise, if you switch from a tab with less controls to a tab with more, the
					// additional controls will not be displayed.
					Objects.requireNonNull(((Composite) selection.getControl()).getParent()).requestLayout();
					((Composite) selection.getControl()).layout();
				}
			}
		}
	}

	/**
	 * Get the tab which should be selected 
	 * @return the tab to be selected or {@code null} if no tab should be selected
	 */
	private @Nullable CTabItem getTabToSelect() {
		IChildControl childToSelect = getChildToSelect();
		if (tabFolder != null) {
			for (CTabItem item : tabFolder.getItems()) {
				if (!item.isDisposed() && item.getData(KEY_TAB_CHILD_CONTROL) == childToSelect) {
					return item;
				}
			}
		}
		return null;
	}
	
	/**
	 * @return the currently selected child or {@code null} if no child is selected
	 */
	public @Nullable IChildControl getSelectedChild() {
		return selectedChild;
	}
	
	/**
	 * Get child which should be selected. The method tries to check whether an item is available and returns it. The preferences are: 1. The lastly selected
	 * item; 2. Item with the same name as the lastly selected item; 3. Item with the same index as the lastly selected item; If none of the above is available
	 * then this method returns the first available item behind the lastly selected item; If no item behind the lastly selected item is available then the
	 * nearest item before the lastly selected item is returned; If none of the above is available then the first available item is returned; Otherwise
	 * {@code null} is returned
	 * @return the child to be selected or {@code null} if no child is available
	 */
	public @Nullable IChildControl getChildToSelect() {
		// try to locate the lastly selected child in the current list of children
		int newSelectionIdx = 0;
		IChildControl selectedChildLoc = selectedChild;
		int lastSelectionIdx = children.indexOf(selectedChildLoc);
		if ((lastSelectionIdx < 0) && (selectedChildLoc != null)) {
			// the lastly selected item is not in the current list, try to find item with the same name
			String selectedChildName = selectedChildLoc.getChild().getName();
			IChildControl childWithSameName = CollectionsUtils.nullableOptionalGet(children.stream()
					.filter(ch -> ch.getChild().getName().equals(selectedChildName))
					.findFirst());
			lastSelectionIdx = children.indexOf(childWithSameName);
			if (lastSelectionIdx < 0) {
				// child with the same name was not found, use the last selection index
				lastSelectionIdx = selectedChildIdx;
			}
		}
		// update index so it is in range [0 .. children.size]
		if (lastSelectionIdx >= children.size()) {
			lastSelectionIdx = children.size() - 1;
		}
		if (lastSelectionIdx < 0) {
			lastSelectionIdx = 0;
		}
		// select appropriate child from the current children list based on the previous selection (preferences: 1. the lastly selected item,
		// 2. first available item behind the lastly selected item, 3. first available item before the lastly selected item)
		if ((lastSelectionIdx >= 0) && (lastSelectionIdx < children.size())) {
			newSelectionIdx = lastSelectionIdx;
			if (!children.get(lastSelectionIdx).getChild().isAvailable()) {
				while ((newSelectionIdx < children.size()) && !children.get(newSelectionIdx).getChild().isAvailable()) {
					newSelectionIdx++;
				}
				if (newSelectionIdx >= children.size()) {
					newSelectionIdx = lastSelectionIdx - 1;
					while ((newSelectionIdx >= 0) && !children.get(newSelectionIdx).getChild().isAvailable()) {
						newSelectionIdx--;
					}
				}
			}
		}
		return ((newSelectionIdx >= 0) && (newSelectionIdx < children.size())) ? children.get(newSelectionIdx) : null;
	}

	/**
	 * Create new CTabItem.
	 * @param childControl childControl for which tab is created
	 * @param numColumns number of columns set to newly created grid layout for new composite
	 */
	public void createTabItem(@NonNull IChildControl childControl, int numColumns) {
		createTabItem(childControl, numColumns, -1);
	}
	
	/**
	 * Create new CTabItem.
	 * @param childControl childControl for which tab is created
	 * @param numColumns number of columns set to newly created grid layout for new composite
	 * @param index the position on which should be CTabItem created in a parent. If index is out of bounds of the parent, new CTabItem is created at the end.
	 */
	public void createTabItem(@NonNull IChildControl childControl, int numColumns, int index) {
		CTabFolder tabFolderLoc = tabFolder;
		if (tabFolderLoc != null) {
			if ((index < 0) || (index > tabFolderLoc.getItemCount())) {
				index = tabFolderLoc.getItemCount();
			}
			CTabItem tabItem = new CTabItem(tabFolderLoc, SWT.NONE, index);
			Composite contentComposite = createComposite(tabFolderLoc, numColumns, childControl);
			tabItem.setControl(contentComposite);
			tabItem.setText(childControl.getChild().getUiName());
			tabItem.setData(KEY_TAB_CHILD_CONTROL, childControl);	
			tabItem.setToolTipText(UtilsText.EMPTY_STRING);
			contentComposite.requestLayout();
		}
	}
	
	/**
	 * Set the currently selected child.
	 * @param selectedChild to set
	 */
	public void setSelectedChild(@Nullable IChildControl selectedChild) {
		this.selectedChild = selectedChild;
		selectedChildIdx = children.indexOf(selectedChild);
	}

	/**
	 * Enables html tooltip for CTabFolder
	 * @param folder the folder instance
	 */
	private static void createHtmlTooltipFolder(@NonNull final CTabFolder folder) {
		SWTFactoryProxy.INSTANCE.createHtmlTooltip(folder, new ITooltipProvider2() {
			/* (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.swt.ITooltipProvider2#getToolTipText(org.eclipse.swt.widgets.Event)
			 */
			@Override
			public String getToolTipText(Event event) {
				CTabItem item = folder.getItem(new Point(event.x, event.y));
				if (item != null) {
					Object property = item.getData(KEY_TAB_CHILD_CONTROL);
					if (property instanceof IChildControl) {
						return ToolTipableFormatter.getToolTipText(new ToolTipableMarkdownDescriptionDecorator(((IChildControl) property).getChild()));
					}
				}
				return null;
			}
			/* (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.swt.ITooltipProvider#getToolTipText()
			 */
			@Override
			public String getToolTipText() {
				return null;
			}
		});
	}

}
