/**
 * Copyright 2017-2019 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.text.MessageFormat;
import java.util.logging.Logger;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.model.config.ArrayConfig;
import com.nxp.swtools.periphs.model.config.ComponentInstanceConfig;
import com.nxp.swtools.periphs.model.config.ConfigSetConfig;
import com.nxp.swtools.periphs.model.config.IChild;
import com.nxp.swtools.periphs.model.config.ScalarConfig;
import com.nxp.swtools.periphs.model.config.SetConfig;
import com.nxp.swtools.periphs.model.config.SetConfig.SetPresence;
import com.nxp.swtools.periphs.model.config.StructConfig;
import com.nxp.swtools.periphs.model.data.SettingOptions;

/**
 * Class used for creating setting configuration controls.
 * @author Juraj Ondruska
 */
public class ChildControlFactory {
	/** Logger of the class */
	private static final @NonNull Logger LOGGER = LogManager.getLogger(ChildControlFactory.class);
	
	/**
	 * Create control.
	 * @param child the setting configuration to create control for
	 * @param controllerWrapper containing the generic controller
	 * @return the newly created setting configuration control if visible in the UI, {@code null} otherwise
	 */
	public static @Nullable IChildControl create(@NonNull IChild child, @NonNull IControllerWrapper controllerWrapper) {
		if (child instanceof ScalarConfig) {
			switch (((ScalarConfig) child).getType()) {
			case ENUM:
				return new ScalarComboControl((ScalarConfig) child, controllerWrapper);
			case BOOL:
				return new ScalarCheckboxControl((ScalarConfig) child, controllerWrapper);
			case FLOAT:
			case INTEGER:
			case STRING:
				return new ScalarTextControl((ScalarConfig) child, controllerWrapper);
			case VARIABLE:
				return null;
			case INFO:
				if (child.isOptionSet(SettingOptions.UI_INFO_COPY_BUTTON)) {
					return new ScalarCopyControl((ScalarConfig) child, controllerWrapper);
				}
				//$FALL-THROUGH$
			default:
				return new ScalarControl((ScalarConfig) child, controllerWrapper);
			}
		}
		if (child instanceof StructConfig) {
			return new StructControl((StructConfig) child, controllerWrapper);
		}
		if (child instanceof ArrayConfig) {
			return new ArrayControl((ArrayConfig) child, controllerWrapper);
		}
		if (child instanceof SetConfig) {
			return new SetControl((SetConfig) child, controllerWrapper);
		}
		if (child instanceof ConfigSetConfig) {
			return new ConfigSetControl((ConfigSetConfig) child, controllerWrapper);
		}
		if (child instanceof SetPresence) {
			return new SetPresenceControl((SetPresence) child, controllerWrapper);
		}
		if (child instanceof ComponentInstanceConfig) {
			return new ComponentInstanceControl((ComponentInstanceConfig) child, controllerWrapper);
		}
		LOGGER.severe(MessageFormat.format("Unexpected class {0} of setting with ID {1}", child.getClass().getSimpleName(), child.getId())); //$NON-NLS-1$
		return new ChildControlBase(child, controllerWrapper);
	}
	
	/**
	 * Create control.
	 * @param child the setting configuration to create control for
	 * @param overwriteOptions UI options that are merged into options of this child created from model
	 * @param controllerWrapper containing the generic controller
	 * @return the newly created setting configuration control if visible in the UI, {@code null} otherwise
	 */
	public static @Nullable IChildControl create(@NonNull IChild child, @Nullable ControlOptions overwriteOptions, @NonNull IControllerWrapper controllerWrapper) {
		IChildControl childControl = create(child, controllerWrapper);
		if ((childControl != null) && (overwriteOptions != null)) {
			childControl.getControlOptions().merge(overwriteOptions);
		}
		return childControl;
	}
}
