/**
 * Copyright 2017-2019 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.util.List;
import java.util.Objects;
import java.util.logging.Logger;

import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.fieldassist.ControlDecoration;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;

import com.nxp.swtools.common.ui.utils.swt.ControlDecorationUtils;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.ui.utils.swt.ScrolledCompositeHelper;
import com.nxp.swtools.common.ui.utils.swt.widgets.LabelAmpersand;
import com.nxp.swtools.common.ui.utils.swt.widgets.ReadOnlyText;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.periphs.controller.ValidationHelper;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.utils.GcUtils;
import com.nxp.swtools.periphs.model.config.ArrayConfig;
import com.nxp.swtools.periphs.model.config.ChildValidationHelper;
import com.nxp.swtools.periphs.model.config.IChild;
import com.nxp.swtools.periphs.model.config.ISettingConfig;
import com.nxp.swtools.provider.analytics.ActionAnalyticsBuilder;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsImages;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;

/**
 * Base class representing control of a setting configuration.
 * @author Juraj Ondruska
 */
public class ChildControlBase implements IChildControl {
	/** Logger of the class */
	protected static final Logger LOGGER = LogManager.getLogger(ChildControlBase.class);
	/** Key for the error decoration of a control (set as data of the control). Use with {@link Control#getData(String)} */
	@Deprecated
	public static final @NonNull String KEY_ERROR_DECORATION = ControlDecorationUtils.KEY_ERROR_DECORATION;
	/** The controller */
	public final @NonNull IControllerWrapper controllerWrapper;
	/** The setting configuration */
	protected final @NonNull IChild child;
	/** The composite in which the control is created */
	protected @Nullable Composite parent;
	/** Control of the label */
	protected @Nullable Control labelControl;
	/** The main mainControl */
	protected @Nullable Control mainControl;
	/** UI options */ 
	protected @NonNull ControlOptions controlOptions;
	/** signals whether {@link #dispose()} has been called */
	private boolean disposed;
	/** ColSpan given in create method */
	protected int currentColSpan = 0;
	
	/**
	 * @param child the setting configuration to create content for
	 * @param controllerWrapper containing the generic controller
	 */
	public ChildControlBase(@NonNull IChild child, @NonNull IControllerWrapper controllerWrapper) {
		this.child = child;
		this.controllerWrapper = controllerWrapper;
		this.controlOptions = new ControlOptions(child);
		this.disposed = true;
	}
	
	/**
	 * @param child the setting configuration to create content for
	 * @param controlOptions UI options that overwrite options of this child
	 * @param controllerWrapper containing the generic controller
	 */
	public ChildControlBase(@NonNull IChild child, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		this(child, controllerWrapper);
		getControlOptions().merge(controlOptions);
	}
	
	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl#create(org.eclipse.swt.widgets.Composite, int)
	 */
	@Override
	public void create(@NonNull Composite composite, int colSpan) {
		parent = composite;
		final int maxEms = 25;
		currentColSpan = colSpan;
		Control labelControlLoc = getControlOptions().isLabelHidden() ? null : createLabelControl(composite);
		labelControl = labelControlLoc;
		Control mainControlLoc = createMainControl(composite);
		if ((getControlOptions().isLabelHidden()) && (mainControlLoc != null) && (!getControlOptions().isBorderHidden())) {
			createErrorDecoration(mainControlLoc, SWT.TOP | SWT.LEFT);
		}
		mainControl = mainControlLoc;
		int labelSpan = (mainControlLoc == null) ? colSpan : 1;
		int controlColSpan = (labelControlLoc == null) ? colSpan : (colSpan - 1);
		if (labelControlLoc != null) {
			if (labelControlLoc.getLayoutData() == null) {
				GridDataComponents labelControlLayoutData = new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false, labelSpan, 1);
				labelControlLayoutData.maxWidth = GcUtils.getEmWidth(labelControlLoc, true) * maxEms;
				labelControlLoc.setLayoutData(labelControlLayoutData);
			}
		}
		if (mainControlLoc != null) {
			if (mainControlLoc.getLayoutData() == null) {
				GridDataComponents mainControlLayoutData = new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, controlColSpan, 1);
				mainControlLoc.setLayoutData(mainControlLayoutData);
			}
		}
		disposed = false;
	}
	
	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.IChildControl#update(com.nxp.swtools.periphs.gui.editor.IChildControl.UpdateType)
	 */
	@Override
	public void update(@NonNull UpdateType updateType) {
		Control labelControlLoc = labelControl;
		Control mainControlLoc = mainControl;
		if (((labelControlLoc == null) || labelControlLoc.isDisposed()) && ((mainControlLoc == null) || mainControlLoc.isDisposed())) {
			return; // nothing to update (no UI controls)
		}
		boolean visible = child.isAvailable();
		boolean enabled = (updateType != UpdateType.FORCE_DISABLE) && child.isEnabled();
		UpdateType updateChildrenType = child.isEnabled() ? updateType : UpdateType.FORCE_DISABLE;
		boolean layoutChange = false;
		String toolTipText = visible ? ToolTipableFormatter.getToolTipText(new ToolTipableMarkdownDescriptionDecorator(child)) : UtilsText.EMPTY_STRING;
		if ((labelControlLoc != null) && !labelControlLoc.isDisposed()) {
			SWTFactoryProxy.INSTANCE.setHtmlTooltip(labelControlLoc, toolTipText);
			layoutChange = labelControlLoc.isVisible() != visible;
			labelControlLoc.setEnabled(enabled);
			labelControlLoc.setVisible(visible);
			((GridDataComponents) Objects.requireNonNull(labelControlLoc.getLayoutData())).exclude = !visible;
			SWTFactoryProxy.INSTANCE.setTestId(labelControlLoc, TestIDs.PERIPHS_SETTING_LABEL + child.getId());
			if (visible) {
				updateLabelContent(labelControlLoc, updateChildrenType);
			}
		}
		if ((mainControlLoc != null) && !mainControlLoc.isDisposed()) {
			SWTFactoryProxy.INSTANCE.setHtmlTooltip(mainControlLoc, toolTipText);
			layoutChange = layoutChange || (mainControlLoc.isVisible() != visible);
			mainControlLoc.setEnabled(enabled);
			mainControlLoc.setVisible(visible);
			((GridDataComponents) Objects.requireNonNull(mainControlLoc.getLayoutData())).exclude = !visible;
			SWTFactoryProxy.INSTANCE.setTestId(mainControlLoc, TestIDs.PERIPHS_SETTING_CONTROL + child.getId());
			if (visible) {
				updateMainContent(mainControlLoc, updateChildrenType);
			}
		}
		if (layoutChange) {
			if (parent != null) {
				parent.requestLayout();
			}
		}
	}

	/**
	 * Update error decoration attached to the control by the previous call of {@link #createErrorDecoration(Control, int)}
	 * @param controlWithDecoration with the decoration to update
	 */
	protected void updateErrorDecoration(@NonNull Control controlWithDecoration) {
		final int problemLevelOfChild = ValidationHelper.getHighestSeveritySettingValidationProblemLevel(child);
		final StringBuilder result = new StringBuilder();
		String status = null;
		int statusLevel = ErrorLevels.MIN_LEVEL - 1;
		if (((status = child.getError()) != null) || (problemLevelOfChild >= ErrorLevels.LEVEL_ERROR)) {
			statusLevel = ErrorLevels.LEVEL_ERROR;
			result.append(com.nxp.swtools.utils.Messages.get().ToolTipableFormater_Errors);
		} else if (((status = child.getWarning()) != null) || (problemLevelOfChild == ErrorLevels.LEVEL_WARNING)) {
			statusLevel = ErrorLevels.LEVEL_WARNING;
			result.append(com.nxp.swtools.utils.Messages.get().ToolTipableFormater_Warnings);
		} else if (((status = child.getInfo()) != null) || (problemLevelOfChild == ErrorLevels.LEVEL_INFORMATION)) {
			statusLevel = ErrorLevels.LEVEL_INFORMATION;
			result.append(com.nxp.swtools.utils.Messages.get().ToolTipableFormater_Infos);
		}
		final List<@NonNull String> dependencyDescriptions = ValidationHelper.getSettingValidationProblems(child, statusLevel).stream().map(
				p -> p.getDependency().getDescription()).collect(CollectorsUtils.toList());
		final String dependencyMessage = dependencyDescriptions.isEmpty() ? null : ChildValidationHelper.mergeProblems(
				dependencyDescriptions.toArray(new @Nullable String[dependencyDescriptions.size()]));
		final String statusAndDependencyMessage = ChildValidationHelper.mergeProblems(new @Nullable String[] {status, dependencyMessage});
		if (!UtilsText.isEmpty(statusAndDependencyMessage)) {
			result.append(UtilsText.LF);
			result.append(statusAndDependencyMessage);
		}
		final String decoratorDescription = (result.length() > 0) ? result.toString() : null;
		if (decoratorDescription != null) {
			ControlDecorationUtils.setImage(controlWithDecoration, ToolsImages.getStatusDecoratorImg(statusLevel));
		}
		ControlDecorationUtils.updateControlDecoration(controlWithDecoration, decoratorDescription);
	}

	/**
	 * Create error decoration for the given control. The decoration is hidden by default. Its state should be updated by calling
	 * {@link #updateErrorDecoration(Control)}
	 * @param controlToDecorate control for which to create decoration
	 * @param position of the decoration, possible values are defined in {@link ControlDecoration#ControlDecoration(Control, int)}
	 */
	protected void createErrorDecoration(@NonNull Control controlToDecorate, int position) {
		ControlDecorationUtils.createErrorDecoration(controlToDecorate, position);
	}
	
	/**
	 * Update content of a label.
	 * @param label the control to update
	 * @param updateType whether to disable children (if there are some children)
	 */
	protected void updateLabelContent(@NonNull Control label, @NonNull UpdateType updateType) {
		if ((label instanceof Label) && (updateType != UpdateType.PROBLEM_DECORATION)) {
			((Label) label).setText(child.getUiName());
		}
		updateErrorDecoration(label);
	}
	
	/**
	 * Update main content of a setting configuration, e.g. an editor.
	 * @param contentControl the mainControl to update
	 * @param updateType whether to disable children (if there are some children)
	 */
	protected void updateMainContent(@NonNull Control contentControl, @NonNull UpdateType updateType) {
		if ((contentControl instanceof ReadOnlyText) && (updateType != UpdateType.PROBLEM_DECORATION)) {
			((ReadOnlyText) contentControl).setText(child.getUiName());
		}
		updateErrorDecoration(contentControl);
	}
	
	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.IChildControl#dispose()
	 */
	@Override
	public void dispose() {
		if (labelControl != null) {
			labelControl.dispose();
		}
		if (mainControl != null) {
			mainControl.dispose();
		}
		labelControl = null;
		mainControl = null;
		disposed = true;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl#isDisposed()
	 */
	@Override
	public boolean isDisposed() {
		return disposed;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.IChildControl#getChild()
	 */
	@Override
	public @NonNull IChild getChild() {
		return child;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl#getContent()
	 */
	@Override
	public @Nullable Composite getContent() {
		return parent;
	}
	
	/**
	 * Create label control.
	 * @param composite to create label in
	 * @return the newly created control or {@code null} if there should not be label
	 */
	public @Nullable Control createLabelControl(@NonNull Composite composite) {
		Label label = new LabelAmpersand(composite, SWT.WRAP);
		SWTFactoryProxy.INSTANCE.setTestId(label, TestIDs.PERIPHS_SETTING_LABEL + child.getId());
		label.setText(child.getUiName());
		return label;
	}
	
	/**
	 * Create main mainControl.
	 * @param composite to create mainControl in
	 * @return the newly created mainControl or {@code null} if there should not be content
	 */
	public @Nullable Control createMainControl(@NonNull Composite composite) {
		ReadOnlyText text = new ReadOnlyText(composite, getSwtStyle());
		text.setText(child.getUiName());
		return text;
	}
	
	/**
	 * Get integer style bits for SWT based on defined options (currently only hideBorder).
	 * @return SWT style bits
	 */
	public int getSwtStyle() {
		return getSwtStyle(this);
	}
	
	/**
	 * Get integer style bits for SWT based on defined options (currently only hideBorder).
	 * @param childControl child for which style should be calculated
	 * @return SWT style bits
	 */
	public static int getSwtStyle(@NonNull IChildControl childControl) {
		if (childControl.getControlOptions().isBorderHidden()) {
			return SWT.NONE;
		} else {
			return SWT.BORDER;
		}
	}
	
	/**
	 * Returns current status icon
	 * @param element to get status from
	 * @return current status icon or {@code null} when there is no problem
	 */
	public static @Nullable Image getStatusIcon(@NonNull IChild element) {
		if (element.getError() != null) {
			return ToolsImages.getStatusIcon(ErrorLevels.LEVEL_ERROR);
		} else if (element.getWarning() != null) {
			return ToolsImages.getStatusIcon(ErrorLevels.LEVEL_WARNING);
		} else if (element.getInfo() != null) {
			return ToolsImages.getStatusIcon(ErrorLevels.LEVEL_INFORMATION);
		}
		return null;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl#getControlOptions()
	 */
	@Override
	public @NonNull ControlOptions getControlOptions() {
		return this.controlOptions;
	}

	/**
	 * Adds scroll listener that scrolls with ScrolledComposite which is parent of contentContainer
	 * @param control to which add this scroll listener
	 * @param childOfScrolledComposite child of ScrolledComposite which should be scrolled
	 */
	public static void addScrollListener(@NonNull Control control, @NonNull Composite childOfScrolledComposite) {
		control.addListener(SWT.MouseWheel, new Listener() {
			@Override
			public void handleEvent(Event event) {
				event.doit = false; // Do not execute handle in original event listener
				ScrolledCompositeHelper.scrollByNumberOfLines(childOfScrolledComposite, -event.count);
			}
		});
	}

	/**
	 * Registers copy and paste menu to right mouse button click
	 * @param control that will support copy and paste
	 */
	protected void registerCopyPasteMenu(@NonNull Control control) {
		MenuManager menuMgr = new MenuManager();
		ISettingConfig setting = null;
		if (getChild() instanceof ISettingConfig) {
			setting = (ISettingConfig) getChild();
			final ISettingConfig finalSetting = setting;
			menuMgr.setRemoveAllWhenShown(true);
			menuMgr.addMenuListener(new IMenuListener() {
				@Override
				public void menuAboutToShow(IMenuManager manager) {
					// Copy
					manager.add(ActionAnalyticsBuilder.action(new Runnable() {
						@Override
						public void run() {
							controllerWrapper.getController().copy(finalSetting);
						}
					})
					.id(TestIDs.PERIPHS_CHILD_CONTROL_COPY)
					.text(Messages.get().ChildControl_Copy)
					.image(ToolsImages.getImageDescriptor(IToolsImages.ICON_COPY))
					.enabled(controllerWrapper.getController().canCopy(finalSetting))
					.build());
					// Paste
					manager.add(ActionAnalyticsBuilder.action(new Runnable() {
						@Override
						public void run() {
							controllerWrapper.getController().paste(finalSetting);
						}
					})
					.id(TestIDs.PERIPHS_CHILD_CONTROL_PASTE)
					.text(Messages.get().ChildControl_Paste)
					.image(ToolsImages.getImageDescriptor(IToolsImages.ICON_COPY))
					.enabled(controllerWrapper.getController().canPaste(finalSetting))
					.build());
					// Paste as new item
					if (finalSetting instanceof ArrayConfig) {
						ArrayConfig finalArray = (ArrayConfig) finalSetting;
						manager.add(ActionAnalyticsBuilder.action(new Runnable() {
							@Override
							public void run() {
								controllerWrapper.getController().pasteAsNewItem(finalArray);
							}
						})
						.id(TestIDs.PERIPHS_CHILD_CONTROL_PASTE_AS_NEW_ITEM)
						.text(Messages.get().ChildControl_Paste_AsNewItem)
						.image(ToolsImages.getImageDescriptor(IToolsImages.ICON_COPY))
						.enabled(controllerWrapper.getController().canPasteAsNewItem(finalArray))
						.build());
					}
				}
			});
			control.setMenu(menuMgr.createContextMenu(control));
		}
	}

}
