/**
 * Copyright 2017-2019 NXP
 * Created: Dec 7, 2017
 */

package com.nxp.swtools.periphs.gui.view;


import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IDialogLabelKeys;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import com.nxp.swtools.common.ui.utils.swt.FontFactory;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionMap;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.model.config.ComponentConfig;
import com.nxp.swtools.periphs.model.config.ComponentInstanceConfig;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.progress.ProgressUtils;

/**
 * Dialog used for removing components/component instances from peripherals tool.
 * @author David Danaj (b57899/nxa30572)
 */
public class RemoveDialog extends MessageDialog {
	/** List with names of components for removal */
	private @Nullable List<@NonNull String> components;
	/** Map with component instances for removal. Key is name of a functional group,
	 * values are component instance configs defined in each functional group */
	private @Nullable CollectionMap<String, ComponentInstanceConfig> componentInstances;
	/** Horizontal indent from left border to show that component instances are in functional group */
	private static final int INDENT_COMPONENT_INSTANCES_FROM_LEFT = 15;
	/** Vertical space between UI part with removing components text and UI part with removing component instances text */
	private static final int SPACE_BETWEEN_COMPONENT_AND_COMPONENTINSTANCE_MESSAGES = 15;
	/** Vertical space between composite with messages and question at the bottom of the dialog */
	private static final int SPACE_BETWEEN_MESSAGE_AND_QUESTION = 15;

	/**
	 * Constructor.
	 * @param shell the parent shell
	 * @param components {@link #components}
	 * @param componentInstances {@link #componentInstances}
	 */
	private RemoveDialog(@Nullable Shell shell, @Nullable List<@NonNull String> components,
			@Nullable CollectionMap<String, ComponentInstanceConfig> componentInstances) {
		super(shell, Messages.get().RemoveDialog_Title, null, null, MessageDialog.QUESTION,
				new String[] { JFaceResources.getString(IDialogLabelKeys.YES_LABEL_KEY),
						JFaceResources.getString(IDialogLabelKeys.NO_LABEL_KEY) }, SWT.NONE);
		this.components = components;
		this.componentInstances = componentInstances;
	}

	/**
	 * Opens new dialog with information about which components and component instances will be removed, asking for a confirmation
	 * from a user.
	 * @param shell the parent shell
	 * @param componentConfigs LinkedHashSet with component configs that should be displayed in a dialog
	 * @param componentInstanceConfigs List of component instance configs that should be displayed in a dialog
	 * @return answer from the user
	 */
	public static boolean open(@Nullable Shell shell, @Nullable LinkedHashSet<@NonNull ComponentConfig> componentConfigs, @Nullable List<@NonNull ComponentInstanceConfig> componentInstanceConfigs) {
		List<@NonNull String> components = getComponentsNames(componentConfigs);
		CollectionMap<String, ComponentInstanceConfig> componentInstancesToRemove = getComponentInstancesNames(componentConfigs, componentInstanceConfigs);
		RemoveDialog dialog = new RemoveDialog(shell, components, componentInstancesToRemove);
		return dialog.open() == 0;
	}

	/**
	 * Convert linkedHashSet of component configs to list containing names of components.
	 * @param componentConfigs LinkedHashSet with component configs that should be converted
	 * @return list with names of components
	 */
	private static @Nullable List<@NonNull String> getComponentsNames(@Nullable LinkedHashSet<@NonNull ComponentConfig> componentConfigs) {
		if ((componentConfigs != null) && (!componentConfigs.isEmpty())) {
			return componentConfigs.stream()
				.map(cc -> cc.getUiName())
				.collect(CollectorsUtils.toList());
		}
		return null;
	}

	/**
	 * Creates CollectionMap from names of component instances.
	 * @param componentConfigs LinkedHashSet with component for removal. Used to obtain component instances from other functional
	 * groups
	 * @param componentInstanceConfigs List of component instance configs for removal
	 * @return CollectionMap of names of functional groups and their component instance configs that would be removed. The key is name of
	 * the functional group, values are component instance configs to be removed in each functional group.
	 */
	private static @Nullable CollectionMap<String, ComponentInstanceConfig> getComponentInstancesNames(
			@Nullable LinkedHashSet<@NonNull ComponentConfig> componentConfigs,
			@Nullable List<@NonNull ComponentInstanceConfig> componentInstanceConfigs) {
		CollectionMap<String, ComponentInstanceConfig> componentInstancesToRemove = new CollectionMap<>();
		// get component instances that have to be removed based on components that are being removed
		if ((componentConfigs != null) && (!componentConfigs.isEmpty())) {
			Controller.getInstance().getProfile().getFunctionalGroups().stream() // get all fnGroups
				.forEach(g -> componentConfigs.stream() // get components used in a group
					.map(cc -> g.getInstancesOfType(cc.getType()))	 // get all instances of that component
					.filter(compInstConfigs -> !compInstConfigs.isEmpty()) // filter out empty collections of component instances
					.flatMap(compInstConfigs -> compInstConfigs.stream()) // get each component instance
					.forEach(cic -> componentInstancesToRemove.add(g.getUiName(), cic))); // add instance to collection map
		}

		// find functionalgroups of component instances that have to be removed and add them to the collectionMap
		if ((componentInstanceConfigs != null) && (!componentInstanceConfigs.isEmpty())) {
			Controller.getInstance().getProfile().getFunctionalGroups().stream()
				.forEach(g -> componentInstanceConfigs.stream()
					.filter(cc -> g.getInstances().containsValue(cc))
					.forEach(filteredCC -> componentInstancesToRemove.add(g.getUiName(), filteredCC)));
		}
		return componentInstancesToRemove.isEmpty() ? null : componentInstancesToRemove;
	}

	/**
	 * Creates message to display in a dialog. Each line in a message looks like: componentInstanceName [componentName]
	 * @param fnGroup name of functional group in which component instance configs are defined
	 * @return String to display in a dialog with all component instance configs names and their components
	 */
	private @NonNull String getRemoveComponentInstancesMessage(@NonNull String fnGroup) {
		String resultMessage = UtilsText.EMPTY_STRING;
		if (componentInstances != null) {
			resultMessage = Objects.requireNonNull(componentInstances.get(fnGroup)).stream()
					.map((ci) -> (ci.getUiName() + " [" + ci.getComponent().getLabel(ci.getExpressionContext()) + "]")) //$NON-NLS-1$ //$NON-NLS-2$
					.collect(Collectors.joining(UtilsText.CRLF));
		}
		return UtilsText.safeString(resultMessage);
	}

	/**
	 * Creates message to display in a dialog. Each line contains name of a component for removal.
	 * @return String with components' names for removal.
	 */
	private @NonNull String getRemoveComponentsMessage() {
		String resultMessage = UtilsText.EMPTY_STRING;
		List<@NonNull String> componentsLoc = components;
		if (componentsLoc != null) {
			resultMessage = componentsLoc.stream()
				.collect(Collectors.joining(UtilsText.CRLF));
		}
		return UtilsText.safeString(resultMessage);
	}


	/**
	 * Creates label with confirmation question at the bottom of the dialog.
	 * @param parent parent composite
	 * @return newly created control for a custom area
	 */
	@Override
	protected @Nullable Control createCustomArea(@Nullable Composite parent) {
		Label confirmation = new Label(parent, SWT.BOTTOM);
		confirmation.setText(Messages.get().RemoveDialog_Confirmation);
		GridDataFactory
			.fillDefaults()
			.align(SWT.BEGINNING, SWT.END)
			.grab(true, false)
			.indent(0, SPACE_BETWEEN_MESSAGE_AND_QUESTION)
			.applyTo(confirmation);
		return confirmation;
	}

	/**
	 * Creates messages in a dialog. Contains information about components and component instances for removal.
	 * @param composite parent composite
	 * @return newly created control for a message area
	 */
	@Override
	protected Control createMessageArea(Composite composite) {
		super.createMessageArea(composite);
		Composite messageComposite = new Composite(composite, SWT.NONE);
		SWTFactoryProxy.INSTANCE.setTestId(messageComposite, TestIDs.PERIPHS_REMOVE_DIALOG_COMPOSITE);
		// set dialog appearance
		GridDataFactory
			.fillDefaults()
			.align(SWT.FILL, SWT.BEGINNING)
			.grab(true, false)
			.hint(convertHorizontalDLUsToPixels(IDialogConstants.MINIMUM_MESSAGE_AREA_WIDTH), SWT.DEFAULT)
			.applyTo(messageComposite);
		messageComposite.setLayout(new GridLayout(1, true));
		createComponentsPart(messageComposite);
		createComponentInstancesPart(messageComposite);
		return messageComposite;
	}

	/**
	 * Creates part of a dialog where functional groups, component instances and their respective components are shown.
	 * @param messageComposite parent composite
	 */
	private void createComponentInstancesPart(@NonNull Composite messageComposite) {
		CollectionMap<String, ComponentInstanceConfig> componentInstancesLoc = componentInstances;
		// if there is something to be removed
		if ((componentInstancesLoc != null) && (!componentInstancesLoc.isEmpty())) {
			// header message
			Label removingComponentInstances = new Label(messageComposite, SWT.NONE);
			removingComponentInstances.setText(Messages.get().RemoveDialog_RemoveComponentInstances);
			FontFactory.changeStyle(removingComponentInstances, SWT.BOLD);
			// if there are any components for removal, add vertical space between them and component instances header
			if (components != null) {
				GridData removingComponentInstancesLayoutData = new GridData(SWT.LEFT, SWT.TOP, true, true);
				removingComponentInstancesLayoutData.verticalIndent = SPACE_BETWEEN_COMPONENT_AND_COMPONENTINSTANCE_MESSAGES;
				removingComponentInstances.setLayoutData(removingComponentInstancesLayoutData);
			}
			// for each functional group
			for (Entry<String,Collection<ComponentInstanceConfig>> entry : componentInstancesLoc.entrySet()) {
				// create header with information about current functional group ("From fnGroupName:")
				StringBuilder fromFnGroup = new StringBuilder();
				fromFnGroup.append(Messages.get().RemoveDialog_From);
				fromFnGroup.append(UtilsText.SPACE);
				fromFnGroup.append(entry.getKey());
				Label functionalGroupLabel = new Label(messageComposite, SWT.NONE);
				functionalGroupLabel.setText(fromFnGroup.toString());
				// create label with information about component instances in this functional group
				Label componentInstancesInFnGroup = new Label(messageComposite, SWT.NONE);
				componentInstancesInFnGroup.setText(getRemoveComponentInstancesMessage(UtilsText.safeString(entry.getKey())));
				GridData instancesLayoutData = new GridData(SWT.LEFT, SWT.TOP, true, true);
				instancesLayoutData.horizontalIndent = INDENT_COMPONENT_INSTANCES_FROM_LEFT;
				componentInstancesInFnGroup.setLayoutData(instancesLayoutData);
			}
		}
	}

	/**
	 * Creates part of a dialog where components are shown.
	 * @param messageComposite parent composite
	 */
	private void createComponentsPart(@NonNull Composite messageComposite) {
		// if there is something to remove
		if ((components != null) && (!components.isEmpty())) {
			// create header
			Label removingComponents = new Label(messageComposite, SWT.NONE);
			removingComponents.setText(Messages.get().RemoveDialog_RemoveComponents);
			FontFactory.changeStyle(removingComponents, SWT.BOLD);
			// create label with components' names
			Label componentsLabel = new Label(messageComposite, SWT.NONE);
			componentsLabel.setText(getRemoveComponentsMessage());
		}
	}

	/**
	 * Removes component instances and components. Opens dialogs if removing last instances of a components that have global
	 * config sets.
	 * @param componentConfigs LinkedHashSet of component configs that have to be removed. Can be the same as in
	 * {@link #open(Shell, LinkedHashSet, List)} method
	 * @param componentInstanceConfigs List of component instance configs that have to be removed. Can be the same as in
	 * {@link #open(Shell, LinkedHashSet, List)} method
	 * @param caller the method caller (used for identification in events)
	 * @return {@code true} if anything was successfully removed, {@code false} otherwise
	 */
	public static boolean remove(@Nullable LinkedHashSet<@NonNull ComponentConfig> componentConfigs,
			@Nullable Collection<@NonNull ComponentInstanceConfig> componentInstanceConfigs, @NonNull Object caller) {
		Controller controller = Controller.getInstance();
		Supplier<Boolean> removeSupplier = () -> {
			boolean anyRemoved = false;
			if (componentConfigs != null) {
				anyRemoved |= controller.removeComponents(componentConfigs.stream()
						.map(cc -> cc.getId())
						.collect(CollectorsUtils.toList()), caller);
			}
			if (componentInstanceConfigs != null) {
				anyRemoved |= controller.removeComponentInstances(componentInstanceConfigs.stream()
						.map(cic -> cic.getId())
						.collect(CollectorsUtils.toList()), true, caller);
				Set<@NonNull String> componentsToCheck = componentInstanceConfigs.stream()
						.map(cic -> cic.getComponent().getId())
						.collect(Collectors.toSet());
				RemoveComponentInstancesDialog.removeLastInstanceDialog(componentsToCheck);
			}
			return new Boolean(anyRemoved);
		};
		Boolean result = ProgressUtils.run(() -> CollectionsUtils.nullableOptionalGet(controller.runTransaction(removeSupplier)));
		return result == null ? false : result.booleanValue();
	}
}
