/**
 * Copyright 2017-2019 NXP
 * Created: Jul 19, 2017
 */
package com.nxp.swtools.periphs.gui.view;

import java.text.MessageFormat;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IDialogLabelKeys;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionMap;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.model.config.ComponentInstanceConfig;
import com.nxp.swtools.periphs.model.data.ConfigurationComponent;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.progress.ProgressUtils;

/**
 * Dialog used for removing component instances
 * @author Tomas Rudolf - nxf31690
 */
public class RemoveComponentInstancesDialog extends MessageDialog {

	/** ID of the dialog */
	public static final @NonNull String ID = "com.nxp.swtools.periphs.gui.view.removeComponentDialog"; //$NON-NLS-1$
	/** Controller instance */
	@NonNull Controller controller = Controller.getInstance();
	/** Reference to checkbox table viewer */
	@Nullable CheckboxTableViewer checkboxTableViewer;
	/** Cancel button */
	@Nullable Button cancelButton;
	/** Remove button */
	@Nullable Button removeButton;
	/** List of instances */
	@NonNull Collection <@NonNull ComponentInstanceConfig> componentInstances;
	/** ViewSite to work with */
	@NonNull Shell shell;

	/**
	 * Open a dialog
	 * @param shell current shell
	 * @param instances component instances to remove
	 * @return the created dialog
	 */
	public static @NonNull RemoveComponentInstancesDialog open(@NonNull Shell shell, @NonNull Collection<@NonNull ComponentInstanceConfig> instances) {
		RemoveComponentInstancesDialog dialog = new RemoveComponentInstancesDialog(shell, instances);
		dialog.setBlockOnOpen(false);
		dialog.open();
		return dialog;
	}

	/**
	 * Constructor.
	 * @param shell current shell
	 * @param instances component instances which should be removed
	 */
	private RemoveComponentInstancesDialog(@NonNull Shell shell, @NonNull Collection<@NonNull ComponentInstanceConfig> instances) {
		super(shell, Messages.get().ComponentsView_RemoveComponentInstancesDialog_Title, null, null, MessageDialog.CONFIRM,
				new String[] { JFaceResources.getString(IDialogLabelKeys.YES_LABEL_KEY),
						JFaceResources.getString(IDialogLabelKeys.NO_LABEL_KEY) }, SWT.NONE);
		this.shell = shell;
		componentInstances = instances;
	}

	/**
	 * Crates dialog UI and prepares it for usage
	 */
	@Override
	protected Control createMessageArea(Composite parent) {
		Composite container = (Composite) super.createMessageArea(parent);
		Composite messageComposite = new Composite(container, SWT.NONE);
		// set dialog appearance
		GridDataFactory
			.fillDefaults()
			.align(SWT.FILL, SWT.BEGINNING)
			.grab(true, false)
			.hint(convertHorizontalDLUsToPixels(IDialogConstants.MINIMUM_MESSAGE_AREA_WIDTH), SWT.DEFAULT)
			.applyTo(messageComposite);
		messageComposite.setLayout(new GridLayout(1, true));
		Label messageHeader = new Label(messageComposite, SWT.NONE);
		messageHeader.setText(Messages.get().ComponentsView_RemoveComponentInstancesDialog_Message);
		createTableViewer(messageComposite);
		return container;
	}

	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		this.removeButton = createButton(parent, IDialogConstants.YES_ID, JFaceResources.getString(IDialogLabelKeys.YES_LABEL_KEY), true);
		this.cancelButton = createButton(parent, IDialogConstants.NO_ID, JFaceResources.getString(IDialogLabelKeys.NO_LABEL_KEY), false);
	}

	@Override
	protected void buttonPressed(int buttonId) {
		if (buttonId == IDialogConstants.YES_ID) {
			okPressed();
		} else if (buttonId == IDialogConstants.NO_ID) {
			cancelPressed();
		}
		super.buttonPressed(buttonId);
	}

	@Override
	protected void okPressed() {
		// Remove component instances
		CollectionMap<@NonNull String, @NonNull String> instancesToRemove = new CollectionMap<>();
		if (checkboxTableViewer != null) {
			List<?> checked = CollectionsUtils.safeList(checkboxTableViewer.getCheckedElements());
			// Iterate over all selected items
			for (Object selectedItem : checked) {
				if ((selectedItem != null) && (selectedItem instanceof ComponentInstanceConfig)) {
					ComponentInstanceConfig selectedItemInstance = (ComponentInstanceConfig) selectedItem;
					componentInstances.stream()
						.filter(x -> x.getName().equals(selectedItemInstance.getName()))
						.forEach(x -> instancesToRemove.add(x.getComponent().getId(), x.getName()));
				}
			}
		}
		if (removeButton != null) {
			controller.runTransaction(() -> {
				ProgressUtils.run((m) -> controller.removeComponentInstances(instancesToRemove, removeButton));
				removeLastInstanceDialog(instancesToRemove.keySet());
			});
		}
		super.okPressed();
	}

	/**
	 * Creates table viewer in given composite
	 * @param composite to create TableViewer in
	 */
	private void createTableViewer(@NonNull Composite composite) {
		Composite tableViewerComposite = new Composite(composite, SWT.NONE);
		GridLayout layout = new GridLayout(1, true);
		tableViewerComposite.setLayout(layout);
		tableViewerComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		CheckboxTableViewer viewer = CheckboxTableViewer.newCheckList(tableViewerComposite, SWT.NONE);
		this.checkboxTableViewer = viewer;
		SWTFactoryProxy.INSTANCE.setTestId(viewer.getTable(), TestIDs.PERIPHS_REMOVE_COMPONENT_SHELL_LIST);
		viewer.setContentProvider(ArrayContentProvider.getInstance());
		viewer.addCheckStateListener(new ICheckStateListener() {
			@Override
			public void checkStateChanged(CheckStateChangedEvent event) {
				setButtonState(removeButton, viewer.getCheckedElements());
			}
		});
		viewer.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public String getText(Object element) {
				assert (element instanceof ComponentInstanceConfig);
				return ((ComponentInstanceConfig) element).getUiName();
			}
		});
		viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			// do not allow selection in names column
			@Override
			public void selectionChanged(SelectionChangedEvent event) {
				viewer.getTable().deselectAll();
				tableViewerComposite.setFocus();
				tableViewerComposite.forceFocus();
			}
		});
		viewer.setInput(componentInstances);
		viewer.setAllChecked(true);
		GridData tableLayoutData = new GridData(SWT.FILL, SWT.FILL, true, true);
		viewer.getTable().setLayoutData(tableLayoutData);
		Label confirmation = new Label(composite, SWT.NONE);
		confirmation.setText(Messages.get().ComponentsView_RemoveComponentInstancesDialog_Confirmation);
	}

	/**
	 * Set state of the button based on current selection
	 * @param button button to configure it's state
	 * @param checked array of checked objects in a table
	 */
	static void setButtonState(@Nullable Button button, Object @Nullable[] checked) {
		if (button != null) {
			button.setEnabled(checked != null ? (checked.length != 0) : (false));
		}
	}

	/**
	 * Format user message to show name of component instance and component for which it is configured.
	 * @param componentInstance component instance to format output for
	 * @return formatted text for user with name of component instance and component name
	 */
	public static @NonNull String formatInstanceOutput(@NonNull ComponentInstanceConfig componentInstance) {
		StringBuilder output = new StringBuilder();
		output.append(componentInstance.getUiName());
		output.append(UtilsText.SPACE);
		output.append(UtilsText.LEFT_BRACKET);
		output.append(componentInstance.getComponent().getLabel(componentInstance.getExpressionContext()));
		output.append(UtilsText.RIGHT_BRACKET);
		return UtilsText.safeString(output.toString());
	}

	/**
	 * Creates dialog after removing last component instance if needed and removes component if user confirms.
	 * Checks if component is configured and has no component instances configured.
	 * If component has global settings, ask user whether it should be removed, otherwise remove it without asking.
	 * @param componentsToCheck which components have to be checked
	 */
	public static void removeLastInstanceDialog(@Nullable Set<@NonNull String> componentsToCheck) {
		Controller controller = Controller.getInstance();
		if (componentsToCheck != null) {
			for (String affectedComponent : componentsToCheck) {
				if ((controller.getProfile().isComponentConfigured(affectedComponent))
						&& (controller.getAllComponentInstances(affectedComponent).isEmpty())) {
					boolean shouldBeRemoved = true;
					ConfigurationComponent configurationComponent = controller.getConfigurationComponent(affectedComponent);
					if ((configurationComponent != null) && (configurationComponent.getGlobalConfigSet() != null)) {
						shouldBeRemoved = MessageDialog.openQuestion(null,
								Messages.get().ComponentsView_RemoveComponentDialog_RemovingLastInstance_Title,
								MessageFormat.format(
										UtilsText.safeString(Messages.get().ComponentsView_RemoveComponentDialog_RemovingLastInstance_Message),
										configurationComponent.getLabel(controller.getProfile().getExpressionContext()))
								);
					}
					if (shouldBeRemoved) {
						ProgressUtils.run((m) -> controller.removeComponents(CollectionsUtils.asList(affectedComponent), ComponentsView.class));
					}
				}
			}
		}
	}
}
