/**
 * Copyright 2018-2019 NXP
 * Created: Jul 13, 2018
 */
package com.nxp.swtools.periphs.gui.view;

import java.io.IOError;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.variables.VariablesPlugin;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.json.JSONException;
import org.json.JSONObject;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.model.data.AvailableComponents;
import com.nxp.swtools.periphs.model.data.ConfigurationComponentTypeId;
import com.nxp.swtools.provider.configuration.ConfigChangeReason;
import com.nxp.swtools.provider.configuration.ISharedConfiguration;
import com.nxp.swtools.provider.configuration.SharedConfigurationAdapter;
import com.nxp.swtools.provider.configuration.SharedConfigurationFactory;
import com.nxp.swtools.utils.support.markdown.MarkDownSupport;
import com.nxp.swtools.utils.view.ToolView;

/**
 * Displays component documentation in the markdown syntax
 * @author Viktar Paklonski
 */
public class DocumentationView extends ToolView {

	/** Logger of class */
	static final @NonNull Logger LOGGER = LogManager.getLogger(DocumentationView.class);
	/** ID of the view */
	public static final @NonNull String ID = "com.nxp.swtools.periphs.gui.view.documentationView"; //$NON-NLS-1$
	/** markdown keyword */
	private static final @NonNull String RAW_DATA_PREFIX = "data:"; //$NON-NLS-1$
	/** file scheme name */
	private static final @NonNull String FILE_SCHEME = "file"; //$NON-NLS-1$ 
	/** The platform specific */
	transient @NonNull SWTFactoryProxy swtFactory = SWTFactoryProxy.INSTANCE;
	/** HTML viewer */
	private @Nullable Composite viewer = null;
	/** HTML text */
	private @NonNull String content = UtilsText.EMPTY_STRING;
	/** HTML Document Tag */
	private static final @NonNull String HTML_DOC_TAG = "component-doc-link"; //$NON-NLS-1$
	/** TypeId of component that opened this view */
	@Nullable ConfigurationComponentTypeId componentTypeId;

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.ViewPart#init(org.eclipse.ui.IViewSite)
	 */
	@Override
	public void init(IViewSite site) throws PartInitException {
		setSite(site);
		final String typeId = UtilsText.safeString(site.getSecondaryId());
		final AvailableComponents availComps = Controller.getInstance().getMcu().getAvailableComponents();
		ConfigurationComponentTypeId componentTypeIdLoc = availComps.getConfigCompTypeId(typeId);
		componentTypeId = componentTypeIdLoc;
		final boolean willBeClosed = updateCurrentView(typeId, availComps, componentTypeId);
		if (willBeClosed) {
			return;
		}
		if (componentTypeIdLoc == null) {
			return;
		}
		final String docContent = componentTypeIdLoc.loadDocumentation();
		if ((docContent != null) && docContent.contains(HTML_DOC_TAG)) {
			openDocInBrowser(docContent, componentTypeIdLoc.getFileLocation().toAbsolutePath().toString());
		}
		content = MarkDownSupport.markDownToHtml(UtilsText.safeString(docContent), new Function<@NonNull String, @NonNull String>() {
			@Override
			public String apply(String t) {
				if (t.startsWith(RAW_DATA_PREFIX)) {
					// e.g. "data:image/png;base64,<...>" 
					return t; // skip embedded data 
				}
				URI uri;
				try {
					uri = new URI(t);
				} catch (final URISyntaxException e) {
					LOGGER.log(Level.SEVERE, String.format("Invalid URI in '%1s' with typeId '%2s': %3s", ConfigurationComponentTypeId.class.getName(), typeId, t), e); //$NON-NLS-1$
					return t;
				}
				final String scheme = UtilsText.safeString(uri.getScheme());
				if (!scheme.isEmpty() && !scheme.equals(FILE_SCHEME)) {
					// e.g. https://www.nxp.com/
					return t; // don't know how to process non-file URIs
				}
				Path filePath = Paths.get(uri.getSchemeSpecificPart()); // cannot use Paths.get(uri); since it requires FILE_SCHEME
				if (!filePath.isAbsolute()) {
					try {
						filePath = componentTypeIdLoc.resolvePath(filePath).toAbsolutePath();
					} catch (IOError e) {
						LOGGER.log(Level.SEVERE, String.format("Failed to resolve URI in '%1s' with typeId '%2s': %3s", ConfigurationComponentTypeId.class.getName(), typeId, t), e); //$NON-NLS-1$
					}
				}
				return filePath.toUri().toString();
			}
		});
	}

	/**
	 * Checks that current view should still be open and closes it if not
	 * @param type of component
	 * @param availComps available components
	 * @param typeId of current component
	 * @return {@code true} if the view will be closed by this function or {@code false} if not
	 */
	boolean updateCurrentView(@Nullable String type, @NonNull AvailableComponents availComps, @Nullable ConfigurationComponentTypeId typeId) {
		if (type == null) {
			hideSelf();
			return true;
		}
		if (availComps.getConfigComps().isEmpty()) {
			hideSelf();
			return true;
		}
		if (typeId == null) {
			LOGGER.severe(String.format("A '%1s' with typeId '%2s' is missing among available components", ConfigurationComponentTypeId.class.getName(), type)); //$NON-NLS-1$
			hideSelf();
			return true;
		}
		if (!typeId.isDocumentationPresent()) {
			LOGGER.severe(String.format("No documentation found for '%1s' with typeId '%2s'", ConfigurationComponentTypeId.class.getName(), type)); //$NON-NLS-1$
			hideSelf();
			return true;
		}
		return false;
	}

	/**
	 * Hides (and closes a single instance of) the view.
	 */
	void hideSelf() {
		final Display display = Display.getCurrent();
		if (display != null) {
			display.asyncExec(new Runnable() {
				@Override
				public void run() {
					IViewSite site = getViewSite();
					assert site != null;
					site.getWorkbenchWindow().getActivePage().hideView(DocumentationView.this);
				}
			});
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createPartControl(Composite parent) {
		assert parent != null;
		final Composite viewerLoc = swtFactory.createHtmlViewer(parent, SWT.NONE);
		if (viewerLoc != null) {
			swtFactory.setHtmlViewerText(viewerLoc, content);
			viewer = viewerLoc;
		}
		SharedConfigurationFactory.getSharedConfigurationSingleton().addListener(new SharedConfigurationAdapter() {
			@Override
			public void configurationReloaded(ISharedConfiguration sharedConfig, ConfigChangeReason reason) {
				updateCurrentView((componentTypeId != null) ? componentTypeId.getTypeId() : UtilsText.EMPTY_STRING,
						Controller.getInstance().getMcu().getAvailableComponents(), componentTypeId);
				switch (reason) {
					case NEW_CONFIG: {
						hideSelf();
						break;
					}
					default: {
						// Do not react
					}
				}
			}
		});
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#setFocus()
	 */
	@Override
	public void setFocus() {
		final Composite viewerLoc = viewer;
		if (viewerLoc != null) {
			viewerLoc.setFocus();
		}
	}

	/**
	 * Open documentation view.
	 * @param viewSite with binding to the view
	 * @param typeId of the component to show documentation for
	 * @param activate whether to activate the view after creating
	 * @return {@code true} if the view was successfully opened, {@code false} otherwise
	 */
	public static boolean open(@NonNull IViewSite viewSite, @NonNull String typeId, boolean activate) {
		return open(viewSite.getPage(), typeId, activate);
	}

	/**
	 * Open documentation view.
	 * @param workbenchPage with binding to the view
	 * @param typeId of the component to show documentation for
	 * @param activate whether to activate the view after creating
	 * @return {@code true} if the view was successfully opened, {@code false} otherwise
	 */
	public static boolean open(@Nullable IWorkbenchPage workbenchPage, @NonNull String typeId, boolean activate) {
		if (workbenchPage != null) {
			try {
				workbenchPage.showView(ID, typeId, activate ? IWorkbenchPage.VIEW_ACTIVATE : IWorkbenchPage.VIEW_VISIBLE);
				return true;
			} catch (@SuppressWarnings("unused") PartInitException e) {
				return false;
			}
		}
		return false;
	}

	/**
	 * Try to open the documentation in browser
	 * The path to the .html file can be: ${ENV_PATH}/doc/html_${CPU}/group__flexcan__driver.html or 
	 * a relative path to components folder. If ${CPU} is found, it will be replaced with current processor.
	 * @param docContent - component documentation
	 * @param componentLocation - full path to the .component file
	 */
	private void openDocInBrowser(@NonNull String docContent, @NonNull String componentLocation) {
		try {
			JSONObject jsonOb = new JSONObject(docContent);
			String link = jsonOb.get(HTML_DOC_TAG).toString();
			// try to replace the &{CPU} with the mcu name
			link = link.replace("${CPU}", Controller.getInstance().getMcu().getMcuIdentification().getMcu()); //$NON-NLS-1$
			// try to replace the &{CPU_FAMILY} with the family of the MCU
			link = link.replace("${CPU_FAMILY}", Controller.getInstance().getMcu().getFamily()); //$NON-NLS-1$
			// replace the environment variables declared in the link. If the link doesn't contains env var, the link will be the same
			link = VariablesPlugin.getDefault().getStringVariableManager().performStringSubstitution(link, true);
			if (!Paths.get(link).isAbsolute()) {
				link = Paths.get(componentLocation, link).normalize().toString();
			}
			SWTFactoryProxy.INSTANCE.openUrlInExternalBrowser(UtilsText.safeString(link));
			hideSelf();
		} catch (JSONException | CoreException e) {
			LOGGER.info(e.getMessage());
		}
	}
}
