/**
 * Copyright 2018-2019 NXP
 */
package com.nxp.swtools.periphs.gui.view;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IViewSite;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.controller.events.EventTypes;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.ComponentSettingView;
import com.nxp.swtools.periphs.model.config.ComponentConfig;
import com.nxp.swtools.periphs.model.config.ComponentInstanceConfig;
import com.nxp.swtools.periphs.model.config.FunctionalGroup;
import com.nxp.swtools.periphs.model.config.IChild;
import com.nxp.swtools.periphs.model.data.ConfigurationComponent;
import com.nxp.swtools.periphs.model.data.ConfigurationComponentTypeId;
import com.nxp.swtools.provider.analytics.ActionAnalyticsBuilder;
import com.nxp.swtools.provider.configuration.storage.periphs.StoragePeriphsComponentInstance;
import com.nxp.swtools.provider.configuration.storage.periphs.StoragePeriphsFuncGroup;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsImages;

/**
 * Class used for creating context menu for viewer displaying components.
 * @author Tomas Rudolf
 * @author Juraj Ondruska
 */
public class ComponentMenuCreator {
	/** Child selection provider */
	@NonNull IComponentSelectionProvider selectionProvider;
	/** Peripheral selection provider */
	@Nullable IPeripheralSelectionProvider peripheralSelectionProvider;
	/** Menu options */
	@NonNull ComponentMenuOptions options;
	/** Wrapper containing the generic controller */
	@NonNull IControllerWrapper controllerWrapper;
	/**
	 * Constructor.
	 * @param selectionProvider the child selection provider
	 * @param options menu options
	 * @param controllerWrapper containing the generic controller
	 */
	public ComponentMenuCreator(@NonNull IComponentSelectionProvider selectionProvider, @NonNull ComponentMenuOptions options, @NonNull IControllerWrapper controllerWrapper) {
		this(selectionProvider, null, options, controllerWrapper);
	}
	
	/**
	 * Constructor.
	 * @param selectionProvider the child selection provider
	 * @param peripheralSelectionProvider peripheral selection provider
	 * @param options menu options
	 * @param controllerWrapper containing the generic controller
	 */
	public ComponentMenuCreator(@NonNull IComponentSelectionProvider selectionProvider, @Nullable IPeripheralSelectionProvider peripheralSelectionProvider,
			@NonNull ComponentMenuOptions options, @NonNull IControllerWrapper controllerWrapper) {
		this.selectionProvider = selectionProvider;
		this.peripheralSelectionProvider = peripheralSelectionProvider;
		this.options = options;
		this.controllerWrapper = controllerWrapper;
	}
	
	/**
	 * @param child component
	 * @return type ID of the component on success, {@code null} otherwise 
	 */
	static @Nullable ConfigurationComponentTypeId getTypeId(@NonNull IChild child) {
		if (child instanceof ComponentConfig) {
			return ((ComponentConfig) child).getConfigCompTypeId();
		}
		if (child instanceof ComponentInstanceConfig) {
			return ((ComponentInstanceConfig) child).getConfigCompTypeId();
		}
		return null;
	}
	
	/**
	 * Create context menu.
	 * @param control Control for which to create menu
	 * @param viewSite with the viewer
	 * @param selectionProviderArg selection provider
	 */
	public void createMenu(@NonNull Control control, @NonNull IViewSite viewSite, @Nullable ISelectionProvider selectionProviderArg) {
		createMenu(CollectionsUtils.asList(control), viewSite, selectionProviderArg);
	}
	
	/**
	 * Create context menu.
	 * @param controls list of Controls for which to create menu
	 * @param viewSite with the viewer
	 * @param selectionProviderArg selection provider
	 */
	public void createMenu(@NonNull List<@NonNull Control> controls, @NonNull IViewSite viewSite, @Nullable ISelectionProvider selectionProviderArg) {
		MenuManager menuMgr = createMenuManager(viewSite);
		for (Control control : controls) {
			Menu menu = menuMgr.createContextMenu(control);
			control.setMenu(menu);
			if (selectionProviderArg != null) {
				viewSite.registerContextMenu(menuMgr, selectionProviderArg);
			}
		}
	}

	/**
	 * Create menu manager
	 * @param viewSite needed for menu actions
	 * @return new menu manager
	 */
	private @NonNull MenuManager createMenuManager(@NonNull IViewSite viewSite) {
		MenuManager menuMgr = new MenuManager();
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {

			/**
			 * Checks if something is selected
			 * @return {@code true} if is something selected, otherwise returns {@code false}
			 */
			private boolean isSomethingSelected() {
				return !selectionProvider.getSelection().isEmpty();
			}

			/**
			 * Checks if system component is selected
			 * @return {@code true} if system component is selected, otherwise returns {@code false}
			 */
			private boolean isSystemComponentSelected() {
				for (IChild child : selectionProvider.getSelection()) {
					if ((child instanceof ComponentConfig) && (((ComponentConfig) child).getId().equals(ConfigurationComponent.SYSTEM_COMPONENT_ID))) {
						return true;
					}
				}
				return false;
			}

			/**
			 * Checks if component instance is documented
			 * @return {@code true} if instance is documented, otherwise returns {@code false}
			 */
			private boolean isDocumentedComponentSelected() {
				final Set<@NonNull ConfigurationComponentTypeId> set = new HashSet<>();
				for (IChild child : selectionProvider.getSelection()) {
					final ConfigurationComponentTypeId typeId = getTypeId(child);
					if (typeId == null) {
						return false;
					}
					set.add(typeId);
				}
				if (set.size() != 1) {
					return false;
				}
				final ConfigurationComponentTypeId componentTypeId = (ConfigurationComponentTypeId) set.toArray()[0];
				return componentTypeId.isDocumentationPresent();
			}

			/**
			 * Checks if any of selected items is component
			 * @return {@code true} if selection contains any component, otherwise returns {@code false}
			 */
			private boolean isComponentInSelection() {
				for (IChild child : selectionProvider.getSelection()) {
					if (child instanceof ComponentConfig) {
						return true;
					}
				}
				return false;
			}

			/**
			 * Get currently selected item
			 * @return first of currently selected items
			 */
			@Nullable IChild getSelectedItem() {
				Collection<? extends @NonNull IChild> selection = selectionProvider.getSelection();
				return selection.isEmpty() ? null : selection.iterator().next();
			}

			/**
			 * Get functional group of given child
			 * @param child to get functional group from
			 * @return functional group of given child
			 */
			@Nullable FunctionalGroup getFunctionalGroup(@NonNull IChild child) {
				FunctionalGroup functionalGroup = child.getChildContext().getFunctionalGroup();
				return functionalGroup;
			}

			/**
			 * Checks if selected items are in given group
			 * @param group for check
			 * @return {@code true} if selected items are in given group, {@code false} otherwise
			 */
			private boolean areSelectedItemsIn(@NonNull FunctionalGroup group) {
				boolean result = true;
				Iterator<? extends @Nullable IChild> selectedItemsIterator = selectionProvider.getSelection().iterator();
				while (selectedItemsIterator.hasNext()) {
					IChild selectedItem = selectedItemsIterator.next();
					if (selectedItem instanceof ComponentInstanceConfig) {
						ComponentInstanceConfig instanceConfig = (ComponentInstanceConfig) selectedItem;
						if (getFunctionalGroup(instanceConfig) != group) {
							result = false;
						}
					}
				}
				return result;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.action.IMenuListener#menuAboutToShow(org.eclipse.jface.action.IMenuManager)
			 */
			@Override
			public void menuAboutToShow(IMenuManager manager) {
				final IPeripheralSelectionProvider peripheralProvider = peripheralSelectionProvider;
				if (manager == null) {
					return;
				}
				addOpenInstanceAction(manager, viewSite);
				if (peripheralProvider != null) {
					manager.add(ActionAnalyticsBuilder.action(new Runnable() {
						@Override
						public void run() {
							final String peripheral = peripheralProvider.getSelection();
							if (peripheral != null) {
								PeripheralsView.configurePeripheral(peripheral, controllerWrapper.getController().getFunctionalGroup(), ComponentMenuCreator.this, controllerWrapper);
							}
						}
					})
					.id(TestIDs.PERIPHS_ACTION_ADD_NEW_COMPONENT)
					.text(Messages.get().ComponentMenuCreator_actionAddComponentInstance)
					.image(ToolsImages.getImageDescriptor(IToolsImages.ICON_ADD_CIRCLE))
					.enabled(peripheralProvider.getSelection() != null)
					.build());
				}
				
				MenuManager moveMenu = new MenuManager(Messages.get().ComponentsView_Menu_MoveTo);
				for (FunctionalGroup group : Controller.getInstance().getProfile().getFunctionalGroups()) {
					moveMenu.add(ActionAnalyticsBuilder.action(new Runnable() {
						@Override
						public void run() {
							// Move
							boolean change = false;
							Iterator<? extends @Nullable IChild> selectedItemsIterator = selectionProvider.getSelection().iterator();
							while (selectedItemsIterator.hasNext()) {
								IChild selectedItem = selectedItemsIterator.next();
								if (selectedItem instanceof ComponentInstanceConfig) {
									ComponentInstanceConfig instanceConfig = (ComponentInstanceConfig) selectedItem;
									FunctionalGroup oldGroup = instanceConfig.getChildContext().getFunctionalGroup();
									if (oldGroup == null) {
										return;
									}
									instanceConfig.getChildContext().setFunctionalGroup(group);
									oldGroup.removeInstance(instanceConfig.getName());
									group.addInstance(instanceConfig);
									change = true;
								}
							}
							if (change) {
								controllerWrapper.getController().handleSettingChange(EventTypes.CHANGE, ComponentMenuCreator.this, UtilsText.safeString(
										com.nxp.swtools.periphs.controller.Messages.get().Controller_Action_MoveComponentInstance));
							}
						}
					})
					.id(TestIDs.PERIPHS_ACTION_MOVE_INSTANCE + UtilsText.UNDERSCORE + group.getUiName())
					.text(group.getUiName())
					.image(null)
					.enabled(!areSelectedItemsIn(group))
					.build());
				}
				moveMenu.setVisible(getSelectedItem() instanceof ComponentInstanceConfig);

				MenuManager copyMenu = new MenuManager(Messages.get().ComponentsView_Menu_CopyTo);
				for (FunctionalGroup group : Controller.getInstance().getProfile().getFunctionalGroups()) {
					copyMenu.add(ActionAnalyticsBuilder.action(new Runnable() {
						@Override
						public void run() {
							// Copy
							boolean change = false;
							Iterator<? extends @Nullable IChild> selectedItemsIterator = selectionProvider.getSelection().iterator();
							while (selectedItemsIterator.hasNext()) {
								IChild selectedItem = selectedItemsIterator.next();
								if (selectedItem instanceof ComponentInstanceConfig) {
									ComponentInstanceConfig instanceConfig = (ComponentInstanceConfig) selectedItem;
									StoragePeriphsComponentInstance storageComponent = new StoragePeriphsComponentInstance(
											instanceConfig.getStorageComponent());
									StoragePeriphsFuncGroup storageFuncGroup = group.getStorageFuncGroup();
									Collection<@NonNull String> usedNames = controllerWrapper.getController().getUsedNamesInGroupsWithSamePrefix(group.getIdPrefix());
									String newName = storageComponent.getName();
									if (usedNames.contains(newName)) {
										newName = UtilsText.createUniqueName(newName + ComponentsView.SUFFIX_COPY, usedNames, false);
									}
									storageComponent.setName(newName);
									storageFuncGroup.addInstance(storageComponent);
									controllerWrapper.getController().reloadFromSharedConfig();
									if (group.getName().equals(controllerWrapper.getController().getFunctionalGroup().getName())) {
										// If copying to current functional group then open editor with new instance
										ComponentSettingView.open(viewSite, storageComponent.getType(), storageComponent.getName(), false, true);
									}
									change = true;
								}
							}
							if (change) {
								controllerWrapper.getController().handleSettingChange(EventTypes.CHANGE, ComponentMenuCreator.this, UtilsText.safeString(
										com.nxp.swtools.periphs.controller.Messages.get().Controller_Action_CopyComponentInstance));
							}
						}
					})
					.id(TestIDs.PERIPHS_ACTION_COPY_INSTANCE + UtilsText.UNDERSCORE + group.getUiName())
					.text(group.getUiName())
					.image(null)
					.enabled(true)
					.build());
				}
				copyMenu.setVisible(getSelectedItem() instanceof ComponentInstanceConfig);

				manager.add(ActionAnalyticsBuilder.action(new Runnable() {
					@Override
					public void run() {
						// display documentation
						final Collection<? extends @NonNull IChild> selection = selectionProvider.getSelection();
						assert !selection.isEmpty() : "Invalid selection - action supposed to be disabled"; //$NON-NLS-1$;
						final IChild selectedItem = (IChild) selection.toArray()[0];
						assert selectedItem != null;
						final ConfigurationComponentTypeId typeId = getTypeId(selectedItem);
						assert (typeId != null) : "Invalid selection - action supposed to be disabled"; //$NON-NLS-1$
						DocumentationView.open(viewSite, typeId.getTypeId(), true);
					}
				})
				.id(TestIDs.PERIPHS_ACTION_SHOW_DOCUMENTATION)
				.text(Messages.get().ComponentsView_ComponentPopUp_Documentation)
				.image(ToolsImages.getImageDescriptor(IToolsImages.ICON_DOCUMENTATION))
				.enabled(isSomethingSelected() && isDocumentedComponentSelected() && (!isSystemComponentSelected()))
				.build());

				manager.add(ActionAnalyticsBuilder.action(new Runnable() {
					@Override
					public void run() {
						// remove
						Collection<? extends @NonNull IChild> selection = selectionProvider.getSelection();
						if (!selection.isEmpty()) {
							LinkedHashSet<@NonNull ComponentConfig> componentsToRemove = new LinkedHashSet<>();
							List<@NonNull ComponentInstanceConfig> componentInstancesToRemove = new ArrayList<>();
							for (IChild selectedItem : selection) {
								// when component is selected
								if (selectedItem instanceof ComponentConfig) {
									ComponentConfig componentConfig = (ComponentConfig) selectedItem;
									if (!componentConfig.getId().equals(ConfigurationComponent.SYSTEM_COMPONENT_ID)) {
										componentsToRemove.add(componentConfig);
									}
								// when component instance is selected
								} else if (selectedItem instanceof ComponentInstanceConfig) {
									ComponentInstanceConfig componentInstance = (ComponentInstanceConfig) selectedItem;
									componentInstancesToRemove.add(componentInstance);
								}
							}
							Shell shell = viewSite.getShell();
							if ((shell != null) && (selection.size() > 1) && options.isAskWhatToRemove()) { // ask what to remove only when there are at least two instances
								assert (componentsToRemove.isEmpty())
									: "When the option 'askWhatToRemove' is set to 'true', it is not possible to remove component configs"; //$NON-NLS-1$
								RemoveComponentInstancesDialog.open(shell, componentInstancesToRemove);
							} else {
								boolean confirmed = (shell == null) ? true : RemoveDialog.open(shell, componentsToRemove, componentInstancesToRemove);
								if (confirmed) {
									RemoveDialog.remove(componentsToRemove, componentInstancesToRemove, this);
								}
							}
						}
					}
				})
				.id(TestIDs.PERIPHS_ACTION_REMOVE_COMPONENT)
				.text(Messages.get().ComponentsView_ComponentPopUp_Remove)
				.image(ToolsImages.getImageDescriptor(IToolsImages.ICON_REMOVE_CROSS_CIRCLE))
				.enabled((isSomethingSelected()) && (!isSystemComponentSelected()))
				.build());

				if (!isComponentInSelection()) {
					addEnableOrDisableInstancesAction(manager, isSomethingSelected());
				}

				manager.add(moveMenu);
				manager.add(copyMenu);
			}
		});
		return menuMgr;
	}

	/**
	 * Create context menu.
	 * @param viewer in which to create menu
	 * @param viewSite with the viewer
	 */
	public void createMenu(@NonNull Viewer viewer, @NonNull IViewSite viewSite) {
		final Control control = viewer.getControl();
		if (control != null) {
			createMenu(control, viewSite, viewer);
		}
	}

	/**
	 * Adds new action to given manager. New menu item contains "open" when only 1 instance is present, otherwise it contains cascade "open" item with children
	 * items that consist of component instances names.
	 * @param manager manager to add action to
	 * @param viewSite view site in which component instance view is opened
	 */
	void addOpenInstanceAction(@NonNull IMenuManager manager, @NonNull IViewSite viewSite) {
		MenuManager openMenu = new MenuManager(Messages.get().ComponentsView_Menu_Open);
		Collection<? extends @NonNull IChild> selection = selectionProvider.getSelection();
		Object[] selections = selection.toArray();
		int amountOfSelectedItems = selections.length;
		if (amountOfSelectedItems == 1) {
			Object selectedItem = selections[0];
			if (selectedItem instanceof ComponentInstanceConfig) {
				ComponentInstanceConfig instanceConfig = (ComponentInstanceConfig) selectedItem;
				String actionName = Messages.get().ComponentsView_Menu_Open;
				manager.add(ActionAnalyticsBuilder.action(() -> ComponentSettingView.open(viewSite, instanceConfig.getComponent().getId(), instanceConfig.getName(), false, true))
						.id(UtilsText.EMPTY_STRING)
						.text(actionName)
						.image(null)
						.enabled(true)
						.build()
						);
			}
		} else {
			for (IChild child : selection) {
				if (child instanceof ComponentInstanceConfig) {
					ComponentInstanceConfig instance = (ComponentInstanceConfig) child;
					String enabledMessage = instance.isEnabled()? UtilsText.EMPTY_STRING : UtilsText.LEFT_BRACKET + Messages.get().ConfigurationState_Disabled + UtilsText.RIGHT_BRACKET;
					String actionName = String.format("%s %s", instance.getUiName(), enabledMessage); //$NON-NLS-1$
					openMenu.add(
							ActionAnalyticsBuilder.action(() -> ComponentSettingView.open(viewSite, instance.getComponent().getId(), instance.getName(), false, true))
							.id(UtilsText.EMPTY_STRING)
							.text(actionName)
							.image(null)
							.enabled(true)
							.build()
							);
				}
			}
			manager.add(openMenu);
		}
	}

	/**
	 * Creates either "enable" or "disable" menu action for one instance or sub-menu with these options for each instance from selection
	 * @param manager the menu manager
	 * @param somethingSelected whether selection is not empty
	 */
	void addEnableOrDisableInstancesAction(IMenuManager manager, boolean somethingSelected) {
		Collection<? extends @NonNull IChild> selection = selectionProvider.getSelection();
		if (!selection.isEmpty()) {
			Object[] selections = selection.toArray();
			int amountOfSelectedItems = selections.length;
			if (amountOfSelectedItems == 1) {
				Object selectedItem = selections[0];
				if (selectedItem instanceof ComponentInstanceConfig) {
					ComponentInstanceConfig componentInstanceConfig = (ComponentInstanceConfig) selectedItem;
					String message = componentInstanceConfig.isEnabled() ? Messages.get().ComponentsView_Menu_EnabledState : Messages.get().ComponentsView_Menu_DisabledState;
					manager.add(ActionAnalyticsBuilder.action(new Runnable() {
						@Override
						public void run() {
							final ComponentInstanceConfig componentInstanceConfigLoc = componentInstanceConfig;
							final String actionLabel = componentInstanceConfigLoc.isEnabled() ? com.nxp.swtools.periphs.controller.Messages.get().Controller_Action_DisableComponentInstance
									: com.nxp.swtools.periphs.controller.Messages.get().Controller_Action_EnableComponentInstance;
							componentInstanceConfigLoc.setEnabled(!componentInstanceConfigLoc.isEnabled());
							controllerWrapper.getController().handleSettingChange(EventTypes.CHANGE, ComponentMenuCreator.this, UtilsText.safeString(actionLabel));
						}
					})
					.id(TestIDs.PERIPHS_ACTION_ENABLE_COMPONENT)
					.text(message)
					.image(null)
					.enabled(somethingSelected)
					.build());
				}
			} else {
				MenuManager enableDisableMenu = new MenuManager(Messages.get().ComponentsView_Menu_EnableSlashDisable);
				for (Object selectedItem : selections) {
					if (selectedItem instanceof ComponentInstanceConfig) {
						ComponentInstanceConfig componentInstanceConfig = (ComponentInstanceConfig) selectedItem;
						String message = (componentInstanceConfig.isEnabled() ? Messages.get().ComponentsView_Menu_EnabledState : Messages.get().ComponentsView_Menu_DisabledState)
								+ UtilsText.SPACE + componentInstanceConfig.getUiName();
						enableDisableMenu.add(ActionAnalyticsBuilder.action(new Runnable() {
							@Override
							public void run() {
								final ComponentInstanceConfig componentInstanceConfigLoc = componentInstanceConfig;
								final String actionLabel = componentInstanceConfigLoc.isEnabled() ? com.nxp.swtools.periphs.controller.Messages.get().Controller_Action_DisableComponentInstance
										: com.nxp.swtools.periphs.controller.Messages.get().Controller_Action_EnableComponentInstance;
								componentInstanceConfigLoc.setEnabled(!componentInstanceConfigLoc.isEnabled());
								controllerWrapper.getController().handleSettingChange(EventTypes.CHANGE, ComponentMenuCreator.this, UtilsText.safeString(actionLabel));
							}
						})
						.id(TestIDs.PERIPHS_ACTION_ENABLE_COMPONENT)
						.text(message)
						.image(null)
						.enabled(somethingSelected)
						.build());
					}
				}
				manager.add(enableDisableMenu);
			}
		}
	}

	/**
	 * Interface used for providing peripheral selection.
	 * @author Juraj Ondruska
	 */
	public interface IPeripheralSelectionProvider {
		/**
		 * @return current peripheral selection
		 */
		@Nullable String getSelection();
	}
	
	/**
	 * Interface used for providing component selection.
	 * @author Juraj Ondruska
	 */
	public interface IComponentSelectionProvider {
		/**
		 * @return current child selection
		 */
		@NonNull Collection<? extends @NonNull IChild> getSelection();
	}
	
	/**
	 * Class holding options used by the {@link ComponentMenuCreator}.
	 * @author Juraj Ondruska
	 */
	public static class ComponentMenuOptions {
		/** Whether to ask which children should be removed if more of them are selected */
		private final boolean askWhatToRemove;

		/**
		 * Constructor.
		 * @param askWhatToRemove whether to ask which children should be removed if more of them are selected
		 */
		public ComponentMenuOptions(boolean askWhatToRemove) {
			this.askWhatToRemove = askWhatToRemove;
		}

		/**
		 * @return whether to ask which children should be removed if more of them are selected
		 */
		public boolean isAskWhatToRemove() {
			return askWhatToRemove;
		}
	}
}