/**
 * Copyright 2017-2019 NXP
 */
package com.nxp.swtools.periphs.gui.view;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.IViewSite;
import org.osgi.framework.Version;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.ComparatorHelpers;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.common.utils.version.VersionUtils;
import com.nxp.swtools.kex.MfactConstants;
import com.nxp.swtools.periphs.controller.APeriphController;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.model.config.ComponentConfig;
import com.nxp.swtools.periphs.model.config.ComponentInstanceConfig;
import com.nxp.swtools.periphs.model.config.IChildProvidable;
import com.nxp.swtools.periphs.model.config.PeriphsProfile;
import com.nxp.swtools.periphs.model.data.ConfigurationComponent;
import com.nxp.swtools.periphs.model.data.ConfigurationComponentTypeId;
import com.nxp.swtools.periphs.model.data.SWComponent;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.provider.configuration.SharedConfigurationFactory;
import com.nxp.swtools.provider.toolchainproject.ISdkComponentInProject;
import com.nxp.swtools.provider.toolchainproject.IToolchainProjectWithSdk;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.progress.ProgressUtils;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsColors.SwToolsColors;
import com.nxp.swtools.utils.resources.ToolsImages;
import com.nxp.swtools.utils.storage.StorageHelper;

/**
 * Dialog used for creating new components.
 * @author Juraj Ondruska
 */
public class AddComponentDialog extends Dialog {
	/** Logger of the class */
	private static final @NonNull Logger LOGGER = LogManager.getLogger(AddComponentDialog.class);
	/** Associated view site */
	private final @NonNull IViewSite viewSite;
	/** The controller */
	final @NonNull IControllerWrapper controllerWrapper;
	/** Peripheral for which to create component */
	final @Nullable String peripheral;
	/** List of ConfigurationComponentTypeIds that were created by this dialog - used to be able to get result of the dialog */
	final @Nullable List<@NonNull ConfigurationComponentTypeId> configCompsTypeIds;
	/** Whether filtering of latest version components is enabled */
	boolean latestVersionFilter = true;
	/** Whether filtering tool-chain project components is enabled */
	boolean toolchainFilter = true;
	/** Whether the filtering tool-chain project components is allowed and the options is visible in the GUI */
	boolean toolchainFilterAllowed;
	/** Add button */
	@Nullable Button addButton;
	/** Reference to table viewer */
	private @Nullable TableViewer tableViewer;
	/** Whether to open view after component(s) creation */
	private final boolean openView;
	/** Functional group for which to add components */
	private final @NonNull String functionalGroup;
	/** Storage helper used for saving/restoring values to/from preferences */
	final @NonNull StorageHelper storageHelper = new StorageHelper(MfactConstants.TOOL_PERIPHERALS_ID);
	/** Result of the dialog - instance that was added by the dialog */
	private @Nullable IChildProvidable result;
	/** This number is used to compute the table's height; number of rows that should be visible by default */
	private static final int DEFAULT_ROWS_NUM = 12;
	/** This number is used to compute the table's width given its height */
	private static final float DEFAULT_VIEWER_WIDTH_HEIGHT_RATIO = 16f / 9f;
	/** Key to the toolchain filter preference in the preference storage */
	static final @NonNull String KEY_TOOLCHAIN_FILTER = AddComponentDialog.class.getName() + "toolchainFilter"; //$NON-NLS-1$
	/** Key to the latest versions filter preference in the preference storage */
	static final @NonNull String KEY_LATEST_VERSIONS_FILTER = AddComponentDialog.class.getName() + "latestVersionsFilter"; //$NON-NLS-1$


	/**
	 * Open a dialog.
	 * @param viewSite associated view site
	 * @param forPeripheral peripheral for which to add component
	 * @param functionalGroup for which to add components
	 * @param openView whether to open view after component(s) creation
	 * @param toolchainFilterAllowed whether the toolchain filter is allowed and should be visible
	 * @param configCompsTypeIds list of ConfigurationComponentTypeIds that should be visible in dialog
	 * @param controllerWrapper containing the generic controller
	 * @return the created dialog
	 */
	public static @NonNull AddComponentDialog open(@NonNull IViewSite viewSite, @Nullable String forPeripheral,
			@NonNull String functionalGroup, boolean openView, boolean toolchainFilterAllowed, @Nullable List<@NonNull ConfigurationComponentTypeId> configCompsTypeIds, @NonNull IControllerWrapper controllerWrapper) {
		AddComponentDialog dialog = new AddComponentDialog(viewSite, forPeripheral, functionalGroup, openView, toolchainFilterAllowed, configCompsTypeIds, controllerWrapper);
		dialog.setBlockOnOpen(true);
		dialog.updateAddButtonState();
		dialog.open();
		return dialog;
	}
	
	/**
	 * Open a dialog.
	 * @param viewSite associated view site
	 * @param forPeripheral peripheral for which to add component
	 * @param functionalGroup for which to add components
	 * @param openView whether to open view after component(s) creation
	 * @param toolchainFilterAllowed whether the toolchain filter is allowed and should be visible
	 * @param controllerWrapper containing the generic controller
	 * @return the created dialog
	 */
	public static @NonNull AddComponentDialog open(@NonNull IViewSite viewSite, @Nullable String forPeripheral,
			@NonNull String functionalGroup, boolean openView, boolean toolchainFilterAllowed, @NonNull IControllerWrapper controllerWrapper) {
		return open(viewSite, forPeripheral, functionalGroup, openView, toolchainFilterAllowed, null, controllerWrapper);
	}

	/**
	 * Enables/disables the "OK"/add button depending on the current selection
	 */
	protected void updateAddButtonState() {
		Button addButtonLoc = addButton;
		TableViewer tableViewerLoc = tableViewer;
		if ((addButtonLoc != null) && (tableViewerLoc != null)) {
			setButtonState(addButtonLoc, tableViewerLoc.getSelection());
		}
	}

	/**
	 * Open a dialog.
	 * @param viewSite associated view site
	 * @param forPeripheral peripheral for which to add component
	 * @param controllerWrapper containing the generic controller
	 * @return the created dialog
	 */
	public static @NonNull AddComponentDialog open(@NonNull IViewSite viewSite, @Nullable String forPeripheral, @NonNull IControllerWrapper controllerWrapper) {
		return open(viewSite, forPeripheral, Controller.getInstance().getFunctionalGroup().getName(), true, true, controllerWrapper);
	}
	
	/**
	 * Open a dialog.
	 * @param viewSite associated view site
	 * @param configCompsTypeIds list of ConfigurationComponentTypeIds that should be visible in dialog
	 * @param controllerWrapper containing the generic controller
	 * @return the created dialog
	 */
	public static @NonNull AddComponentDialog openForComponents(@NonNull IViewSite viewSite, @Nullable List<@NonNull ConfigurationComponentTypeId> configCompsTypeIds, @NonNull IControllerWrapper controllerWrapper) {
		return open(viewSite, null, Controller.getInstance().getFunctionalGroup().getName(), true, true, configCompsTypeIds, controllerWrapper);
	}

	/**
	 * Constructor.
	 * @param viewSite associated view site
	 * @param forPeripheral peripheral for which to add component
	 * @param functionalGroup for which to add components
	 * @param openView whether to open view after component(s) creation
	 * @param toolchainFilterAllowed whether the toolchain filter is allowed and should be displayed
	 * @param configCompsTypeIds list of ConfigurationComponentTypeIds that should be visible in dialog
	 * @param controllerWrapper containing the generic controller
	 */
	private AddComponentDialog(@NonNull IViewSite viewSite, @Nullable String forPeripheral,
			@NonNull String functionalGroup, boolean openView, boolean toolchainFilterAllowed, 
			@Nullable List<@NonNull ConfigurationComponentTypeId> configCompsTypeIds, @NonNull IControllerWrapper controllerWrapper) {
		super(viewSite.getWorkbenchWindow().getShell());
		this.latestVersionFilter = storageHelper.loadBool(KEY_LATEST_VERSIONS_FILTER, this.latestVersionFilter);
		this.toolchainFilterAllowed = toolchainFilterAllowed;
		if (toolchainFilterAllowed) {
			this.toolchainFilter = storageHelper.loadBool(KEY_TOOLCHAIN_FILTER, this.toolchainFilter);
		} else {
			this.toolchainFilter = false;
		}
		this.viewSite = viewSite;
		this.peripheral = forPeripheral;
		this.configCompsTypeIds = configCompsTypeIds;
		this.functionalGroup = functionalGroup;
		this.openView = openView;
		this.result = null;
		this.controllerWrapper = controllerWrapper;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#isResizable()
	 */
	@Override
	protected boolean isResizable() {
		return true;
	}

	/**
	 * Remove configuration components that are not active or worse match for current configuration.
	 * @param configComps Collection of configuration components which are being modified
	 * @param latestVersionOnly Whether to choose only the latest version of each component (set to {@code true}
	 * if choose latest, {@code false} otherwise)
	 * @param toolchainProject Toolchain project for the configuration
	 * @return new collection of configuration components
	 */
	private @NonNull Collection<@NonNull ConfigurationComponentTypeId> filterComponents(
			@NonNull Collection<@NonNull ConfigurationComponentTypeId> configComps,
			boolean latestVersionOnly,
			@Nullable IToolchainProjectWithSdk toolchainProject) {
		Map<@NonNull String, @NonNull List<@NonNull ConfigurationComponentTypeId>> compsByType = configComps.stream().collect(
				Collectors.groupingBy(ConfigurationComponentTypeId::getType, Collectors.toCollection(ArrayList::new)));
		List<@NonNull ConfigurationComponentTypeId> components = new ArrayList<>();
		for (Map.Entry<@NonNull String, @NonNull List<@NonNull ConfigurationComponentTypeId>> entry : compsByType.entrySet()) {
			assert entry.getKey() != null;
			String type = UtilsText.safeString(entry.getKey()); // using {@link UtilsText#safeString(String)} only to avoid dealing with null warning
			ConfigurationComponentTypeId confCompTypeId;
			// 1. add mandatory components from the project
			collectMandatoryComponents(components, entry.getValue());
			if (!latestVersionOnly) {
				// 2. use the same component already in the configuration if possible
				confCompTypeId = controllerWrapper.getController().getProfile().getActiveComponents().getComponentTypeIdByType(type);
				// 3. use the same component in the project if possible
				if ((confCompTypeId == null) && (toolchainProject != null)) {
					confCompTypeId = findComponentInProject(
							(List<@NonNull ConfigurationComponentTypeId>) CollectionsUtils.safeCollection(entry.getValue()),
							toolchainProject);
				}
				// 4. use the best matching component available
				if (confCompTypeId == null) {
					confCompTypeId = controllerWrapper.getController().getMcu().getAvailableComponents().getBestMatchingConfigCompType(entry.getValue());
				}
			} else {
				// 2. use the latest version of the component available
				confCompTypeId = getLatestVersionConfigCompType(entry.getValue());
			}
			if (confCompTypeId != null) {
				if (!components.contains(confCompTypeId)) {
					components.add(confCompTypeId);
				}
			} else {
				LOGGER.severe(String.format("Could not find any matching component for type %1s", type)); //$NON-NLS-1$
			}
		}
		return components;
	}

	/**
	 * Collect mandatory (not referencing a SW Component and/or marked global-only) configuration components.
	 * @param mandatoryComps collection to receive mandatory components
	 * @param configComps configuration components of the same type ({@link ConfigurationComponentTypeId#getType()})
	 */
	public static void collectMandatoryComponents(
			@NonNull List<@NonNull ConfigurationComponentTypeId> mandatoryComps,
			@NonNull List<@NonNull ConfigurationComponentTypeId> configComps) {
		for (ConfigurationComponentTypeId configComp : configComps) {
			if (isMandatoryComponent(configComp) && !mandatoryComps.contains(configComp)) {
				mandatoryComps.add(configComp);
			}
		}
	}

	/**
	 * Finds componentTypeId with the latest SW component version from the list of componentTypeIds of the same type.
	 * @param configComps Components of the same type which are being looked for in configured components
	 * @return if possible, returns componentTypeId with the latest SW component version. If not found, returns null.
	 */
	public static @Nullable ConfigurationComponentTypeId getLatestVersionConfigCompType(@NonNull List<@NonNull ConfigurationComponentTypeId> configComps) {
		if (configComps.isEmpty()) {
			return null;
		}
		ConfigurationComponentTypeId latestComp = configComps.get(0);
		Version highestVersion = getHighestComponentVersion(latestComp);
		// should work for now, since only 1 SW component is expected (and multiple SW components are not supported);
		// once multiple SW components allowed to be referenced, will need to determine the main SW component for all configComps
		for (ConfigurationComponentTypeId comp : configComps) {
			Version otherVersion = getHighestComponentVersion(comp);
			if (otherVersion != null) {
				if ((highestVersion == null) || (otherVersion.compareTo(highestVersion) > 0)) {
					latestComp = comp;
					highestVersion = otherVersion;
				}
			}
		}
		if ((highestVersion == null) && (configComps.size() > 1)) {
			return null;
		}
		return latestComp;
	}

	/**
	 * Retrieves the highest SW component version of componentTypeId
	 * @param componentTypeId instance
	 * @return SW component version or {@code null} if no SW components encountered
	 */
	private static @Nullable Version getHighestComponentVersion(@NonNull ConfigurationComponentTypeId componentTypeId) {
		List<@NonNull SWComponent> swComps = componentTypeId.getConfigurationComponent().getComponents();
		if (swComps.isEmpty()) {
			return null;
		}
		Version highestVersion = swComps.get(0).getVersion();
		// should work for now, since only 1 SW component is expected (and multiple SW components are not supported);
		// once multiple SW components allowed to be referenced, will need to determine the main SW component and its version
		for (SWComponent swComp : swComps) {
			Version otherVersion = swComp.getVersion();
			if (otherVersion != null) {
				if ((highestVersion == null) || (otherVersion.compareTo(highestVersion) > 0)) {
					highestVersion = otherVersion;
				}
			}
		}
		return highestVersion;
	}

	/**
	 * Search for the configuration component among the specified ones referencing an equal SW component in the toolchain project
	 * @param configComps configuration components of the same type ({@link ConfigurationComponentTypeId#getType()})
	 * @param toolchainProject Toolchain project for the configuration
	 * @return configuration component info or {@code null}
	 */
	private static @Nullable ConfigurationComponentTypeId findComponentInProject(
			@NonNull List<@NonNull ConfigurationComponentTypeId> configComps,
			@NonNull IToolchainProjectWithSdk toolchainProject) {
		for (ConfigurationComponentTypeId configComp: configComps) {
			if (isMandatoryComponent(configComp)) {
				// mandatory components are processed elsewhere
				continue;
			}
			List<@NonNull SWComponent> swComps = configComp.getConfigurationComponent().getComponents();
			if (swComps.size() > 1) {
				LOGGER.severe(String.format("Encountered configuration component containing multiple SW component references: %1s", //$NON-NLS-1$
						configComp.getConfigurationComponent()));
 				assert swComps.size() == 1;
			}
			ISdkComponentInProject prjComp = toolchainProject.getSdkComponent(swComps.get(0).getName());
			if (prjComp != null) {
				if (swComps.contains(new SWComponent(prjComp.getID(), prjComp.getVersion()))) {
					return configComp;
				}
			}
		}
		return null;
	}

	/**
	 * Determines whether the component must be offered always. 
	 * @param configComp configuration component
	 * @return {@code true} if mandatory, {@code false} otherwise
	 */
	private static boolean isMandatoryComponent(@NonNull ConfigurationComponentTypeId configComp) {
		return configComp.getConfigurationComponent().getComponents().isEmpty();
	}

	@Override
	protected Control createDialogArea(Composite parent) {
		Composite dialogComposite = (Composite) super.createDialogArea(parent);
		dialogComposite.setLayout(new GridLayout());
		Composite optionsComposite = new Composite(dialogComposite, dialogComposite.getStyle());
		optionsComposite.setLayout(new GridLayout(2, true));  // 2 is SWT.CHECK button count (onlyPrjCompsCheckBox and allCompsCheckBox) 
		dialogComposite.getShell().setText(Messages.get().ComponentsView_SelectComponentDialog_Title);
		final Button onlyPrjCompsCheckBox;
		if (toolchainFilterAllowed) {
			onlyPrjCompsCheckBox = new Button(optionsComposite, SWT.CHECK);
			SWTFactoryProxy.INSTANCE.setTestId(onlyPrjCompsCheckBox, TestIDs.PERIPHS_ADD_COMPONENT_SHELL_PROJECT_CHECKBOX);
			onlyPrjCompsCheckBox.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, false, false));
			onlyPrjCompsCheckBox.setText(Messages.get().AddComponentDialog_Filter_ShowOnlyCompsFromToolchainProject);
		} else {
			onlyPrjCompsCheckBox = null;
		}
		final Button onlyLatestCompsCheckBox = new Button(optionsComposite, SWT.CHECK);
		SWTFactoryProxy.INSTANCE.setTestId(onlyLatestCompsCheckBox, TestIDs.PERIPHS_ADD_COMPONENT_SHELL_LATEST_CHECKBOX);
		onlyLatestCompsCheckBox.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, false, false));
		onlyLatestCompsCheckBox.setText(Messages.get().AddComponentDialog_Filter_ShowOnlyLatestComps);
		// create list viewer
		final TableViewer viewer = new TableViewer(dialogComposite, SWT.FULL_SELECTION | SWT.SINGLE | SWT.V_SCROLL | SWT.H_SCROLL | SWT.BORDER);
		tableViewer = viewer;
		Table table = viewer.getTable();
		SWTFactoryProxy.INSTANCE.enableHtmlTooltipFor(viewer);
		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		SWTFactoryProxy.INSTANCE.setTestId(table, TestIDs.PERIPHS_ADD_COMPONENT_SHELL_LIST);
		viewer.setContentProvider(ArrayContentProvider.getInstance());
		viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			@Override
			public void selectionChanged(SelectionChangedEvent e) {
				updateAddButtonState();
			}
		});
		createTableColumns(viewer);
		final String peripheralInstance = peripheral;
		Collection<@NonNull ConfigurationComponentTypeId> components;
		// get components based on component IDs given on dialog creation
		if (configCompsTypeIds != null) {
			components = configCompsTypeIds;
		} else {
			// get components based on peripheral
			components = peripheralInstance != null ? controllerWrapper.getController().getComponentsOfPeripheral(peripheralInstance) : controllerWrapper.getController().getComponents();
		}
		// remove component if its "hidden" flag is set to true and it is not used in project
		filterNonDisplayableComponents(components, controllerWrapper.getController().getProfile());
		IToolchainProjectWithSdk toolchainProject = SharedConfigurationFactory.getSharedConfigurationSingleton().getToolchainProject();
		final Collection<@NonNull ConfigurationComponentTypeId> latestComps = filterComponents(components, true, toolchainProject);
		final Collection<@NonNull ConfigurationComponentTypeId> filteredComps = filterComponents(components, false, toolchainProject);
		if (onlyPrjCompsCheckBox != null) {
			onlyPrjCompsCheckBox.addSelectionListener(new SelectionAdapter() {
				/* (non-Javadoc)
				 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
				 */
				@Override
				public void widgetSelected(SelectionEvent e) {
					toolchainFilter = onlyPrjCompsCheckBox.getSelection();
					storageHelper.saveBool(KEY_TOOLCHAIN_FILTER, toolchainFilter);
					setViewerInputComps(viewer, components, latestComps, filteredComps);
					onlyLatestCompsCheckBox.setEnabled(!onlyPrjCompsCheckBox.getSelection());
				}
			});
		}
		onlyLatestCompsCheckBox.addSelectionListener(new SelectionAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			@Override
			public void widgetSelected(SelectionEvent e) {
				latestVersionFilter = onlyLatestCompsCheckBox.getSelection();
				storageHelper.saveBool(KEY_LATEST_VERSIONS_FILTER, latestVersionFilter);
				setViewerInputComps(viewer, components, latestComps, filteredComps);
			}
		});
		if ((toolchainProject == null) || !toolchainProject.wasProjectDetected()) {
			toolchainFilter = false;
			if (onlyPrjCompsCheckBox != null) {
				onlyPrjCompsCheckBox.setEnabled(false);
			}
		}
		if (onlyPrjCompsCheckBox != null) {
			onlyPrjCompsCheckBox.setSelection(toolchainFilter);
		}
		onlyLatestCompsCheckBox.setEnabled(!toolchainFilter);
		onlyLatestCompsCheckBox.setSelection(latestVersionFilter);
		viewer.addFilter(new ViewerFilter() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
			 */
			@Override
			public boolean select(Viewer viewerForSelect, Object parentElement, Object element) {
				ConfigurationComponentTypeId component = (ConfigurationComponentTypeId) element;
				// filter components which are not in the current tool-chain project
				if (toolchainFilter && (toolchainProject != null)) {
					for (SWComponent swComponent : component.getConfigurationComponent().getComponents()) {
						ISdkComponentInProject sdkComponent = toolchainProject.getSdkComponent(swComponent.getName());
						if (sdkComponent == null) {
							return false;
						}
					}
				}
				// filter out global-only components which are configured already
				ComponentConfig configuredComponent = controllerWrapper.getController().getConfiguredComponent(component.getConfigurationComponent().getId());
				if (configuredComponent != null) {
					return !configuredComponent.getComponent().isGlobalOnly();
				}
				return true;
			}
		});
		viewer.setComparator(new ViewerComparator() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ViewerComparator#compare(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
			 */
			@Override
			public int compare(Viewer viewerLoc, Object e1, Object e2) {
				if ((e1 instanceof ConfigurationComponentTypeId) && (e2 instanceof ConfigurationComponentTypeId)) {
					ConfigurationComponentTypeId typeId1 = (ConfigurationComponentTypeId) e1;
					ConfigurationComponentTypeId typeId2 = (ConfigurationComponentTypeId) e2;
					String name1Lower = UtilsText.safeString(typeId1.getConfigurationComponent().getUIName(controllerWrapper.getController().getFunctionalGroup().getExpressionContext()).toLowerCase());
					String name2Lower = UtilsText.safeString(typeId2.getConfigurationComponent().getUIName(controllerWrapper.getController().getFunctionalGroup().getExpressionContext()).toLowerCase());
					int comparisonResult = ComparatorHelpers.compareSignalNames(name1Lower, name2Lower);
					if ((comparisonResult == 0)
							&& (typeId1.getConfigurationComponent().getComponents().size() == 1)
							&& (typeId2.getConfigurationComponent().getComponents().size() == 1)) {
						Version ver1 = typeId1.getConfigurationComponent().getComponents().get(0).getVersion();
						Version ver2 = typeId2.getConfigurationComponent().getComponents().get(0).getVersion();
						if ((ver1 != null) && (ver2 != null)) {
							return ver1.compareTo(ver2);
						}
					}
					return comparisonResult;
				}
				return 0;
			}

		});
		setViewerInputComps(viewer, components, latestComps, filteredComps);
		GridData viewerLayoutData = new GridData(SWT.FILL, SWT.FILL, true, true);
		viewerLayoutData.heightHint = table.getItemHeight() * DEFAULT_ROWS_NUM;
		viewerLayoutData.widthHint = (int) (viewerLayoutData.heightHint * DEFAULT_VIEWER_WIDTH_HEIGHT_RATIO);
		viewer.getControl().setLayoutData(viewerLayoutData);
		if (table.getItemCount() > 0) {
			table.select(0);
		}
		viewer.addDoubleClickListener(new IDoubleClickListener() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.IDoubleClickListener#doubleClick(org.eclipse.jface.viewers.DoubleClickEvent)
			 */
			@Override
			public void doubleClick(DoubleClickEvent event) {
				StructuredSelection selection = (StructuredSelection) event.getSelection();
				Object data = selection.getFirstElement();
				assert (data instanceof ConfigurationComponentTypeId);
				if (canAddComponent((ConfigurationComponentTypeId) data)) {
					createComponentsFromSelection(selection, viewer, controllerWrapper);
					close();
				}
			}
		});
		packColumns();
		return dialogComposite;
	}

	/**
	 * Removes ConfigurationComponentTypeIds from list that should not be visible in UI.
	 * @param components list of ConfigurationComponentTypeId to filter out
	 * @param profile current profile
	 */
	public static void filterNonDisplayableComponents(@NonNull Collection<@NonNull ConfigurationComponentTypeId> components, @NonNull PeriphsProfile profile) {
		components.removeIf(c -> (c.getConfigurationComponent().isHidden()) && (!profile.getActiveComponents().getConfigCompTypeIds().contains(c)));
		components.removeIf(c -> (c.getConfigurationComponent().getId().equals(ConfigurationComponent.SYSTEM_COMPONENT_ID)));
	}

	/**
	 * Changes input for the specified viewer to the given components depending on the internal filter state.
	 * @param viewer component list viewer
	 * @param unfilteredComps an original collection of components
	 * @param latestComps a collection of filtered out latest components from <i>unfilteredComps</i> components
	 * @param filteredComps a collection of filtered out equivalent/compatible components from <i>unfilteredComps</i> components
	 */
	protected void setViewerInputComps(@NonNull TableViewer viewer, @NonNull Collection<@NonNull ConfigurationComponentTypeId> unfilteredComps, @NonNull Collection<@NonNull ConfigurationComponentTypeId> latestComps, @NonNull Collection<@NonNull ConfigurationComponentTypeId> filteredComps) {
		if (toolchainFilter) {
			viewer.setInput(filteredComps);
		} else {
			if (latestVersionFilter) {
				viewer.setInput(latestComps);
			} else {
				viewer.setInput(unfilteredComps);
			}
		}
	}
	
	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		addButton = createButton(parent, IDialogConstants.OK_ID, Messages.get().ComponentsView_SelectComponentDialog_OK, true);
		SWTFactoryProxy.INSTANCE.setTestId(addButton, TestIDs.PERIPHS_ADD_COMPONENT_SHELL_OK_BUTTON);
		createButton(parent, IDialogConstants.CANCEL_ID, Messages.get().ComponentsView_RemoveComponentDialog_Cancel, false);
	}

	@Override
	protected void okPressed() {
		if (tableViewer != null) {
			ISelection selection = tableViewer.getSelection();
			createComponentsFromSelection(selection, AddComponentDialog.this, controllerWrapper);
		}
		super.okPressed();
	}

	@Override
	protected void cancelPressed() {
		close();
	}
	
	/**
	 * Get error text for a configuration component.
	 * @param configComp to get error of
	 * @param controllerWrapper containing the generic controller
	 * @param peripheral name
	 * @return error or {@code null} in case of no error
	 */
	public static @Nullable String getError(@NonNull ConfigurationComponentTypeId configComp, @NonNull IControllerWrapper controllerWrapper, @Nullable String peripheral) {
		APeriphController controller = controllerWrapper.getController();
		String peripheralLoc = peripheral;
		if ((peripheralLoc != null) && !controller.getMcu().isPeripheralAvailableForCore(controller.getFunctionalGroup().getCore(), peripheralLoc)) {
			return MessageFormat.format(UtilsText.safeString(Messages.get().AddComponentDialog_ComponentPeripheralUnavailableForCore_Tooltip), peripheralLoc, controller.getFunctionalGroup().getCore());
		}
		Collection<@NonNull String> periphs = controller.getAvailablePeripherals(controller.getFunctionalGroup(), configComp.getConfigurationComponent());
		if (!periphs.isEmpty()) {
			Collection<@NonNull String> corePeriphs = controller.filterPeripheralsForCore(controller.getFunctionalGroup().getCore(), periphs);
			if (corePeriphs.isEmpty()) {
				return MessageFormat.format(UtilsText.safeString(Messages.get().AddComponentDialog_ComponentPeripheralUnavailableForCore_Tooltip), String.join(UtilsText.COMMA_SPACE, periphs), controller.getFunctionalGroup().getCore());
			}
		}
		return null;
	}

	/**
	 * Get warning text for a configuration component.
	 * @param configComp to get warning of
	 * @param controllerWrapper containing the generic controller
	 * @return warning or {@code null} in case of no warning
	 */
	static @Nullable String getWarning(@NonNull ConfigurationComponentTypeId configComp, @NonNull IControllerWrapper controllerWrapper) {
		APeriphController controller = controllerWrapper.getController();
		StringJoiner resultJoiner = new StringJoiner(UtilsText.LF);
		IToolchainProjectWithSdk toolchainProject = SharedConfigurationFactory.getSharedConfigurationSingleton().getToolchainProject();
		if ((toolchainProject != null) && toolchainProject.wasProjectDetected()) {
			for (SWComponent swComponent : configComp.getConfigurationComponent().getComponents()) {
				ISdkComponentInProject sdkComponent = toolchainProject.getSdkComponent(swComponent.getName());
				if (sdkComponent != null) {
					Version componentVersion = new Version(swComponent.getVersionStr());
					Version sdkComponentVersion = sdkComponent.getVersion();
					if (!VersionUtils.equivalent(sdkComponentVersion, componentVersion)) {
						resultJoiner.add(MessageFormat.format(
								UtilsText.safeString(com.nxp.swtools.periphs.controller.Messages.get().Controller_ProblemComponentVersionMismatchWithVersions),
								controller.getMcu().getAvailableComponents().getSdkComponentName(swComponent.getName()),
								componentVersion,
								sdkComponentVersion));
					}
				} else {
					resultJoiner.add(MessageFormat.format(
							UtilsText.safeString(com.nxp.swtools.periphs.controller.Messages.get().Controller_ProblemComponentMissing),
							controller.getMcu().getAvailableComponents().getSdkComponentName(swComponent.getName())));
				}
			}
		}
		return resultJoiner.length() == 0 ? null : resultJoiner.toString();
	}
	
	/**
	 * Get info text for a configuration component.
	 * @param configComp to get info of
	 * @param controllerWrapper containing the generic controller
	 * @param functionalGroup name
	 * @return info or {@code null} in case of no info
	 */
	static @Nullable String getInfo(@NonNull ConfigurationComponentTypeId configComp, @NonNull IControllerWrapper controllerWrapper, @NonNull String functionalGroup) {
		APeriphController controller = controllerWrapper.getController();
		ConfigurationComponent configurationComponent = configComp.getConfigurationComponent();
		if (controller.hasModeWithMasterPeripheralNotDefined(configurationComponent, functionalGroup)) {
			return null;
		}
		boolean isPeripheralAvailable = controller.isAnyPeripheralAvailable(functionalGroup, configurationComponent);
		if (!isPeripheralAvailable) {
			Collection<@NonNull String> peripherals = controller.getUsablePeripherals(controller.getFunctionalGroup(), configurationComponent);
			return MessageFormat.format(UtilsText.safeString(Messages.get().AddComponentDialog_NoAvailablePeripherals_Tooltip), peripherals);
		}
		return null;
	}

	/**
	 * Get problem message for the configuration component.
	 * @param configComp to get message for
	 * @param controllerWrapper containing the generic controller
	 * @param functionalGroup name
	 * @param peripheral name
	 * @return message or {@code null} in case of no problem(s)
	 */
	static @Nullable String getProblemMessage(@NonNull ConfigurationComponentTypeId configComp, @NonNull IControllerWrapper controllerWrapper, @NonNull String functionalGroup, @Nullable String peripheral) {
		String tooltipText = getError(configComp, controllerWrapper, peripheral);
		if (tooltipText == null) {
			tooltipText = getWarning(configComp, controllerWrapper);
		}
		if (tooltipText == null) {
			tooltipText = getInfo(configComp, controllerWrapper, functionalGroup);
		}
		return tooltipText;
	}

	/**
	 * Create columns of the table.
	 * @param viewer to create columns in
	 */
	private void createTableColumns(@NonNull TableViewer viewer) {
		TableViewerColumn configCompcolumn = new TableViewerColumn(viewer, SWT.NONE);
		String localFunctionalGroup = functionalGroup;
		String localPeripheral = peripheral;
		configCompcolumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				assert (element instanceof ConfigurationComponentTypeId);
				return ((ConfigurationComponentTypeId) element).getConfigurationComponent().getLabel(controllerWrapper.getController().getFunctionalGroup().getExpressionContext());
			}
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getForeground(java.lang.Object)
			 */
			@Override
			public Color getForeground(Object element) {
				assert (element instanceof ConfigurationComponentTypeId);
				ConfigurationComponentTypeId configComp = (ConfigurationComponentTypeId) element;
				if (canAddComponent(configComp)) {
					return null;
				}
				return SwToolsColors.DISABLED_FG.getColor();
			}

			/**
			 * Get error text for a configuration component.
			 * @param configComp to get error of
			 * @return error or {@code null} in case of no error
			 */
			private @Nullable String getError(@NonNull ConfigurationComponentTypeId configComp) {
				return AddComponentDialog.getError(configComp, controllerWrapper, localPeripheral);
			}

			/**
			 * Get warning text for a configuration component.
			 * @param configComp to get warning of
			 * @return warning or {@code null} in case of no warning
			 */
			private @Nullable String getWarning(@NonNull ConfigurationComponentTypeId configComp) {
				return AddComponentDialog.getWarning(configComp, controllerWrapper);
			}
			
			/**
			 * Get info text for a configuration component.
			 * @param configComp to get info of
			 * @return info or {@code null} in case of no info
			 */
			private @Nullable String getInfo(@NonNull ConfigurationComponentTypeId configComp) {
				return AddComponentDialog.getInfo(configComp, controllerWrapper, localFunctionalGroup);
			}
			
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getImage(java.lang.Object)
			 */
			@Override
			public Image getImage(Object element) {
				assert (element instanceof ConfigurationComponentTypeId);
				ConfigurationComponentTypeId configComp = (ConfigurationComponentTypeId) element;
				if (getError(configComp) != null) {
					return ToolsImages.getStatusDecoratorImg(ErrorLevels.LEVEL_ERROR);
				} else if (getWarning(configComp) != null) {
					return ToolsImages.getStatusDecoratorImg(ErrorLevels.LEVEL_WARNING);
				} else if (getInfo(configComp) != null) {
					return ToolsImages.getStatusDecoratorImg(ErrorLevels.LEVEL_INFORMATION);
				}
				return ToolsImages.getImage(IToolsImages.ICON_EMPTY_7x8P);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.CellLabelProvider#getToolTipText(java.lang.Object)
			 */
			@Override
			public String getToolTipText(Object element) {
				assert (element instanceof ConfigurationComponentTypeId);
				ConfigurationComponentTypeId configComp = (ConfigurationComponentTypeId) element;
				return AddComponentDialog.getProblemMessage(configComp, controllerWrapper, localFunctionalGroup, peripheral);
			}
		});
		TableViewerColumn categoryColumn = new TableViewerColumn(viewer, SWT.NONE);
		categoryColumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				assert (element instanceof ConfigurationComponentTypeId);
				ConfigurationComponentTypeId configComp = (ConfigurationComponentTypeId) element;
				return configComp.getConfigurationComponent().getCategory();
			}
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getForeground(java.lang.Object)
			 */
			@Override
			public Color getForeground(Object element) {
				assert (element instanceof ConfigurationComponentTypeId);
				ConfigurationComponentTypeId configComp = (ConfigurationComponentTypeId) element;
				if (canAddComponent(configComp)) {
					return null;
				}
				return SwToolsColors.DISABLED_FG.getColor();
			}
		});
		TableViewerColumn requiredSdkCompColumn = new TableViewerColumn(viewer, SWT.NONE);
		requiredSdkCompColumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				assert (element instanceof ConfigurationComponentTypeId);
				ConfigurationComponentTypeId configComp = (ConfigurationComponentTypeId) element;
				return configComp.getConfigurationComponent().getComponents().stream()
						.map(c -> c.getName() + " [" + c.getVersion() + "]") //$NON-NLS-1$ //$NON-NLS-2$
						.collect(CollectorsUtils.joining(UtilsText.COMMA_SPACE));
			}
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getForeground(java.lang.Object)
			 */
			@Override
			public Color getForeground(Object element) {
				assert (element instanceof ConfigurationComponentTypeId);
				ConfigurationComponentTypeId configComp = (ConfigurationComponentTypeId) element;
				if (canAddComponent(configComp)) {
					return null;
				}
				return SwToolsColors.DISABLED_FG.getColor();
			}
		});
		configCompcolumn.getColumn().setText(Messages.get().AddComponentDialog_Column_Name);
		categoryColumn.getColumn().setText(Messages.get().AddComponentDialog_Column_Category);
		requiredSdkCompColumn.getColumn().setText(Messages.get().AddComponentDialog_Column_Version);
	}
	
	/**
	 * Pack the columns. This method packs the columns' widths based on their content (omitting their titles)
	 */
	private void packColumns() {
		TableViewer tableViewerLoc = tableViewer;
		if (tableViewerLoc != null) {
			for (TableColumn col : tableViewerLoc.getTable().getColumns()) {
				col.pack();
			}
		}
	}

	/**
	 * Check whether the component can be added into the profile.
	 * @param configComp to check for
	 * @return {@code true} if the component can be added into profile, {@code false} otherwise
	 */
	static boolean canAddComponent(@NonNull ConfigurationComponentTypeId configComp) {
		String compType = configComp.getConfigurationComponent().getId();
		String configuredComponentTypeId = Controller.getInstance().getProfile().getConfiguredComponentTypeId(compType);
		return (configuredComponentTypeId == null) || configuredComponentTypeId.equals(configComp.getTypeId());
	}

	/**
	 * Create component(s) from the {@link ISelection}.
	 * @param selection the selected component types to create
	 * @param caller the method caller which will be used as an originator of a subsequent UI event.
	 * @param controllerWrapperLoc containing the generic controller
	 */
	protected void createComponentsFromSelection(@Nullable ISelection selection, @NonNull Object caller, @NonNull IControllerWrapper controllerWrapperLoc) {
		if (selection instanceof IStructuredSelection) {
			IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			Collection<@NonNull ConfigurationComponentTypeId> comps = CollectionsUtils.getInstancesOf(
					(List<?>) structuredSelection.toList(),
					ConfigurationComponentTypeId.class
					).collect(Collectors.toList());
			assert (comps != null);
			IChildProvidable createdComponent = createComponents(comps, peripheral, viewSite, functionalGroup, openView, caller, controllerWrapperLoc);
			setResult(createdComponent);
		}
	}
	
	/**
	 * Create given component(s).
	 * @param components to create
	 * @param peripheral for which to create the components or {@code null} for the first available
	 * @param viewSite associated view site for which to create component setting view(s)
	 * @param functionalGroup for which to add components
	 * @param openView whether to open view after component(s) creation
	 * @param caller originator of the UI event(s) invoked by this method
	 * @param controllerWrapper containing the generic controller
	 * @return the last component or component instance that was created or {@code null} if there wasn't anything new created.
	 */
	public static @Nullable IChildProvidable createComponents(@NonNull Collection<@NonNull ConfigurationComponentTypeId> components,
			@Nullable String peripheral, @NonNull IViewSite viewSite, @NonNull String functionalGroup, boolean openView,
			@NonNull Object caller, @NonNull IControllerWrapper controllerWrapper) {
		Controller controller = Controller.getInstance();
		IChildProvidable result = null;
		for (ConfigurationComponentTypeId selectedConfigComp : components) {
			String componentType = selectedConfigComp.getConfigurationComponent().getId();
			if (selectedConfigComp.getConfigurationComponent().isGlobalOnly()) {
				result = ProgressUtils.run(() -> controller.createOrGetComponent(selectedConfigComp.getTypeId(), caller));
				if (result != null) {
					if (openView) {
						controllerWrapper.getGUIController().openConfigurationSettingsView(viewSite, componentType, componentType, true, true);
					}
				} else {
					LOGGER.severe("Component configuration of type: " + componentType + " could not be created"); //$NON-NLS-1$ //$NON-NLS-2$
				}
			} else {
				if (controller.isInstancesLimitReached(functionalGroup)) {
					MessageDialog.openError(viewSite.getShell(), Messages.get().AddComponentDialog_LimitReached_Title,
							Messages.get().AddComponentDialog_LimitReached_Message);
				} else {
					ComponentInstanceConfig componentInstanceConfig = ProgressUtils.run(() -> controller.createComponentInstance(selectedConfigComp.getTypeId(),
							peripheral, functionalGroup, caller));
					result = componentInstanceConfig;
					if (componentInstanceConfig != null) {
						if (openView) {
							controllerWrapper.getGUIController().openConfigurationSettingsView(viewSite, componentInstanceConfig.getComponent().getId(), componentInstanceConfig.getName(), false, true);
						}
					} else {
						LOGGER.severe("Component instance of type: " + componentType + " could not be created"); //$NON-NLS-1$ //$NON-NLS-2$
					}
				}
			}
		}
		return result;
	}

	/**
	 * Set state of the button based on current selection
	 * @param button button to configure it's state
	 * @param iSelection selected item in a viewer
	 */
	static void setButtonState(@NonNull Button button, @Nullable ISelection iSelection) {
		if (iSelection == null) {
			return;
		}
		StructuredSelection selection = (StructuredSelection) iSelection;
		boolean setActive = !selection.isEmpty();
		for (Object data : selection.toList()) {
			assert (data instanceof ConfigurationComponentTypeId);
			ConfigurationComponentTypeId configComp = (ConfigurationComponentTypeId) data;
			String compType = configComp.getConfigurationComponent().getId();
			String configuredComponentTypeId = Controller.getInstance().getProfile().getConfiguredComponentTypeId(compType);
			if ((configuredComponentTypeId != null) && !configuredComponentTypeId.equals(configComp.getTypeId())) {
				setActive = false;
			}
		}
		button.setEnabled(setActive);
	}

	/**
	 * Get result of dialog.
	 * @return Component or ComponentInstance created by this dialog
	 */
	public @Nullable IChildProvidable getResult() {
		return result;
	}
	
	/**
	 * Set result of dialog.
	 * @param result Component or ComponentInstance created by this dialog
	 */
	private void setResult(@Nullable IChildProvidable result) {
		this.result = result;
	}
}
