/**
 * Copyright 2017-2019 NXP
 */
package com.nxp.swtools.periphs.gui.view;

import java.util.HashSet;
import java.util.Set;

import org.eclipse.jface.viewers.IDecoration;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.IViewSite;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.controller.PeriphControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.ComponentSettingView;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.utils.events.IEventListener;
import com.nxp.swtools.utils.resources.ToolsImages;
import com.nxp.swtools.utils.view.ToolView;

/**
 * Basic implementation of a view with added functionality for Peripherals events.
 * @author Juraj Ondruska
 */
public abstract class APeriphsViewBase extends ToolView {
	/** Associated controller wrapper*/
	protected final @NonNull IControllerWrapper controllerWrapper = PeriphControllerWrapper.getInstance();
	/** Listeners which were registered to the controller and need to be unregistered during disposal */
	private final @NonNull Set<@NonNull IEventListener> registeredListeners = new HashSet<>();

	/**
	 * Processes a change in settings and generates an event.
	 * @param eventType type of event to generate
	 * @param actionLabel label for undo point
	 */
	public void handleSettingChange(int eventType, @NonNull String actionLabel) {
		controllerWrapper.getController().handleSettingChange(eventType, this, actionLabel);
	}

	/**
	 * Register listener to the controller. The listener will be automatically unregistered during view disposal.
	 * @param eventTypes types of events to listen for
	 * @param listener to register
	 */
	protected final void registerListener(int eventTypes, @NonNull IEventListener listener) {
		registeredListeners.add(listener);
		controllerWrapper.getController().addListener(eventTypes, listener);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#dispose()
	 */
	@Override
	public void dispose() {
		super.dispose();
		for (IEventListener listener : registeredListeners) {
			controllerWrapper.getController().removeListener(listener);
		}
		registeredListeners.clear();
	}

	/**
	 * Returns image with given name and decorates it with status decoration
	 * @param name name of the image to be decorated
	 * @param problemLevel specifying which status decoration it should return or -1 for image with no decoration
	 * @return image with status decoration or without it if problem level is below zero
	 */
	public static Image getDecoratedImage(@NonNull String name, int problemLevel) {
		if (problemLevel >= ErrorLevels.MIN_LEVEL) {
			return ToolsImages.getImageWithProblemDecorator(name, problemLevel, IDecoration.TOP_LEFT);
		}
		return ToolsImages.getImage(name);
	}

	/**
	 * Open component setting view.
	 * @param componentType type of the component to edit
	 * @param componentName name of component to edit (component type in case global is set to {@code true})
	 * @param global whether editor for global configSet or for component instance should be opened
	 * @return {@code true} if an editor was successfully opened, {@code false} otherwise
	 */
	public boolean openView(@NonNull String componentType, @NonNull String componentName, boolean global) {
		return ComponentSettingView.open(getViewSite(), componentType, componentName, global, true);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.ViewPart#getViewSite()
	 */
	@Override
	public @NonNull IViewSite getViewSite() {
		IViewSite viewSite = super.getViewSite();
		assert (viewSite != null);
		return viewSite;
	}

}
