/**
 * Copyright 2017-2018 NXP
 */
package com.nxp.swtools.periphs.gui.toolbar;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.fieldassist.ControlDecoration;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.nxp.swtools.common.ui.utils.dialogs.ShellProvider;
import com.nxp.swtools.common.ui.utils.swt.ControlDecorationUtils;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.ui.utils.swt.widgets.InstantSearchList;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.controller.events.EventTypes;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.view.componentsettings.ComponentSettingViewHelper;
import com.nxp.swtools.periphs.model.config.ChildContext;
import com.nxp.swtools.periphs.model.config.FunctionalGroup;
import com.nxp.swtools.periphs.model.config.PeriphsProfile;
import com.nxp.swtools.periphs.model.data.mcu.IMcu;
import com.nxp.swtools.provider.SWToolsPlatform;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.provider.configuration.SharedConfigurationFactory;
import com.nxp.swtools.provider.configuration.storage.ProcessorCore;
import com.nxp.swtools.provider.configuration.storage.StorageTool;
import com.nxp.swtools.provider.configuration.storage.periphs.StoragePeriphsFuncGroup;
import com.nxp.swtools.utils.Limits;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.dialog.IChangeListener;
import com.nxp.swtools.utils.dialog.IFunctionalGroupUpdateListener;
import com.nxp.swtools.utils.dialog.ISelectedFunctionProvider;
import com.nxp.swtools.utils.dialog.ISpecificContent;
import com.nxp.swtools.utils.dialog.ToolFunctionModification;
import com.nxp.swtools.utils.events.IEventListener;
import com.nxp.swtools.utils.events.ToolEvent;
import com.nxp.swtools.utils.resources.ToolsColors.SwToolsColors;
import com.nxp.swtools.utils.resources.ToolsImages;
import com.nxp.swtools.utils.toolbar.AFunctionManager;
import com.nxp.swtools.utils.tools.IToolFunction;

/**
 * Class used for managing peripherals functional groups.
 * @author Juraj Ondruska
 * @author Marek Ciz
 */
public class PeriphsToolBarManager extends AFunctionManager {
	/** Logger used by ToolFunctionDialog class */
	protected static final @NonNull Logger LOGGER = LogManager.getLogger(PeriphsToolBarManager.class);

	/**
	 * Constructor.
	 */
	public PeriphsToolBarManager() {
		Controller.getInstance().addListener(EventTypes.CHANGE | EventTypes.SETTING_CHANGE, new IEventListener() {
			/* (non-Javadoc)
			 * @see com.nxp.swtools.utils.events.IEventListener#handle(com.nxp.swtools.utils.events.ToolEvent)
			 */
			@Override
			public void handle(@NonNull ToolEvent event) {
				updateFunctionBar();
			}
		});
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.toolbar.IFunctionManager#getFunctions()
	 */
	@Override
	public @NonNull List<? extends @NonNull IToolFunction> getFunctions() {
		return Controller.getInstance().getProfile().getFunctionalGroups();
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.toolbar.IFunctionManager#getWorkingFunction()
	 */
	@Override
	public @NonNull IToolFunction getWorkingFunction() {
		return Controller.getInstance().getFunctionalGroup();
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.toolbar.IFunctionManager#removeFunction(com.nxp.swtools.utils.toolbar.IToolFunction)
	 */
	@Override
	public boolean removeFunction(@NonNull IToolFunction function) {
		if (function instanceof FunctionalGroup) {
			FunctionalGroup group = (FunctionalGroup) function;
			boolean allowed = MessageDialog.openQuestion(getShell(), Messages.get().PeriphsToolBarManager_RemoveGroupDialog_Title,
					MessageFormat.format(UtilsText.safeString(Messages.get().PeriphsToolBarManager_RemoveGroupDialog_Message), group.getName()));
			if (allowed) {
				ComponentSettingViewHelper.getInstance().removeEditorsOfGroup(group.getStorageFuncGroup());
				return Controller.getInstance().removeFunctionalGroup(group, PeriphsToolBarManager.this);
			}
		}
		return false;
	}

	/**
	 * Get shell. Must be called within UI thread.
	 * @return the shell
	 */
	private static @NonNull Shell getShell() {
		Display display = Display.getCurrent();
		assert (display != null);
		return ShellProvider.getAnyShellParent(display);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.toolbar.IFunctionManager#setWorkingFunction(com.nxp.swtools.utils.toolbar.IToolFunction)
	 */
	@Override
	public void setWorkingFunction(@NonNull IToolFunction function) {
		Controller.getInstance().setFunctionalGroup((FunctionalGroup) function, PeriphsToolBarManager.this);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.utils.toolbar.IFunctionManager#update(java.util.List)
	 */
	@Override
	public void update(List<@NonNull ToolFunctionModification> toolFunctions) {
		Controller controller = Controller.getInstance();
		PeriphsProfile profile = controller.getProfile();
		profile.clearFunctionalGroups();
		toolFunctions.forEach(x -> {
			FunctionalGroup group = (FunctionalGroup) x.getOriginalFunction();
			SpecificContent specific = getToolSpecific(x);
			group.setName(x.getName());
			group.setDescription(x.getDescription());
			group.setCalledFromDefaultInit(x.isCalledFromDefaultInit());
			group.setCore(specific.selectedCore);
			group.setIdPrefix(specific.idPrefix);
			profile.addFunctionalGroup(group);
		});
		// if functional group that was active was deleted, select first one as active
		if (!profile.getFunctionalGroups().contains(controller.getFunctionalGroup())) {
			controller.setFunctionalGroup(profile.getFunctionalGroups().get(0), this);
		}
		controller.handleSettingChange(EventTypes.CHANGE, PeriphsToolBarManager.this,
				UtilsText.safeString(com.nxp.swtools.periphs.controller.Messages.get().Controller_Action_ApplyFunctionProperties));
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.utils.toolbar.IFunctionManager#copyFunction(com.nxp.swtools.utils.dialog.ToolFunctionModification)
	 */
	@Override
	@NonNull public IToolFunction copyFunction(@NonNull ToolFunctionModification functionToCopy) {
		FunctionalGroup group = (FunctionalGroup) functionToCopy.getOriginalFunction();
		ChildContext profileContext = Controller.getInstance().getProfile().getChildContext();
		return new FunctionalGroup(profileContext, new StoragePeriphsFuncGroup(group.getStorageFuncGroup()), group.getMcu());
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.toolbar.IFunctionManager#allowMultipleCallFromInit()
	 */
	@Override
	public boolean allowMultipleCallFromInit() {
		return false;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.toolbar.IFunctionManager#fillSpecificProperties(org.eclipse.swt.widgets.Composite, com.nxp.swtools.utils.dialog.ToolFunctionModification, java.util.List, com.nxp.swtools.utils.dialog.ChangeListener)
	 */
	@Override
	public IFunctionalGroupUpdateListener createSpecificProperties(@NonNull Composite parent, @NonNull ISelectedFunctionProvider selectedFunctionProvider, @NonNull IChangeListener changeListener) {

		ToolFunctionModification function = selectedFunctionProvider.getSelectedFunction();
		if (function != null) {
			// Functional group prefix label
			Label prefixLabel = new Label(parent, SWT.NONE);
			setLabelLayout(prefixLabel, UtilsText.safeString(Messages.get().PeriphsFunctionalGroupDialog_IdPrefix), UtilsText.safeString(Messages.get().PeriphsFunctionalGroupDialog_IdPrefixTooltip));
			// Functional group prefix text widget
			final Text text = new Text(parent, SWT.BORDER);
			setPrefixTextLayout(text);
			// Functional group prefix text widget decorator
			final ControlDecoration decoration = new ControlDecoration(text, SWT.LEFT | SWT.TOP);
			decoration.setImage(ToolsImages.getStatusDecoratorImg(ErrorLevels.LEVEL_ERROR));
			decoration.hide();

			text.addModifyListener(new ModifyListener() {
				/* (non-Javadoc)
				 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
				 */
				@Override
				public void modifyText(ModifyEvent e) {
					ToolFunctionModification functionLoc = selectedFunctionProvider.getSelectedFunction();
					if (functionLoc != null) {
						final SpecificContent specificContentLoc = getToolSpecific(functionLoc);
						specificContentLoc.idPrefix = UtilsText.safeString(text.getText());
						updateIdPrefixError(decoration, functionLoc);
						changeListener.propertiesChange();
					}
				}
			});
			// refresh the GUI error displaying
			updateIdPrefixError(decoration, function);
			// combo for storing processor's cores
			InstantSearchList coreCombo = fillCombo(selectedFunctionProvider, parent, changeListener);
			return getUpdateListener(text, decoration, coreCombo, changeListener);
		}
		else {
			LOGGER.warning("No functional group selected."); //$NON-NLS-1$
			return null;
		}
	}

	/**
	 * Selects function group core in Core combo
	 * @param mcu MCU specific information
	 * @param updatedFunction actual selected Function group
	 * @param specificContent Tool specific content of Function group
	 * @param coreCombo combo showing cores
	 */
	static void selectCoreName(@NonNull IMcu mcu, @NonNull SpecificContent specificContent,
			@NonNull InstantSearchList coreCombo) {
		String selectedCoreName = UtilsText.EMPTY_STRING;
		for (ProcessorCore core : mcu.getCores()) {
			if (specificContent.selectedCore.equals(UtilsText.EMPTY_STRING)) {
				specificContent.selectedCore = core.getID();
			}
			if (specificContent.selectedCore.equals(core.getID())) {
				selectedCoreName = core.getName();
			}
		}
		coreCombo.select(selectedCoreName);
	}

	/**
	 * Set prefix text's layout
	 * @param text ID prefix text
	 */
	private static void setPrefixTextLayout(@NonNull Text text) {
		text.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false));
		text.setTextLimit(Limits.MAX_TOOL_CONFIG_NAME_LENGTH);
		SWTFactoryProxy.INSTANCE.setTestId(text, TestIDs.PERIPHS_PROPERTIES_ID_PREFIX_TEXT);
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(text, Messages.get().PeriphsFunctionalGroupDialog_IdPrefixTooltip);
	}

	/**
	 * Set layout of a Function group core combo
	 * @param label Function group core combo (instantSearchList) label
	 * @param combo Function group core combo (instantSearchList)
	 */
	private static void setComboLayout(@NonNull InstantSearchList combo) {
		combo.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false));
		SWTFactoryProxy.INSTANCE.setTestId(combo, TestIDs.PERIPHS_PROPERTIES_CORES_COMBO);
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(combo, Messages.get().PeriphsFunctionalGroupDialog_CoreTooltip);
	}

	/**
	 * Set label's layout
	 * @param label label to be set
	 * @param text text shown by label
	 * @param id label's id
	 * @param toolTip label's toolTip
	 */
	private static void setLabelLayout(@NonNull Label label, @NonNull String text, @NonNull String toolTip) {
		label.setText(text);
		label.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, false, false));
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(label, toolTip);
	}

	/**
	 * Refresh state of the ID-prefix error decorator.
	 * @param decoration to update
	 * @param function to update the decorator for
	 */
	void updateIdPrefixError(@NonNull ControlDecoration decoration, @NonNull ToolFunctionModification function) {
		if (isIdPrefixValid(function)) {
			decoration.hide();
		} else {
			decoration.show();
		}
		decoration.setDescriptionText(com.nxp.swtools.utils.Messages.get().ToolProperties_InvalidNameMessage);
	}

	/**
	 * Check whether the id-prefix of the function is valid.
	 * @param function to check id-prefix of
	 * @return {@code true} if the id-prefix is valid, {@code false} otherwise
	 */
	boolean isIdPrefixValid(@NonNull ToolFunctionModification function) {
		String idPrefix = getToolSpecific(function).idPrefix;
		return idPrefix.isEmpty() || UtilsText.isIdentifier(idPrefix);
	}

	/**
	 * Check whether the id-prefix of the function is unique.
	 * @param function to check id-prefix of
	 * @param functions to check id-prefix in
	 * @return {@code true} if the id-prefix is valid, {@code false} otherwise
	 */
	boolean isIdPrefixUnique(@NonNull ToolFunctionModification function, @NonNull List<@NonNull ToolFunctionModification> functions) {
		String functionIdPrefix = getToolSpecific(function).idPrefix;
		return functions.stream()
				.filter(x -> x != function)
				.noneMatch(x -> functionIdPrefix.equals(getToolSpecific(x).idPrefix));
	}

	/**
	 * Get the peripherals tool specific content from the function. Create and set one if it does not exist.
	 * @param function to get content from
	 * @return the peripherals tool specific content
	 */
	static @NonNull SpecificContent getToolSpecific(@NonNull ToolFunctionModification function) {
		SpecificContent specificContent = (SpecificContent) function.getToolSpecific();
		if (specificContent == null) {
			FunctionalGroup functionalGroup = (FunctionalGroup) function.getOriginalFunction();
			specificContent = new SpecificContent();
			specificContent.selectedCore = functionalGroup.getCore();
			specificContent.idPrefix = functionalGroup.getIdPrefix();
			function.setToolSpecific(specificContent);
		}
		return specificContent;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.toolbar.IFunctionManager#setCalledFromDefaultInit(com.nxp.swtools.utils.tools.IToolFunction, boolean)
	 */
	@Override
	public void setCalledFromDefaultInit(@NonNull IToolFunction function, boolean called) {
		if (!(function instanceof FunctionalGroup)) {
			return;
		}
		Controller.getInstance().setCalledFromDefaultInit((FunctionalGroup) function, called, PeriphsToolBarManager.this);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.toolbar.IFunctionManager#isToolSpecificPropertiesValid(com.nxp.swtools.utils.dialog.ToolFunctionModification, java.util.List)
	 */
	@Override
	public boolean isToolSpecificPropertiesValid(@NonNull ToolFunctionModification function,
			@NonNull List<@NonNull ToolFunctionModification> functions) {
		return isIdPrefixValid(function);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.toolbar.IFunctionManager#getReservedFunctionName()
	 */
	@Override
	public String getReservedFunctionName() {
		return FunctionalGroup.NAME_RESERVED;
	}

	/**
	 * Class used for holding peripherals tool specific content.
	 * @author Juraj Ondruska
	 */
	static class SpecificContent implements ISpecificContent {
		/** Selected core */
		@NonNull String selectedCore = UtilsText.EMPTY_STRING;
		/** Selected id-prefix */
		@NonNull String idPrefix = UtilsText.EMPTY_STRING;

		/**
		 * Implicit constructor
		 */
		public SpecificContent() {
			super();
		}

		/**
		 * Copy constructor
		 * @param copyFrom function to create copy from
		 */
		public SpecificContent(@NonNull SpecificContent copyFrom) {
			this.selectedCore = copyFrom.selectedCore;
			this.idPrefix = copyFrom.idPrefix;
		}

		/* (non-Javadoc)
		 * @see com.nxp.swtools.utils.dialog.ISpecificContent#createCopy()
		 */
		@Override
		public @NonNull ISpecificContent createCopy() {
			return new SpecificContent(this);
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.toolbar.IFunctionManager#createNewFunction()
	 */
	@Override
	public @NonNull IToolFunction createNewFunction(@NonNull String functionName) {
		Controller controller = Controller.getInstance();
		if (controller.isMcuSupported()) {
			FunctionalGroup group = new FunctionalGroup(controller.getProfile().getChildContext(), controller.getMcu());
			group.setName(functionName);
			group.setDescription(UtilsText.EMPTY_STRING);
			group.setCalledFromDefaultInit(false);
			if (!group.setCoreFrom(SharedConfigurationFactory.getSharedConfigurationSingleton().getCommonConfig())) {
				group.setMasterCoreAsUsed();
			}
			return group;
		} else {
			return controller.getFunctionalGroup();
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.toolbar.IFunctionManager#getMaxToolFunctions()
	 */
	@Override
	public int getMaxToolFunctions() {
		return Limits.MAX_TOOL_CONFIGS;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.toolbar.IFunctionManager#getDefaultFunctionName()
	 */
	@Override
	public @NonNull String getDefaultFunctionName() {
		return FunctionalGroup.NAME_PREFIX_FOR_ADDING;
	}

	/**
	 * Get MCU specific information
	 * @param function functional group function
	 * @return MCU specific information
	 */
	static @Nullable IMcu getMcu(@NonNull ToolFunctionModification function) {
		IMcu mcu = null;
		IToolFunction toolFunction = function.getOriginalFunction();
		if (toolFunction instanceof FunctionalGroup) {
			FunctionalGroup functionalGroup = (FunctionalGroup) toolFunction;
			mcu = functionalGroup.getMcu();
		}
		return mcu;
	}

	/**
	 * Returns Update listener with updated Core combo and Text widget information
	 * @param text text widget
	 * @param decoration Text widget decorator
	 * @param coreCombo MCU core selection combo
	 * @param changeListener Listener that listens to Tool specific composite
	 * @return updated Update listener with Core combo and Text widget information
	 */
	@NonNull IFunctionalGroupUpdateListener getUpdateListener(@NonNull Text text, @NonNull ControlDecoration decoration,
			@Nullable InstantSearchList coreCombo, @NonNull IChangeListener changeListener) {
		// listener that guarantees, that prefix will change due to Function name changes
		IFunctionalGroupUpdateListener updateListener = new IFunctionalGroupUpdateListener() {

			/* (non-Javadoc)
			 * @see com.nxp.swtools.utils.dialog.FunctionalGroupUpdateListener#updateFunctionalGroup(com.nxp.swtools.utils.dialog.ToolFunctionModification)
			 */
			@Override
			public void updateFunctionalGroup(@NonNull ToolFunctionModification updatedFunction) {
				SpecificContent specificContentLoc = getToolSpecific(updatedFunction);
				// set text after registering the listener to update its error state
				text.setText(specificContentLoc.idPrefix);
				// refresh the GUI error displaying
				updateIdPrefixError(decoration, updatedFunction);
				IMcu mcuLoc = PeriphsToolBarManager.getMcu(updatedFunction);
				if (mcuLoc != null) {
					if (mcuLoc.isMulticore() && (coreCombo != null)) {
						selectCoreName(mcuLoc, specificContentLoc, coreCombo);
						ProcessorCore selectedCoreOfProject = SharedConfigurationFactory.getSharedConfigurationSingleton().getCommonConfig().getSelectedCore();
						ArrayList<@NonNull String> wrongCores = new ArrayList<>(mcuLoc.getCores().size());
						// Add warnings to not matching cores in combo box
						for (ProcessorCore core : mcuLoc.getCores()) {
							if (selectedCoreOfProject != null) {
								if (!core.getID().equals(selectedCoreOfProject.getID())) {
									wrongCores.add(core.getName());
								}
							}
						}
						Image warningImg = ToolsImages.getStatusDecoratorImg(ErrorLevels.LEVEL_WARNING);
						coreCombo.setStatusItemProvider(it -> wrongCores.contains(it)?warningImg:null);
						// Add warning if wrong core is selected
						updateErrorDecorationOfCoreCombo(updatedFunction, coreCombo);
					}
					changeListener.propertiesChange();
				}
			}
		};
		return updateListener;
	}

	/**
	 * Fills combo with MCU select core options and adds selection listener
	 * @param functionProvider selected function provider
	 * @param parent parent composite
	 * @param changeListener Listener that listens to Tool specific composite
	 * @return combo with MCU select core options and adds selection listener
	 */
	@Nullable InstantSearchList fillCombo(@NonNull ISelectedFunctionProvider functionProvider, @NonNull Composite parent, @NonNull IChangeListener changeListener) {
		InstantSearchList combo = null;
		final ToolFunctionModification function = functionProvider.getSelectedFunction();
		if (function != null) {
			IMcu mcu = getMcu(function);
			if (mcu != null) {
				// show combo, if MCU has multiple cores
				if (mcu.isMulticore()) {
					// Core label
					Label label = new Label(parent, SWT.NONE);
					// Core combo
					combo = new InstantSearchList(parent, SWT.BORDER);
					setLabelLayout(label, UtilsText.safeString(Messages.get().PeriphsFunctionalGroupDialog_Core),
							UtilsText.safeString(Messages.get().PeriphsFunctionalGroupDialog_CoreTooltip));
					setComboLayout(combo);
					ControlDecorationUtils.createErrorDecoration(combo, SWT.LEFT | SWT.TOP);
					ControlDecorationUtils.setImage(combo, ToolsImages.getStatusDecoratorImg(ErrorLevels.LEVEL_WARNING));
					// Add cores
					for (ProcessorCore core : mcu.getCores()) {
						combo.add(core.getName());
					}
					// Add listener
					final InstantSearchList comboLoc = combo;
					combo.addSelectionListener(new SelectionAdapter() {
						@Override
						public void widgetSelected(SelectionEvent e) {
							final ToolFunctionModification currentFunction = functionProvider.getSelectedFunction();
							if (currentFunction != null) {
								final String comboSelectionText = UtilsText.safeString(comboLoc.getText());
								final ProcessorCore coreFromSelection = CollectionsUtils.findFirst(mcu.getCores(), x -> x.getName().equals(comboSelectionText));
								getToolSpecific(currentFunction).selectedCore = (coreFromSelection != null) ? coreFromSelection.getID() : UtilsText.EMPTY_STRING;
								// Default background and decoration is hidden
								comboLoc.setForeground(null);
								ControlDecorationUtils.hide(comboLoc);
								updateErrorDecorationOfCoreCombo(currentFunction, comboLoc);
								changeListener.propertiesChange();
							}
						}

					});
				}
			}
		}
		return combo;
	}

	/**
	 * Updates error decorator of cores combo
	 * @param function modified function
	 * @param combo combo to update
	 */
	void updateErrorDecorationOfCoreCombo(@NonNull ToolFunctionModification function, @NonNull InstantSearchList combo) {
		final ProcessorCore selectedCoreOfProject = SharedConfigurationFactory.getSharedConfigurationSingleton().getCommonConfig().getSelectedCore();
		final String selectedCoreName = UtilsText.safeString(combo.getText());
		final IMcu mcu = getMcu(function);
		final SpecificContent toolContent = getToolSpecific(function);
		if (selectedCoreOfProject == null) {
			return;
		}
		if (mcu == null) {
			return;
		}
		final ProcessorCore coreFromSelection = mcu.getCore(toolContent.selectedCore);
		if (coreFromSelection == null) {
			return;
		}
		if (!selectedCoreName.equals(selectedCoreOfProject.getName())) {
			// Set warning to combo box if wrong core is selected
			String message = MessageFormat.format(UtilsText.safeString(com.nxp.swtools.periphs.model.Messages.get().FunctionalGroup_DifferentSelectedCore),
					coreFromSelection.getName(), selectedCoreOfProject.getName());
			combo.setForeground(SwToolsColors.WARNING_FG.getColor());
			ControlDecorationUtils.setDescription(combo, message);
			ControlDecorationUtils.show(combo);
		} else {
			// Clear settings and hide
			combo.setForeground(combo.getDisplay().getSystemColor(SWT.COLOR_LIST_FOREGROUND));
			ControlDecorationUtils.setDescription(combo, UtilsText.EMPTY_STRING);
			ControlDecorationUtils.hide(combo);
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.toolbar.IFunctionManager#getStorageTool()
	 */
	@Override
	public @Nullable StorageTool getStorageTool() {
		return SWToolsPlatform.getSharedConfigurationSingleton().getTools().getPeripherals();
	}
}
