/*******************************************************************************
 * Copyright (c) 2018 NXP
 *******************************************************************************/
package com.freescale.s32ds.cdt.core;

import static org.eclipse.cdt.core.CCorePlugin.log;

import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Vector;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.eclipse.cdt.core.CCorePlugin;
import org.eclipse.cdt.core.ErrorParserManager;
import org.eclipse.cdt.core.IConsoleParser;
import org.eclipse.cdt.core.IErrorParser;
import org.eclipse.cdt.core.IErrorParser2;
import org.eclipse.cdt.core.IErrorParser3;
import org.eclipse.cdt.core.IErrorParserNamed;
import org.eclipse.cdt.core.IMarkerGenerator;
import org.eclipse.cdt.core.ProblemMarkerInfo;
import org.eclipse.cdt.core.errorparsers.ErrorParserNamedWrapper;
import org.eclipse.cdt.core.language.settings.providers.IWorkingDirectoryTracker;
import org.eclipse.cdt.core.resources.ACBuilder;
import org.eclipse.cdt.internal.core.Cygwin;
import org.eclipse.cdt.internal.core.IErrorMarkeredOutputStream;
import org.eclipse.cdt.internal.core.resources.ResourceLookup;
import org.eclipse.cdt.utils.EFSExtensionManager;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.URIUtil;
import org.osgi.service.prefs.BackingStoreException;

import com.freescale.s32ds.cdt.core.internal.errorparsers.CompilationPathResolverManager;
import com.freescale.s32ds.cdt.core.internal.errorparsers.ErrorParserDelegateManager;

@SuppressWarnings("restriction")
public class FSLErrorParserManager extends ErrorParserManager implements IConsoleParser, IWorkingDirectoryTracker, IErrorParserProcessor {

	private static final String EMPTY_STRING = ""; //$NON-NLS-1$

	public final static char S32DS_ERROR_PARSER_DELIMITER = ';';

	@Deprecated
	public final static String S32DS_PREF_ERROR_PARSER = CCorePlugin.PLUGIN_ID + ".errorOutputParser"; //$NON-NLS-1$

	public static final String S32DS_BUILD_CONTEXT = "build"; //$NON-NLS-1$

	private int numberOpens;

	private int lineCounter = 0;

	private final IProject project;
	private final IMarkerGenerator markerGenerator;

	private Collection<IErrorParser> collectionsOfErrorParsers;
	private List<ProblemMarkerInfo> listOfErrors;
	private List<ProblemMarkerInfo> listOfPendingErrors;

	private final Vector<URI> directoryStack;
	private IErrorParserManagerDelegate[] delegates;
	private ICompilationPathResolver[] resolvers;

	private final URI baseDirectoryURI;

	private IErrorParser3 previousErrorParser = null;

	private String previousLine;

	private OutputStream outputStream;

	private final StringBuilder currentLine = new StringBuilder();

	private final StringBuilder scratchBuffer = new StringBuilder();

	private boolean hasErrors = false;

	private String cachedFileName = null;
	private URI cachedWorkingDirectory = null;
	private IFile cachedFile = null;

	private static boolean isCygwin = true;

	private List<String> listOfErrorMsgLines = new ArrayList<>();

	private boolean isErrorMsgComplete = false;

	private final OutputLineQueue outputLineQueue = new OutputLineQueue();

	public FSLErrorParserManager(ACBuilder builder) {
		this(builder.getProject(), builder);
	}

	public FSLErrorParserManager(IProject project, IMarkerGenerator markerGenerator) {
		this(project, markerGenerator, null);
	}

	public FSLErrorParserManager(IProject project, IMarkerGenerator markerGenerator, String[] parsersIds) {
		this(project, (URI) null, markerGenerator, parsersIds);
	}

	@SuppressWarnings("null")
	@Deprecated
	public FSLErrorParserManager(IProject project, IPath workingDirectory, IMarkerGenerator markerGenerator, String[] parsersIds) {
		this(project, (workingDirectory != null || workingDirectory.isEmpty()) ? null
				: org.eclipse.core.filesystem.URIUtil.toURI(workingDirectory), markerGenerator, parsersIds);
	}

	public FSLErrorParserManager(IProject project, URI baseDirectoryURI, IMarkerGenerator markerGenerator, String[] parsersIds) {
		super(project, baseDirectoryURI, markerGenerator, parsersIds);
		this.project = project;
		this.markerGenerator = markerGenerator;
		this.directoryStack = new Vector<>();
		this.listOfErrors = new ArrayList<>();
		this.listOfPendingErrors = new ArrayList<>();
		if (parsersIds == null) {
			parsersIds = FSLErrorParserExtensionManager.getDefaultErrorParserIds();
		}
		collectionsOfErrorParsers = Stream.of(parsersIds).map(FSLErrorParserManager::getErrorParserCopy).filter(Objects::nonNull)
				.collect(Collectors.toList());
		if (baseDirectoryURI != null)
			this.baseDirectoryURI = baseDirectoryURI;
		else if (project != null) {
			this.baseDirectoryURI = project.getLocationURI();
		} else {
			this.baseDirectoryURI = org.eclipse.core.filesystem.URIUtil.toURI(System.getProperty("user.dir")); // CWD //$NON-NLS-1$
		}
		this.resolvers = CompilationPathResolverManager.getResolvers();
		this.delegates = ErrorParserDelegateManager.getDelegates();
		for (IErrorParserManagerDelegate delegate : this.delegates)
			delegate.initialize(this);
	}

	@Override
	public Collection<IErrorParser> getErrorParsers() {
		return collectionsOfErrorParsers;
	}

	@Override
	public IProject getProject() {
		return project;
	}

	@Override
	@Deprecated
	public IPath getWorkingDirectory() {
		return org.eclipse.core.filesystem.URIUtil.toPath(getWorkingDirectoryURI());
	}

	@Override
	public URI getWorkingDirectoryURI() {
		if (!this.directoryStack.isEmpty()) {
			return this.directoryStack.lastElement();
		}
		return this.baseDirectoryURI;
	}

	@Override
	public void pushDirectory(IPath path) {
		if (path != null) {
			URI uri;
			URI workingDirectoryURI = getWorkingDirectoryURI();
			if (!path.isAbsolute()) {
				uri = URIUtil.append(workingDirectoryURI, path.toString());
			} else {
				uri = toURI(path);
				if (uri == null) {
					return;
				}
			}
			pushDirectoryURI(uri);
		}
	}

	@Override
	public void pushDirectoryURI(URI uri) {
		if (uri != null) {
			if (uri.isAbsolute()) {
				this.directoryStack.addElement(uri);
			} else {
				this.directoryStack.addElement(URIUtil.makeAbsolute(uri, getWorkingDirectoryURI()));
			}
		}
	}

	@Override
	@Deprecated
	public IPath popDirectory() {
		return org.eclipse.core.filesystem.URIUtil.toPath(popDirectoryURI());
	}

	@Override
	public URI popDirectoryURI() {
		int i = this.directoryStack.size();
		if (i != 0) {
			URI dir = this.directoryStack.lastElement();
			this.directoryStack.removeElementAt(i - 1);
			return dir;
		}
		return this.baseDirectoryURI;
	}

	@Override
	public int getDirectoryLevel() {
		int size = directoryStack.size();
		return size;
	}

	@Override
	@Deprecated
	protected void collectFiles(IProject parent, final List<IResource> result) {
		try {
			parent.accept(proxy -> {
				if (proxy.getType() == IResource.FILE) {
					result.add(proxy.requestResource());
					return false;
				}
				return true;
			}, IResource.NONE);
		} catch (CoreException e) {
			CCorePlugin.log(e.getStatus());
		}
	}

	@Override
	public boolean processLine(String line) {
		registerOutputLine(line);
		for (IErrorParserManagerDelegate delegate : this.delegates) {
			if (delegate.processLine(line, this))
				return true;
		}
		return processErrorParsers(line, collectionsOfErrorParsers);
	}

	public void registerOutputLine(String line) {
		registerOutputLine(line, false);
	}

	public void registerOutputLine(String line, boolean duplicated) {
		outputLineQueue.add(new OutputLine(line, duplicated));
	}

	@Override
	public boolean processErrorParsers(String line, Collection<IErrorParser> parsers) {

		// need for checking if error message hang-up CRs: ENGR00360323,
		// ENGR00359283
		lineCounter++;
		int sizeBeforeCycle = listOfErrorMsgLines.size();
		String trimLine = line.trim();
		ProblemMarkerInfo marker = null;
		pointer: for (IErrorParser errorParser : parsers) {
			int parserTypes = IErrorParser2.NONE;
			IErrorParser currentErrorParser = errorParser;
			if (errorParser instanceof ErrorParserNamedWrapper) {
				currentErrorParser = ((ErrorParserNamedWrapper) errorParser).getErrorParser();
			}
			if (currentErrorParser instanceof IErrorParser2) {
				parserTypes = ((IErrorParser2) currentErrorParser).getProcessLineBehaviour();
			}
			int emptyType = 0;
			if ((parserTypes & IErrorParser2.KEEP_LONGLINES) == emptyType) {
				int longLine = 1000;
				if (trimLine.length() > longLine)
					continue;
			}
			String parsedLine = trimLine;
			if ((parserTypes & IErrorParser2.KEEP_UNTRIMMED) != emptyType) {
				parsedLine = line;
			} else {
				outputLineQueue.registerAlias(line, trimLine);
			}
			boolean result = false;
			try {
				try {
					result = currentErrorParser.processLine(parsedLine, this);
				} catch (Exception e) {
					String id = EMPTY_STRING;
					if (errorParser instanceof IErrorParserNamed) {
						id = ((IErrorParserNamed) errorParser).getId();
					}
					String message = MessageFormat.format("Errorparser {0} failed parsing line [{1}]", id, parsedLine); //$NON-NLS-1$
					CCorePlugin.log(message, e);
				}
				if (result) {
					if (previousErrorParser instanceof IFSLErrorParser3 && previousErrorParser != currentErrorParser) {
						// Because previous error parser errors occurred before
						// the current parser,
						// and because some error parser will only report errors
						// inside the IErrorParser3.flush()
						// call, we need to store the current pending error list
						// so the the previous parser's
						// errors are listed before the current ones, if any.
						List<ProblemMarkerInfo> temporaryErrors = listOfPendingErrors;
						listOfPendingErrors = new ArrayList<>();
						((IFSLErrorParser3) previousErrorParser).flush(this);
						listOfPendingErrors.addAll(temporaryErrors);
					}
					previousErrorParser = currentErrorParser instanceof IErrorParser3 ? (IErrorParser3) currentErrorParser : null;
					flushPendingErrors();
					break pointer;
				}
				flushPendingErrors();
			} finally {
				if (listOfErrors.size() > 0) {
					if (marker == null)
						marker = listOfErrors.get(listOfErrors.size() - 1);
					listOfErrors.clear();
				}
			}
		}

		if (parsers.isEmpty()) {
			flushPendingErrors();
		} else if (sizeBeforeCycle == 1 && listOfErrorMsgLines.size() == sizeBeforeCycle) {
			// If some message hangs up in fErrorMsgLines
			// flush all error parsers and pending errors.
			// CRs: ENGR00360323, ENGR00359283
			flushPendingErrorParserErrors();
		}

		if (listOfErrorMsgLines.isEmpty())
			outputLine(line, marker); // output line to console immediately if
										// it is not a multi-line error message
		else {
			outputPendingErrorMessages(marker);
		}

		return false;
	}

	/**
	 * Conditionally output line to outputStream. If stream supports error
	 * markers, use it, otherwise use conventional stream
	 */
	private void outputLine(String line, ProblemMarkerInfo marker) {
		outputLineQueue.setDone(line, marker);
		outputDoneLines();
	}

	/**
	 * Output lines to console from the head of the queue till it meets first
	 * undone line
	 */
	private void outputDoneLines() {
		OutputLine line;
		while ((line = outputLineQueue.pollDone()) != null) {
			outputToConsole(line.getLine(), line.getInfo());
		}
	}

	private void outputToConsole(String line, ProblemMarkerInfo marker) {
		String l = line + "\n"; //$NON-NLS-1$
		if (outputStream == null) {
			return;
		}
		try {
			if (marker != null && outputStream instanceof IErrorMarkeredOutputStream) {
				IErrorMarkeredOutputStream s = (IErrorMarkeredOutputStream) outputStream;
				s.write(l, marker);
			} else {
				byte[] b = l.getBytes();
				outputStream.write(b, 0, b.length);
			}
		} catch (IOException e) {
			CCorePlugin.log(e);
		}
	}

	/**
	 * Collect all lines of an error/warning/info message in case it spans
	 * multiple lines
	 * 
	 * @param line
	 *            - a single line of an error/warning/info message
	 */
	public void addErrorMsgLine(String line) {
		listOfErrorMsgLines.add(line);
	}

	/**
	 * Indicate that all lines of error/warning/info message have been processed
	 * 
	 * @param isComplete
	 *            - true if entire message has been processed, else false
	 */
	public void setErrorMsgComplete(boolean isComplete) {
		isErrorMsgComplete = isComplete;
	}

	@Override
	public int getLineCounter() {
		return lineCounter;
	}

	@Override
	public IFile findFileName(String partialLocation) {
		if (partialLocation.equals(cachedFileName) && cachedWorkingDirectory != null
				&& org.eclipse.core.filesystem.URIUtil.equals(getWorkingDirectoryURI(), cachedWorkingDirectory)) {
			return cachedFile;
		}
		char windowsSeparator = '\\';
		IPath path = new Path(partialLocation.replace(windowsSeparator, IPath.SEPARATOR));
		IFile fileInWorkcpace = findFileInWorkspace(path);
		fileInWorkcpace = findeFile(partialLocation, path, fileInWorkcpace);
		cachedFileName = partialLocation;
		cachedWorkingDirectory = getWorkingDirectoryURI();
		cachedFile = fileInWorkcpace;
		return fileInWorkcpace;
	}

	private IFile findeFile(String partialLocation, IPath path, IFile fileInWorkcpace) {
		if (fileInWorkcpace == null) {
			int emptyFiles = 0;
			int oneFile = 1;
			path = path.setDevice(null);
			IFile[] filesArray = null;
			IProject[] projects;
			if (project != null) {
				projects = new IProject[] { project };
				filesArray = ResourceLookup.findFilesByName(path, projects, false);
				if (filesArray.length == emptyFiles) {
					filesArray = ResourceLookup.findFilesByName(path, projects, true);
				}
			}
			if (filesArray == null || filesArray.length == emptyFiles) {
				projects = ResourcesPlugin.getWorkspace().getRoot().getProjects();
				filesArray = ResourceLookup.findFilesByName(path, projects, false);
				if (filesArray.length == emptyFiles) {
					filesArray = ResourceLookup.findFilesByName(path, projects, true);
				}
			}
			if (filesArray.length == oneFile) {
				fileInWorkcpace = filesArray[0];
			}
		}
		if (fileInWorkcpace == null && isCygwin && path.isAbsolute()) {
			fileInWorkcpace = findCygwinFile(partialLocation);
		}
		return fileInWorkcpace;
	}

	@Override
	protected IFile findFileInWorkspace(IPath path) {
		URI uri;
		if (path.isAbsolute()) {
			uri = toURI(path);
			if (uri == null) {
				return null;
			}
		} else {
			URI workingDirectoryURI = getWorkingDirectoryURI();
			uri = EFSExtensionManager.getDefault().append(workingDirectoryURI, path.toString());
		}
		return findFileInWorkspace(uri);
	}

	@Override
	protected IFile findFileInWorkspace(URI uri) {
		IFile result = null;
		if (!uri.isAbsolute()) {
			uri = URIUtil.makeAbsolute(uri, getWorkingDirectoryURI());
		}
		result = ResourceLookup.selectFileForLocationURI(uri, project);
		if (result != null && result.isAccessible()) {
			return result;
		}
		return result;
	}

	@Override
	@Deprecated
	public boolean isConflictingName(String fileName) {
		Path path = new Path(fileName);
		IProject[] iProjects = new IProject[] { project };
		return ResourceLookup.findFilesByName(path, iProjects, false).length > 1;
	}

	@Override
	@Deprecated
	public IFile findFilePath(String filePath) {
		IPath path = new Path(filePath);
		IFile file = findFileInWorkspace(path);
		boolean isCygPath = file == null && isCygwin;
		if (isCygPath) {
			file = findCygwinFile(filePath);
		}
		return (file != null && file.exists()) ? file : null;
	}

	private IFile findCygwinFile(String filePath) {
		try {
			return findFileInWorkspace(new Path(Cygwin.cygwinToWindowsPath(filePath)));
		} catch (UnsupportedOperationException e) {
			isCygwin = false;
		} catch (Exception e) {
			log(e);
		}
		return null;
	}

	@Override
	public void generateMarker(IResource file, int lineNumber, String desc, int severity, String varName) {
		generateExternalMarker(file, lineNumber, desc, severity, varName, null);
	}

	/**
	 * Output the pending lines to the output stream that were accumulated in
	 * the fErrorMsgLines buffer in case of multiple lines error messages.
	 * 
	 * @param marker
	 *            the marke associated with the lines
	 */
	private void outputPendingErrorMessages(ProblemMarkerInfo marker) {
		if (isErrorMsgComplete == true) {
			// only output a multi-line message once all lines of the message
			// have been processed
			Iterator<String> iter = listOfErrorMsgLines.iterator();
			while (iter.hasNext()) {
				outputLine(iter.next(), marker);
			}
			listOfErrorMsgLines.clear();
			isErrorMsgComplete = false;
		}
	}

	@Override
	public void generateExternalMarker(IResource file, int lineNumber, String desc, int severity, String varName, IPath externalPath) {
		if (file == null)
			file = project;

		if ((externalPath != null) && !externalPath.toFile().exists()) {

			for (ICompilationPathResolver resolver : resolvers) {

				IPath path = resolver.resolve(externalPath, project);

				if ((path != null) && path.toFile().exists()) {
					externalPath = path;
					break;
				}
			}
		}

		ProblemMarkerInfo problemMarkerInfo = new ProblemMarkerInfo(file, lineNumber, desc, severity, varName, externalPath);
		outputPendingErrorMessages(problemMarkerInfo);
		listOfPendingErrors.add(problemMarkerInfo);
	}

	/**
	 * Add the given marker to the list of error markers.
	 *
	 * @param problemMarkerInfo
	 *            - The marker to be added.
	 * @since 5.4
	 */
	@Override
	public void addProblemMarker(ProblemMarkerInfo problemMarkerInfo) {
		listOfPendingErrors.add(problemMarkerInfo);
	}

	/**
	 * Persist any pending error marker to the marker generator, and remove any
	 * pending error message line for each pending error, if required.
	 */
	public void flushPendingErrors() {
		for (int i = 0; i < listOfPendingErrors.size(); i++) {
			ProblemMarkerInfo problemMarkerInfo = listOfPendingErrors.get(i);
			listOfErrors.add(problemMarkerInfo);
			markerGenerator.addMarker(problemMarkerInfo);
			if (problemMarkerInfo.severity == IMarkerGenerator.SEVERITY_ERROR_RESOURCE)
				hasErrors = true;
			if (listOfErrorMsgLines.size() > 0)
				outputLine(listOfErrorMsgLines.remove(0), problemMarkerInfo);
		}
		listOfPendingErrors.clear();
	}

	private void flushPendingErrorParserErrors() {
		for (IErrorParser parser : collectionsOfErrorParsers) {
			if (parser instanceof IFSLErrorParser3) {
				((IFSLErrorParser3) parser).flush(this);
			} else if ((parser instanceof ErrorParserNamedWrapper)
					&& (((ErrorParserNamedWrapper) parser).getErrorParser() instanceof IFSLErrorParser3)) {
				((IFSLErrorParser3) ((ErrorParserNamedWrapper) parser).getErrorParser()).flush(this);
			}
		}
		flushPendingErrors();
	}

	@Override
	public String getPreviousLine() {
		String original = (previousLine) == null ? EMPTY_STRING : previousLine;
		return new String(original);
	}

	@Override
	public void setOutputStream(OutputStream osOutputStream) {
		this.outputStream = osOutputStream;
	}

	@Override
	public OutputStream getOutputStream() {
		numberOpens++;
		return this;
	}

	@Override
	public synchronized void close() throws IOException {
		if (previousErrorParser instanceof IFSLErrorParser3) {
			((IFSLErrorParser3) previousErrorParser).flush(this);
			previousErrorParser = null;
		}
		flushPendingErrors();
		if (numberOpens > 0 && --numberOpens == 0) {
			checkLine(true);
			directoryStack.removeAllElements();
		}

		// set all lines done at the end in order to output them to console
		outputLineQueue.setAllDone();
		outputDoneLines();
	}

	@Override
	public void flush() throws IOException {
		if (this.outputStream != null) {
			this.outputStream.flush();
		}
	}

	@Override
	public synchronized void write(int b) throws IOException {
		char charFromString = (char) b;
		currentLine.append(charFromString);
		checkLine(false);
	}

	@Override
	public synchronized void write(byte[] bytes, int off, int len) throws IOException {
		if (bytes == null) {
			throw new NullPointerException();
		} else if (off != 0 || (len < 0) || (len > bytes.length)) {
			throw new IndexOutOfBoundsException();
		} else if (len == 0) {
			return;
		}
		String str = new String(bytes, 0, len);
		currentLine.append(str);
		checkLine(false);
	}

	private void checkLine(boolean flush) {
		String buffer = currentLine.toString();
		int i = 0;
		while ((i = buffer.indexOf('\n')) != -1) {
			String line = buffer.substring(0, i);
			String suffix = "\r"; //$NON-NLS-1$
			if (line.endsWith(suffix)) {
				line = line.substring(0, line.length() - 1);
			}
			if (!line.equals(previousLine)) {
				processLine(line);
			} else {
				registerOutputLine(line, true);
				outputLine(line, null);
			}
			previousLine = line;
			buffer = buffer.substring(i + 1);
		}
		currentLine.setLength(0);
		if (flush) {
			if (buffer.length() > 0) {
				processLine(buffer);
				previousLine = buffer;
			}
		} else {
			currentLine.append(buffer);
		}
	}

	@Override
	@Deprecated
	public boolean reportProblems() {
		return false;
	}

	private URI toURI(IPath path) {
		URI baseURI = getWorkingDirectoryURI();
		String uriString = path.toString();
		boolean isCorrectPath = path.isAbsolute() && uriString.charAt(0) != IPath.SEPARATOR;
		if (isCorrectPath) {
			uriString = IPath.SEPARATOR + uriString;
		}
		return EFSExtensionManager.getDefault().createNewURIFromPath(baseURI, uriString);
	}

	@Override
	@Deprecated
	public String getScratchBuffer() {
		return this.scratchBuffer.toString();
	}

	@Override
	@Deprecated
	public void appendToScratchBuffer(String newLine) {
		this.scratchBuffer.append(newLine);
	}

	@Override
	@Deprecated
	public void clearScratchBuffer() {
		this.scratchBuffer.setLength(0);
	}

	@Override
	@Deprecated
	public boolean hasErrors() {
		return this.hasErrors;
	}

	public static String[] getUserDefinedErrorParserIds() {
		return FSLErrorParserExtensionManager.getUserDefinedErrorParserIds();
	}

	public static void setUserDefinedErrorParsers(IErrorParserNamed[] errorParsers) throws CoreException {
		FSLErrorParserExtensionManager.setUserDefinedErrorParsers(errorParsers);
	}

	public static String[] getErrorParserAvailableIds() {
		return FSLErrorParserExtensionManager.getErrorParserAvailableIds();
	}

	public static String[] getErrorParserAvailableIdsInContext(String context) {
		return FSLErrorParserExtensionManager.getErrorParserAvailableIdsInContext(context);
	}

	public static String[] getErrorParserExtensionIds() {
		return FSLErrorParserExtensionManager.getErrorParserExtensionIds();
	}

	public static void setDefaultErrorParserIds(String[] iDs) throws BackingStoreException {
		FSLErrorParserExtensionManager.setDefaultErrorParserIds(iDs);
	}

	public static String[] getDefaultErrorParserIds() {
		return FSLErrorParserExtensionManager.getDefaultErrorParserIds();
	}

	public static IErrorParserNamed getErrorParserCopy(String id) {
		return FSLErrorParserExtensionManager.getErrorParserCopy(id, false);
	}

	public static IErrorParserNamed getErrorParserExtensionCopy(String id) {
		return FSLErrorParserExtensionManager.getErrorParserCopy(id, true);
	}

	public static String toDelimitedString(String[] iDs) {
		String result = EMPTY_STRING;
		for (String iD : iDs) {
			if (result.length() == 0) {
				result = iD;
			} else {
				result += S32DS_ERROR_PARSER_DELIMITER + iD;
			}
		}
		return result;
	}

	/**
	 * @since 5.4
	 */
	@Override
	public void shutdown() {
		for (IErrorParser parser : collectionsOfErrorParsers) {
			if (parser instanceof IErrorParser3) {
				((IErrorParser3) parser).shutdown();
			} else if ((parser instanceof ErrorParserNamedWrapper)
					&& (((ErrorParserNamedWrapper) parser).getErrorParser() instanceof IErrorParser3)) {
				((IErrorParser3) ((ErrorParserNamedWrapper) parser).getErrorParser()).shutdown();
			}
		}

		flushPendingErrors();
	}

	public IErrorParser getPreviousErrorParser() {
		return previousErrorParser;
	}

	public void resetPreviousErrorParser() {
		previousErrorParser = null;
	}

	protected IMarkerGenerator getMarkerGenerator() {
		return markerGenerator;
	}
}
