/*******************************************************************************
 * Copyright (c) 2016 EmbSysRegView
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     ravenclaw78 - initial API and implementation
 *******************************************************************************/
package org.eclipse.cdt.embsysregview.views;

import java.util.stream.Stream;

import org.eclipse.cdt.core.model.ICElement;
import org.eclipse.cdt.core.model.ICProject;
import org.eclipse.cdt.debug.core.ICDTLaunchConfigurationConstants;
import org.eclipse.cdt.dsf.datamodel.IDMContext;
import org.eclipse.cdt.dsf.gdb.launching.GdbLaunch;
import org.eclipse.cdt.dsf.service.DsfSession;
import org.eclipse.cdt.embsysregview.Activator;
import org.eclipse.cdt.embsysregview.Messages;
import org.eclipse.cdt.embsysregview.PropertiesKeys;
import org.eclipse.cdt.embsysregview.internal.ui.view.EmbSysRegUi;
import org.eclipse.cdt.embsysregview.internal.ui.view.EmbSysRegViewer;
import org.eclipse.cdt.embsysregview.internal.utils.GUIUtils;
import org.eclipse.cdt.embsysregview.internal.utils.Utils;
import org.eclipse.cdt.embsysregview.properties.IProjectPropertyChangeListener;
import org.eclipse.cdt.embsysregview.properties.ProjectProperties;
import org.eclipse.cdt.embsysregview.properties.PropertiesDialog;
import org.eclipse.cdt.embsysregview.properties.PropertyPageEmbSys;
import org.eclipse.cdt.embsysregview.properties.SettingsUtils;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.PlatformObject;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.ui.AbstractDebugView;
import org.eclipse.debug.ui.DebugUITools;
import org.eclipse.debug.ui.IDebugUIConstants;
import org.eclipse.debug.ui.contexts.DebugContextEvent;
import org.eclipse.debug.ui.contexts.IDebugContextListener;
import org.eclipse.debug.ui.contexts.IDebugContextManager;
import org.eclipse.debug.ui.contexts.IDebugContextService;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.preference.IPreferencePage;
import org.eclipse.jface.preference.PreferenceManager;
import org.eclipse.jface.preference.PreferenceNode;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IPageLayout;
import org.eclipse.ui.IPerspectiveDescriptor;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.ISelectionService;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PerspectiveAdapter;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.navigator.CommonViewer;
import org.eclipse.ui.navigator.resources.ProjectExplorer;
import org.eclipse.ui.part.ViewPart;

/**
 * Generates a View, displaying Registers for an Embedded Device
 */
@SuppressWarnings("restriction")
public class EmbSysRegView extends ViewPart implements DsfSession.SessionEndedListener {
	public static final String ID = "org.eclipse.cdt.embsysregview.views.EmbSysRegView"; //$NON-NLS-1$
	protected EmbSysRegViewer m_viewer;
	private IProject selectedProject;
	/** this changes selection events handling */
	// private DebugEventHandlerProxy eventHandlerProxy;
	private IDebugContextListener fDebugContextListener;
	/** Handles selection when view is on */
	private ISelectionListener projectSelectionListener = (part, selection) -> {
		// is a selection change from some other view
		// in the workbench, which we should pass down to the viewer
		// we also don't want to see selection change from another instances of EmbSys
		if (part.getClass() != EmbSysRegView.this.getClass() && selection instanceof IStructuredSelection) {
			handleWorkbenchSelection((IStructuredSelection) selection);
		}
	};

	public static enum RegSet {

		ALL(true, true, "org.eclipse.cdt.embsysregview.views.AllEmbSysRegView"), //$NON-NLS-1$ 
		EMBSYS(true, false, "org.eclipse.cdt.embsysregview.views.EmbSysRegView"), //$NON-NLS-1$ 
		SPR(false, true, "org.eclipse.cdt.embsysregview.views.SprEmbSysRegView"); //$NON-NLS-1$

		public final boolean embsys;
		public final boolean spr;
		public final String id;

		private RegSet(boolean embsys, boolean spr, String id) {
			this.embsys = embsys;
			this.spr = spr;
			this.id = id;
		}

		public static RegSet byId(String id) {
			return Stream.of(values()).filter(set -> set.id.equals(id)).findAny()
					.orElseThrow(UnsupportedOperationException::new);
		}

		public boolean isSPR() {
			return spr;
		}

	}

	// N.B. both selection listeners may be same, review this.
	/** */
	private IProjectPropertyChangeListener projectPropertiesChangeListener = event -> {
		if (event.getProperties() == null) {
			return;
		}
		final IProject nowProject = event.getProject();
		m_viewer.getEngine().getModel().removeEntry(nowProject);
		if (selectedProject == nowProject) {
			// force re-trigger selection
			selectedProject = null;
			setSelectedProject(nowProject);
		}
	};

	/** */
	private IPropertyChangeListener preferencesChangeListener = event -> {
		if (selectedProject != null) {
			m_viewer.updateView(selectedProject);
		}
	};

	/** */
	private PerspectiveAdapter perspectiveChangeListener = new PerspectiveAdapter() {
		@Override
		public void perspectiveActivated(IWorkbenchPage page, IPerspectiveDescriptor perspectiveDescriptor) {
			super.perspectiveActivated(page, perspectiveDescriptor);
			detectCurrentSelectionInWorkbench();
		}
	};

	private Composite m_parentControl;
	private boolean m_initialized;
	private Action fProjectProperiesPanelAction;

	/**
	 * The constructor.
	 */
	public EmbSysRegView() {
	}

	/**
	 * This is a callback that creates the viewer and initialize it.
	 */
	@Override
	public void createPartControl(final Composite parent) {
		m_parentControl = parent;

		// set listeners
		ProjectProperties.getInstance().addPropertyChangeListener(projectPropertiesChangeListener);
		Activator.getDefault().getPreferenceStore().addPropertyChangeListener(preferencesChangeListener);
		PlatformUI.getWorkbench().getActiveWorkbenchWindow().addPerspectiveListener(perspectiveChangeListener);

		// set up selection handling
		initializeSelectionHandling();

		// initialize viewer control
		initializeViewer();

		IActionBars bars = getViewSite().getActionBars();
		//
		fProjectProperiesPanelAction = new Action() {
			@Override
			public void run() {
				if (selectedProject != null) {
					PropertiesPanelOn(selectedProject);
				}
			}
		};

		fProjectProperiesPanelAction.setText(Messages.EmbSysRegView_5);
		fProjectProperiesPanelAction.setToolTipText(Messages.EmbSysRegView_6);
		fProjectProperiesPanelAction.setImageDescriptor(EmbSysRegUi.getButtonImage());

		bars.getToolBarManager().add(fProjectProperiesPanelAction);
		bars.updateActionBars();

		// handle the case we start after debug session start, so we missed
		// notification.
		// detectCurrentSelectionInWorkbench();

		setupContextListener();
		DsfSession.addSessionEndedListener(this);

		m_initialized = true;
	}

	/** */
	private void setupContextListener() {
		IDebugContextManager contextManager = DebugUITools.getDebugContextManager();
		IDebugContextService contextService = contextManager.getContextService(getSite().getWorkbenchWindow());

		fDebugContextListener = new IDebugContextListener() {

			@Override
			public void debugContextChanged(DebugContextEvent event) {

				if ((event.getFlags() & DebugContextEvent.ACTIVATED) != 0) {
					ISelection s = event.getContext();
					setDebugContext(s);
				}
			}
		};
		contextService.addDebugContextListener(fDebugContextListener);
		setDebugContext(contextService.getActiveContext());
	}

	private void setDebugContext(ISelection activeContext) {
		if (activeContext instanceof IStructuredSelection && !activeContext.isEmpty()) {
			handleWorkbenchSelection((IStructuredSelection) activeContext);
		} else {
			// try out to find selection in workspace
			scanByPartsSelection();
		}
	}

	// --- initialization support ---

	/**
	 * Creates and returns EmbSysRegViewer control.
	 */
	protected EmbSysRegViewer createViewer(Composite parent) {
		return (m_viewer != null) ? m_viewer : new EmbSysRegViewer(this, parent, RegSet.byId(getSite().getId()));
	}

	public EmbSysRegViewer getViewer() {
		return m_viewer;
	}

	/**
	 * Invoked by createPartControl() method when view instance is created.
	 */
	protected void initializeViewer() {
		setViewer(createViewer(m_parentControl));
	}

	/** Sets contained viewer control. */
	protected void setViewer(EmbSysRegViewer viewer) {
		m_viewer = viewer;
		if (m_viewer != null) {
			getSite().setSelectionProvider(m_viewer.getViewer());
		}
	}

	/** Returns whether view has been initialized. */
	public boolean isInitialized() {
		return m_initialized;
	}

	/** shortcut to project properties. */
	private static void PropertiesPanelOn(final IProject project) {
		IPreferencePage page = new PropertyPageEmbSys(project);
		// left side name
		page.setTitle(PropertiesKeys.EMBSYSREGVIEW_NAME);
		PreferenceManager mgr = new PreferenceManager();
		mgr.addToRoot(new PreferenceNode("1", page));
		Shell shell = GUIUtils.getShell();
		String title = "Properties for " + project.getName(); //$NON-NLS-1$
		new PropertiesDialog(title, shell, mgr) {
			{
				create();
				setMessage(page.getTitle());
				open();
			}
		};
	}

	private void initializeSelectionHandling() {
		getSite().getWorkbenchWindow().getSelectionService().addSelectionListener(projectSelectionListener);
	}

	@Override
	public void dispose() {
		ISelectionService selectionService = getSite().getWorkbenchWindow().getSelectionService();
		selectionService.removeSelectionListener(projectSelectionListener);
		ProjectProperties.getInstance().removePropertyChangeListener(projectPropertiesChangeListener);
		Activator.getDefault().getPreferenceStore().removePropertyChangeListener(preferencesChangeListener);
		PlatformUI.getWorkbench().getActiveWorkbenchWindow().removePerspectiveListener(perspectiveChangeListener);
		DsfSession.removeSessionEndedListener(this);
	}

	/**
	 * Passing the focus request to the viewer's control.
	 */
	@Override
	public void setFocus() {
		if (m_viewer != null) {
			m_viewer.setFocus();
		}
	}

	/**
	 * Handles another project selection in workbench.
	 */
	protected void updateView() {
		if (m_viewer != null) {
			m_viewer.updateView(selectedProject);
		}
	}

	private IProject getIProject(Object obj) {
		if (obj instanceof IStructuredSelection) {
			IStructuredSelection selection = (IStructuredSelection) obj;
			Object selectedObject = selection.toArray()[0];
			return getIProject(selectedObject, selection);
		} else {
			return getIProject(obj, null);
		}
	}

	/** */
	private IProject getIProject(Object obj, IStructuredSelection selection) {
		if (obj instanceof IProject) {
			return (IProject) obj;
		}
		// try Platform Object
		if (obj instanceof PlatformObject) {
			PlatformObject platformObject = (PlatformObject) obj;
			ICElement element = platformObject.getAdapter(ICElement.class);
			if (element != null) {
				ICProject cProject = element.getCProject();
				if (cProject != null) {
					return cProject.getProject();
				}
			}
			IResource resource = platformObject.getAdapter(IResource.class);
			if (resource != null) {
				return resource.getProject();
			}
		}
		IResource adapted = Platform.getAdapterManager().getAdapter(obj, IResource.class);
		if (adapted != null) {
			return adapted.getProject();
		}

		// last chance
		ILaunch launch = getILaunch(obj);
		if (launch != null) {
			return getProjectFromLaunch(launch);
		}

		if (selection instanceof ITreeSelection) {
			Object selectionTopParent = ((ITreeSelection) selection).getPaths()[0].getFirstSegment();
			if (selectionTopParent instanceof IProject) {
				return (IProject) selectionTopParent;
			}
		}

		// no more chances: nothing can be retrieved from selected object - return selected project
		return selectedProject;
	}

	protected static IProject getProjectFromLaunch(ILaunch launch) {
		try {
			ILaunchConfiguration launchConfiguration = launch.getLaunchConfiguration();
			if (launchConfiguration == null) {
				return null;
			}
			String projectName = launchConfiguration.getAttribute(
					ICDTLaunchConfigurationConstants.ATTR_PROJECT_NAME, (String) null);
			if (projectName == null || projectName.isEmpty()) {
				return null;
			}
			IPath projectPath = new Path(null, projectName);
			if (projectPath.segmentCount() > 1) {
				projectName = projectPath.lastSegment();
			}
			return Utils.getProjectByName(projectName);
		} catch (Exception e) {
			Activator.log(IStatus.WARNING, e.getMessage(), e);
			return null;
		}
	}

	/**
	 * Focus the view on selected project. input should be null on terminating debug session but kept in selectedProject field
	 */
	protected IStatus setSelectedProject(final IProject project) {
		fProjectProperiesPanelAction.setEnabled(project == null || SettingsUtils.checkProject(project).isOK());
		if (project != selectedProject) {
			if (project != null) {
				selectedProject = project;
				GUIUtils.exec(this::updateView, true);
				return Status.OK_STATUS;
			}
		}
		return Status.CANCEL_STATUS;
	}

	/** Iterate via set of known selection sources. */
	private static ISelection getSelectionFromService() {
		IWorkbenchWindow activeWindow = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		if (activeWindow != null) {
			ISelectionService service = activeWindow.getSelectionService();
			if (service != null) {
				// take selection from Debug View (Debug Perspective)
				ISelection selection = service.getSelection(IDebugUIConstants.ID_DEBUG_VIEW);
				if (selection == null) {
					// or from Project Explorer (C++ Perspective)
					selection = service.getSelection(IPageLayout.ID_PROJECT_EXPLORER);
				}
				return selection;
			}
		}
		return null;
	}

	/**  */
	private boolean handleWorkbenchSelection(final IStructuredSelection selection) {
		boolean result = false;
		// ignore empty selection
		if (!selection.isEmpty()) {
			IProject project = getIProject(selection);
			if (project == null || !project.isAccessible()) {
				cleanViewOnInvalidProjectSelection();
				return false;
			}
			result = setSelectedProject(project).isOK();
			// and finally lock/unlock model (this may be remove debug event handlers)
			setViewerReadonly(project);
		} else {
			if (selectedProject == null || !selectedProject.isAccessible()) {
				cleanViewOnInvalidProjectSelection();
			}
		}
		return result;
	}

	private void cleanViewOnInvalidProjectSelection() {
		selectedProject = null;
		fProjectProperiesPanelAction.setEnabled(false);
		GUIUtils.exec(this::updateView);
	}

	private void setTrackingSession(ILaunch launch) {
		if (launch instanceof GdbLaunch) {
			m_viewer.setTrackingSession(((GdbLaunch) launch).getSession());
		}

	}

	private ILaunch getILaunch(Object obj) {
		if (obj instanceof IAdaptable) {
			return (ILaunch) ((IAdaptable) obj).getAdapter(ILaunch.class);
		}
		return null;
	}

	/** used to find what is selected in LaunchView */
	private void detectCurrentSelectionInWorkbench() {
		// try out selection service
		ISelection selection = getSelectionFromService();
		if (selection instanceof IStructuredSelection && !selection.isEmpty()) {
			handleWorkbenchSelection((IStructuredSelection) selection);
		} else {
			// try out to find selection in workspace
			scanByPartsSelection();
		}
	}

	private void scanByPartsSelection() {
		final IWorkbenchPage activePage = GUIUtils.getActivePage();
		if (activePage != null) {
			// else try by known view
			for (IViewReference viewRef : activePage.getViewReferences()) {
				final IViewPart view = viewRef.getView(false);
				ISelection selection = null;

				// case 1. selection in Debug View
				if (view instanceof AbstractDebugView) {
					AbstractDebugView viewPart = (AbstractDebugView) view;
					Viewer viewer = viewPart.getViewer();
					if (viewer instanceof TreeViewer) {
						selection = viewer.getSelection();
					}
				}

				// case 2. selection in Project Explorer
				if (view instanceof ProjectExplorer) {
					ProjectExplorer viewPart = (ProjectExplorer) view;
					CommonViewer viewer = viewPart.getCommonViewer();
					selection = viewer.getSelection();
				}

				if (selection instanceof IStructuredSelection) {
					IStructuredSelection structuredSelection = (IStructuredSelection) selection;
					if (handleWorkbenchSelection(structuredSelection)) {
						break;
					}
				} else {
					// try out by associated editor
					if (scanByEditor(activePage)) {
						return;
					}
				}
			}

		}
	}

	/**  */
	private boolean scanByEditor(final IWorkbenchPage activePage) {
		final IEditorPart editorPart = activePage.getActiveEditor();
		if (editorPart != null) {
			IResource resource = (IResource) editorPart.getEditorInput().getAdapter(IResource.class);
			if (resource != null) {
				setSelectedProject(resource.getProject());
				return true;
			}
		}
		return false;
	}

	/**
	 * Rules: - works only on Debug Perspective (else, launch == null) - debug context should be available for given selection - launch should be not terminated
	 */
	private void setViewerReadonly(IProject selectedProject) {
		IDMContext cntx = getDebugContext();
		ILaunch launch = getLaunchFromDebugContext(cntx);
		boolean terminated = true;
		IProject project = null;

		if (launch != null) {
			terminated = launch.isTerminated();
			project = getIProject(launch);
			if (project != null) {
				if (selectedProject != null && selectedProject != project) {
					cntx = null;
				}
			}
		}
		final boolean fterminated = terminated;
		final IDMContext con = cntx;
		GUIUtils.exec(() -> m_viewer.setReadOnly(con == null, fterminated));
		setTrackingSession(launch);
		if (con != null && !fterminated) {
			m_viewer.getEngine().getEmbSysRegUpdate(con, false);
		}
	}

	public boolean isSessionTerminated() {
		boolean terminated = true;
		ILaunch launch = getLaunchFromDebugContext(getDebugContext());
		if (launch != null) {
			terminated = launch.isTerminated();
		}
		return terminated;
	}

	private ILaunch getLaunchFromDebugContext(IDMContext cntx) {
		if (cntx != null) {
			return getILaunch(cntx);
		}
		return null;
	}

	private IDMContext getDebugContext() {
		IAdaptable context = DebugUITools.getDebugContext();
		if (context != null) {
			return (IDMContext) context.getAdapter(IDMContext.class);
		}
		return null;
	}

	@Override
	public void sessionEnded(DsfSession session) {
		String id = session.getId();
		if (id.equals(m_viewer.getEngine().getSessionTrackId())) {
			GUIUtils.exec(() -> m_viewer.setReadOnly(true, true));
            m_viewer.getEngine().removeFlagDisableAcsessorReadRegisters();
		}
	}

}
