package org.eclipse.cdt.embsysregview.properties;

import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

import org.eclipse.cdt.core.CProjectNature;
import org.eclipse.cdt.core.model.CoreModel;
import org.eclipse.cdt.core.settings.model.CMacroEntry;
import org.eclipse.cdt.core.settings.model.ICConfigurationDescription;
import org.eclipse.cdt.core.settings.model.ICExternalSetting;
import org.eclipse.cdt.core.settings.model.ICProjectDescription;
import org.eclipse.cdt.core.settings.model.ICSettingEntry;
import org.eclipse.cdt.core.settings.model.ICStorageElement;
import org.eclipse.cdt.core.settings.model.util.CDataUtil;
import org.eclipse.cdt.embsysregview.Activator;
import org.eclipse.cdt.embsysregview.Messages;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Combo;

/**
 * 
 * */
public class SettingsUtils {
	private static final Set<Property> EMPTY_PROPERTY_SET = new HashSet<Property>();
	
	public static final String STORAGE = "org.eclipse.embsys"; //$NON-NLS-1$
	public static final String EMPTY = ""; //$NON-NLS-1$
    
	private SettingsUtils() {}
	
	/**
	 * Writes string property from combo to static property holder.
	 * Do not write same or empty ("") values. 
	 * */
	public static void setPropertyValueFromUI(IProject project, QualifiedName qualifiedName, Combo propertyCombo) {
		try {
			final String oldValue = project.getPersistentProperty(qualifiedName);
			final String newValue = propertyCombo.getText();
			//oldValue may be null if project imported badly or not S32DS project
			if ((oldValue == null && newValue != null) || (!EMPTY.equals(newValue) && !oldValue.equals(newValue))) { 
			  PropertiesHolder.getInstance(project).setPropertyValue(project, qualifiedName, newValue);
			}
		} catch (CoreException e) {
			Activator.log(0, e.getMessage(), e);
		}
	}

	/**
	 * 
	 * */
	public static Set<Property> initializePropertyValues(final IProject project) {

		if (!checkProject(project).isOK()) {
			return EMPTY_PROPERTY_SET;
		}

		PropertiesHolder projectProperties = PropertiesHolder.getInstance(project);
		Set<Property> propertySet = projectProperties.getPropertySet();
		
		if (propertySet.stream().allMatch(p -> projectProperties.scheduleSetFromCProjectJob(project, p).isOK())) {
			projectProperties.saveProjectSettings(project);
		} else {
			propertySet = collectXMLPropertiesFromWorkspace(project);
		}
		
		return propertySet;
	}

	/**
	 * Keep for now if will need to implement possibility to configure
	 * EmbSysRegView both in project settings and in global preferences,
	 * inheritance of global preferences settings by newly created projects, and
	 * possibility to choose what settings to use
	 */
	public static Set<Property> collectXMLPropertiesFromPrefs() {
		Set<Property> propertySet = PropertiesHolder.getInstance(null).getPropertySet();
		IPreferenceStore store = Activator.getDefault().getPreferenceStore();

		for (Property property : propertySet) {
			String name = property.getName();
			String value = store.getString(name);
			property.setValue(value);
		}

		return propertySet;
	}
		
	/** 
	 *
	 * */
	public static IStatus checkProject(final IProject project) {
		if (project != null) {
			try {
				if (project.isOpen() && project.isNatureEnabled(CProjectNature.C_NATURE_ID)) {
					return Status.OK_STATUS;
				}
			} catch (CoreException e) {
				Activator.log(0, e.getMessage(), e);
				return e.getStatus();
			}
		}
		return new Status(IStatus.INFO, Activator.PLUGIN_ID, "Project in not good for any use");
	}

	/**
	 * stores property in .cproject file
	 * 
	 * @param project
	 * @param settingQualifier
	 * @param value
	 * @param monitor
	 * @return
	 */
	public static IStatus storePropertyInCProject(IProject project, String settingQualifier, String value,
			IProgressMonitor monitor) {
        IStatus status = checkProject(project);
		if (!status.isOK()) {
			return status;
		}
		final ICProjectDescription projectDescription = CoreModel.getDefault().getProjectDescription(project, true);
		if (projectDescription == null) {
			return Status.CANCEL_STATUS;
		}
		ICSettingEntry[] entries = clearPreviousSettings(projectDescription, settingQualifier, value);

		if (entries == null) {
			CMacroEntry settingEntry = CDataUtil.createCMacroEntry(settingQualifier, value, ICSettingEntry.NONE);
			entries = new ICSettingEntry[] { settingEntry };
		}
		try {
			ICStorageElement embsysStorage = projectDescription.getStorage(STORAGE, true);
			for (ICSettingEntry entry : entries) {
				embsysStorage.setAttribute(entry.getName(), entry.getValue());
			}

			CoreModel.getDefault().setProjectDescription(project, projectDescription, false, monitor);

		} catch (CoreException e) {
			Activator.log(0, e.getMessage(), e);
		}

		Activator.log(IStatus.INFO, NLS.bind(Messages.property_was_saved_in_cproject, settingQualifier));

		return Status.OK_STATUS;
	}

	/**
	 * Get property from a .cproject file
	 * 
	 * @param configurationId
	 *            - supposed id of configuration where settings are stored
	 * @param project
	 *            - project for which settings are obtained
	 * @param qualifiedName
	 *            - qualified name of the setting
	 * @return Property object
	 * 
	 */
	public static IStatus setPropertyValueFromCProject(IProject project, Property property) {
		IStatus status = checkProject(project);
		if (!status.isOK()) {
			return status;
		}
		
		final String qualifier = property.getQualifier();
		String message = NLS.bind(Messages.setting_was_not_found_no_project, qualifier);
		status = new Status(IStatus.WARNING, Activator.PLUGIN_ID, message);

		ICProjectDescription projectDescription = CoreModel.getDefault().getProjectDescription(project, true);
		if (projectDescription == null) {
			String projectName = project.getName();
			message = NLS.bind(Messages.setting_was_not_found_in_cproject_file, qualifier, projectName);
			return new Status(IStatus.WARNING, Activator.PLUGIN_ID, message);
		}
		
		try {
			//first try primary location
			status = handleProjectSetting(project, property);
			if (status.isOK()) {
				return status;
			}
			//try other locations for backward compatibility
			ICConfigurationDescription configuration = projectDescription.getActiveConfiguration();
			status = handleExternalSetting(project, configuration, property);
			if (status.isOK()) {
				return status;
			}

			ICConfigurationDescription[] configurations = projectDescription.getConfigurations();
			for (ICConfigurationDescription des : configurations) {
				status = handleExternalSetting(project, des, property);
				if (status.isOK()) {
					return status;
				}
			}			

		} catch (CoreException e) {
			message = NLS.bind(Messages.SettingsUtils_exception, e.getMessage());
			Activator.log(0, e.getMessage(), e);
			return new Status(IStatus.WARNING, Activator.PLUGIN_ID, message);
		}

		return status;
	}

	private static ICSettingEntry[] clearPreviousSettings(ICProjectDescription projectDescription,
			String settingQualifier, String value) {
		ICSettingEntry[] entries = null;
		ICConfigurationDescription configuration = projectDescription.getActiveConfiguration();
		entries = handleExternalSettings(configuration, settingQualifier, value);

		if (entries == null) {
			ICConfigurationDescription[] configurations = projectDescription.getConfigurations();
			for (ICConfigurationDescription des : configurations) {
				entries = handleExternalSettings(des, settingQualifier, value);
			}
		}

		return entries;
	}

	private static ICSettingEntry[] handleExternalSettings(ICConfigurationDescription configuration,
			String settingQualifier, String value) {
		if (value == null || value.isEmpty()) {
			return null;
		}
		ICExternalSetting[] externalSettings = configuration.getExternalSettings();
		for (ICExternalSetting setting : externalSettings) {
			ICSettingEntry[] entries = setting.getEntries();
			for (ICSettingEntry entry : entries) {
				if (settingQualifier.equals(entry.getName())) {
					String oldValue = entry.getValue();
					if (value.equals(oldValue)) {
						return entries;
					}
					configuration.removeExternalSetting(setting);
					return updateEntry(setting, settingQualifier, value);
				}
			}
		}
		return null;
	}

	private static ICSettingEntry[] updateEntry(ICExternalSetting setting, String settingQualifier, String value) {
		ICSettingEntry[] entries = setting.getEntries();
		for (int i = 0; i < entries.length; i++) {
			if (settingQualifier.equals(entries[i].getName())) {
				entries[i] = CDataUtil.createCMacroEntry(settingQualifier, value, ICSettingEntry.NONE);
			}
		}

		return entries;
	}	

	private static IStatus handleProjectSetting(final IProject project, final Property property) throws CoreException {
		final String name = project.getName();
		final String qualifier = property.getQualifier();
		final ICProjectDescription cd = CoreModel.getDefault().getProjectDescription(project);
		final ICStorageElement elem = cd.getStorage(STORAGE, false);
		if (elem != null) {
			final String attr = elem.getAttribute(qualifier);
			if (attr != null) {
				property.setValue(attr);
				return Status.OK_STATUS;
			}
		}
		String message = NLS.bind(Messages.setting_was_not_found_in_cproject_file, qualifier, name);
		return new Status(IStatus.WARNING, Activator.PLUGIN_ID, message);
	}

	private static IStatus handleExternalSetting(IProject project, ICConfigurationDescription configuration,
			Property property) {
		String qualifier = property.getQualifier();

		for (ICExternalSetting setting : configuration.getExternalSettings()) {
			Optional<ICSettingEntry> entry = Stream.of(setting.getEntries()).filter(e -> qualifier.equals(e.getName())).findAny();
			if (entry.isPresent()) {
				property.setValue(entry.get().getValue());					
				return Status.OK_STATUS;
			}
		}

		if (property.isMandatory()) {
			return new Status(IStatus.WARNING, Activator.PLUGIN_ID, NLS.bind(Messages.setting_was_not_found_in_cproject_file, qualifier, project.getName()));
		} else {
			return Status.OK_STATUS;
		}
	}
	
	/**
	 * for configuration EmbSysRegView only in project properties
	 * 
	 * @param project
	 * @return
	 */
	private static Set<Property> collectXMLPropertiesFromWorkspace(IProject project) {
		PropertiesHolder propertiesHolder = PropertiesHolder.getInstance(project);
		Set<Property> propertySet = propertiesHolder.getPropertySet();
		try {
			for (Property property : propertySet) {
				QualifiedName qualifiedName = property.getQualifiedName();
				String value = project.getPersistentProperty(qualifiedName);
				if (value == null || value.isEmpty()) {
					return propertySet;
				}
				property.setValue(value);
				propertiesHolder.scheduleUpdateCProjectJob(project, qualifiedName, value);
			}
		} catch (CoreException e) {
			Activator.log(0, e.getMessage(), e);
		}

		return propertySet;
	}
}
