package org.eclipse.cdt.embsysregview.properties;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.cdt.embsysregview.Activator;
import org.eclipse.cdt.embsysregview.Messages;
import org.eclipse.cdt.embsysregview.PropertiesKeys;
import org.eclipse.cdt.embsysregview.internal.IEmbSysChipInfo;
import org.eclipse.cdt.embsysregview.internal.RegisterXMLCollector;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.PlatformObject;
import org.eclipse.jface.fieldassist.ControlDecoration;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.PropertyPage;
import org.jdom2.Attribute;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.input.SAXBuilder;
import org.jdom2.input.sax.XMLReaders;
import org.osgi.framework.Bundle;

/**
 * 
 * */
public class PropertyPageEmbSys extends PropertyPage { 
	private static final String E200_ID = "e200";
//implements IWorkbenchPropertyPage
    
    private Combo architecture;
    private Combo vendor;
    private Combo chip;
    private Combo board;
    private Combo core;
    
    private Label coreLabel;
    private Text descriptionText;
    private Map<String, String> registerDescriptionPaths = new HashMap<>();// bundle
                                                                           // name,
    @SuppressWarnings("unused")
    private String selArchitecture, selVendor, selChip, selBoard, selCore;
                                                                           // name
    private IProject selectedProject;
    private ControlDecoration archDecoration;
    private ControlDecoration vendorDecoration;
    private ControlDecoration chipDecoration;
    private ControlDecoration coreDecoration;
    private Composite composite;    
    
    /**
     * Used for create page from drop-down properties menu.
     * N.B. getElement() returns null at this point  
     * */
    public PropertyPageEmbSys() {
        super();                
        selectedProject = null;
    }
    /**
     * Used for create page directly from command or action.
     * */
    public PropertyPageEmbSys(final IProject project) {
        super();                        
        selectedProject = project;
    }

    @Override
    public boolean performOk() {
        PlatformObject platformObject = (PlatformObject) getElement();
        IProject project = selectedProject;
        if (platformObject != null) {
            project = (IProject) platformObject.getAdapter(IProject.class);
        }

        Map<String, Property> propertyMap = PropertiesHolder.getInstance(project).getPropertyMap();
        SettingsUtils.setPropertyValueFromUI(project, propertyMap.get(PropertiesKeys.ARCHITECTURE).getQualifiedName(), architecture);
        SettingsUtils.setPropertyValueFromUI(project, propertyMap.get(PropertiesKeys.VENDOR).getQualifiedName(), vendor);
        SettingsUtils.setPropertyValueFromUI(project, propertyMap.get(PropertiesKeys.CHIP).getQualifiedName(), chip);
        SettingsUtils.setPropertyValueFromUI(project, propertyMap.get(PropertiesKeys.BOARD).getQualifiedName(), board);
        SettingsUtils.setPropertyValueFromUI(project, propertyMap.get(PropertiesKeys.CORE).getQualifiedName(), core);
        //
        ProjectProperties.getInstance().firePropertiesChangeEvent(project);
        // if arch+vendor+chip is selected then return ok...
        return true;                
    }

    @Override
    protected Control createContents(Composite parent) {
    	// There will be no default, therefore remove default button
        noDefaultAndApplyButton();
        
        //back-up code in order to have selectedProject initialized as property page may be created via default constructor 
        if (selectedProject == null) {
        	IAdaptable elem = getElement();
        	if (elem != null) {
        	 selectedProject = (IProject) elem.getAdapter(IProject.class);
        	}
        }
                        
        setDescription(Messages.settings_page_description);
        
        registerDescriptionPaths = new RegisterXMLCollector().collectPaths();
        composite = new Composite(parent, SWT.NONE);
        Composite left = new Composite(composite, SWT.NONE);
        Composite right = new Composite(composite, SWT.NONE);

        GridLayout gridLayout = new GridLayout(2, false); 
        gridLayout.horizontalSpacing = 10;
        composite.setLayout(gridLayout);

        GridData leftGridData = new GridData(SWT.LEFT, SWT.FILL, false, true);
        leftGridData.widthHint = 180;
        left.setLayoutData(leftGridData);
        left.setLayout(new GridLayout(1, false));        

        right.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
        right.setLayout(new FillLayout());

        final Label architectureLabel = new Label(left, SWT.LEFT);
        architectureLabel.setText("Architecture:"); //$NON-NLS-1$
        architecture = new Combo(left, SWT.DROP_DOWN | SWT.READ_ONLY);
        architecture.setVisibleItemCount(10);
        architecture.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

        final Label vendorLabel = new Label(left, SWT.LEFT);
        vendorLabel.setText("Vendor:"); //$NON-NLS-1$
        vendor = new Combo(left, SWT.DROP_DOWN | SWT.READ_ONLY);
        vendor.setVisibleItemCount(10);
        vendor.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

        final Label chipLabel = new Label(left, SWT.LEFT);
        chipLabel.setText("Chip:"); //$NON-NLS-1$
        chip = new Combo(left, SWT.DROP_DOWN | SWT.READ_ONLY);
        chip.setVisibleItemCount(20);
        chip.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

        coreLabel = new Label(left, SWT.LEFT);
        coreLabel.setText("e200 Core:"); //$NON-NLS-1$
        coreLabel.setVisible(false);
        core = new Combo(left, SWT.DROP_DOWN | SWT.READ_ONLY);
        core.setVisibleItemCount(10);
        core.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
        core.setVisible(false);
        
        
        final Label boardLabel = new Label(left, SWT.LEFT);
        boardLabel.setText("Board:"); //$NON-NLS-1$
        board = new Combo(left, SWT.DROP_DOWN | SWT.READ_ONLY);
        board.setVisibleItemCount(10);
        board.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
       
        //right pane
        final Group descriptionGroup = new Group(right, SWT.NONE);
        descriptionGroup.setText("Chip description"); //$NON-NLS-1$
        FillLayout descriptionLayout = new FillLayout();
        descriptionLayout.marginHeight = 10;
        descriptionLayout.marginWidth = 10;
        descriptionGroup.setLayout(descriptionLayout); //$NON-NLS-1$

        descriptionText = new Text(descriptionGroup, SWT.H_SCROLL | SWT.V_SCROLL | SWT.MULTI | SWT.READ_ONLY | SWT.WRAP);
        descriptionText.setText(""); //$NON-NLS-1$
        FontData[] fD = descriptionText.getFont().getFontData();
        fD[0].setName("Lucida Console"); //$NON-NLS-1$
        Font f = new Font(Display.getCurrent(), fD[0]);
        descriptionText.setFont(f);

        architecture.setSize(500, 30);

        vendor.setSize(500, 30);
        vendor.setEnabled(false);

        chip.setSize(500, 30);
        chip.setEnabled(false);

        board.setSize(500, 30);
        board.setEnabled(false);
        
        core.setSize(500,30);
        core.setEnabled(false);
        
        fillArchitecture();
        addDecorations();

        architecture.addSelectionListener(new SelectionAdapter() {
            @Override
			public void widgetSelected(SelectionEvent event) {
                String newArchitecture = architecture.getItem(architecture.getSelectionIndex());
                if (!newArchitecture.equals(selArchitecture)) {
                    selArchitecture = newArchitecture;
                    hideDecoration(architecture, archDecoration);
                    fillVendor(newArchitecture);
                }
            }
        });

        vendor.addSelectionListener(new SelectionAdapter() {
            @Override
			public void widgetSelected(SelectionEvent event) {
                String newVendor = vendor.getItem(vendor.getSelectionIndex());
                if (!newVendor.equals(selVendor)) {
                    selVendor = newVendor;
                    hideDecoration(vendor, vendorDecoration);
                    fillChip(architecture.getItem(architecture.getSelectionIndex()), newVendor);
                }
            }
        });
               
        chip.addSelectionListener(new SelectionAdapter() {
            @Override
			public void widgetSelected(SelectionEvent event) {
                String newChip = chip.getItem(chip.getSelectionIndex());
                if (!newChip.equals(selChip)) {
                    selChip = newChip;
                    hideDecoration(chip, chipDecoration);
                    // end up for non e200
                    fillCore(architecture.getText(), vendor.getText(), chip.getText());
                    fillBoard(architecture.getItem(architecture.getSelectionIndex()), vendor.getItem(vendor
                            .getSelectionIndex()), newChip);
                }
            }
        });
        
        core.addSelectionListener(new SelectionAdapter () {
        	@Override
			public void widgetSelected(SelectionEvent event) {
                String newCore = core.getItem(core.getSelectionIndex());
                if (!newCore.equals(selCore)) {
                    selCore = newCore;
                    hideDecoration(core, coreDecoration);
                }
            }
        });

        parent.pack();
        return composite;
    }

    @Override
    public void dispose() {
        super.dispose();
        disposeWithChildren(composite);
        disposeDecorations();
    }
            
    private void addDecorations() {
        if (architecture.getSelectionIndex() == -1) {
            archDecoration = showDecoration(architecture, archDecoration);
        }

        if (vendor.getSelectionIndex() == -1) {
            vendorDecoration = showDecoration(vendor, vendorDecoration);
        }

        if (chip.getSelectionIndex() == -1) {
            chipDecoration = showDecoration(chip, chipDecoration);
        }
        
        if (core.getSelectionIndex() == -1) {
        	coreDecoration = showDecoration(core, coreDecoration);        	
        } 
    }

    private void fillArchitecture() {
        for (String entry : getDirList("", "*")) //$NON-NLS-1$ //$NON-NLS-2$
            if (!PropertiesKeys.DTD_FILENAME.equals(entry)) 
                this.architecture.add(entry);        
        restoreStoredSettings();
    }

    private void restoreStoredSettings() {
        PlatformObject platformObject = (PlatformObject) getElement();
        IProject project = selectedProject;
        if (platformObject != null) {
            project = (IProject) platformObject.getAdapter(IProject.class);
        }

        Set<Property> propertySet = SettingsUtils.initializePropertyValues(project);
        if (propertySet.isEmpty()) {        	
        	return;
        }
        
        for (Property property : propertySet) {
            if (property.getValue() == null) {
                return;
            }
        }

        fillControls(project);
    }

    private void fillControls(final IProject project) {
        Map<String, String> props = PropertiesHolder.getInstance(project).getValueMap();
        String store_architecture = props.get(PropertiesKeys.ARCHITECTURE);
        String store_vendor = props.get(PropertiesKeys.VENDOR);
        String store_chip = props.get(PropertiesKeys.CHIP);
        String store_board = props.get(PropertiesKeys.BOARD);
        String store_core = props.get(PropertiesKeys.CORE);
        
        int index = architecture.indexOf(store_architecture);
        if (index != -1) {
        	architecture.setEnabled(true);
            architecture.select(index);
            selArchitecture = store_architecture;
            hideDecoration(architecture, archDecoration);            
            fillVendor(store_architecture);
            //
            vendor.setEnabled(true);
            index = vendor.indexOf(store_vendor);
            if (index != -1) {
                vendor.select(index);
                selVendor = store_vendor;
                hideDecoration(vendor, vendorDecoration);
                fillChip(store_architecture, store_vendor);
                //
                chip.setEnabled(true);
                index = chip.indexOf(store_chip);
                if (index != -1) {
                    chip.select(index);
                    selChip = store_chip;
                    hideDecoration(chip, chipDecoration);
                    if (fillCore(store_architecture, store_vendor, store_chip)) {
                    	index = core.indexOf(store_core);
                    	if (index != -1) {
                    		core.select(index);
                    		hideDecoration(core, coreDecoration); 
                            selCore = store_core;
                    	}
                    }                     
                    fillBoard(store_architecture, store_vendor, store_chip);
                    index = board.indexOf(store_board);
                    if (index != -1) {
                        board.select(index);
                        selBoard = store_board;
                    }
                }
            } else {
                vendor.setText(""); //$NON-NLS-1$
            }
        } else {
            architecture.setText(""); //$NON-NLS-1$
        }
    }
    
    /** */ 
    private boolean fillCore(String selectedArchitecture, String selectedVendor, String selectedChip) {
		if (E200_ID.equals(selectedArchitecture)) {
			core.setEnabled(true);
			core.removeAll();
			coreDecoration = showDecoration(core, coreDecoration);
			// check hardware by extension point here
			IEmbSysChipInfo chipInfo = new DefaultChipInfo();
			Set<String> coreSet = chipInfo.getChipCores(selectedChip);			
			for (String entry : getDirList("/" + selectedArchitecture + "/" + selectedVendor, "*.spr")) {//$NON-NLS-1$ //$NON-NLS-2$				
                //add the only cores given chip contains
				final String coreFile = entry.substring(0, entry.lastIndexOf('.'));
				if (coreSet.contains(coreFile)) {  			
	               core.add(coreFile);
                }
	        }
			if (coreSet.size() == 1) {
                core.setText(coreSet.iterator().next());
                hideDecoration(core, coreDecoration);
			} else {
                core.setText(""); //$NON-NLS-1$
                selCore = null;
			}
		  return true;
		} 
		return false;
	}
    
    /** */
	private void fillVendor(String selectedArchitecture) {
    	if (E200_ID.equals(selectedArchitecture)) {
    		core.setVisible(true); 
    		core.setEnabled(false);    		 
    		core.setText("");
    		core.removeAll();
    		coreLabel.setVisible(true);
    	} else {
    		core.setVisible(false);
    		coreLabel.setVisible(false);
    	}
        chip.setEnabled(false);
        board.setEnabled(false);
        vendor.setEnabled(true);
        chip.setText(""); //$NON-NLS-1$
        board.setText(""); //$NON-NLS-1$
        vendor.removeAll();
        chip.removeAll();
        board.removeAll();
        vendorDecoration = showDecoration(vendor, vendorDecoration);
        chipDecoration = showDecoration(chip, chipDecoration);
        coreDecoration = showDecoration(core, coreDecoration);

        for (String entry : getDirList("/" + selectedArchitecture, "*")) {//$NON-NLS-1$ //$NON-NLS-2$
            vendor.add(entry);
        }

        vendor.setText(""); //$NON-NLS-1$
        descriptionText.setText(""); //$NON-NLS-1$
        selVendor = null;
    }

    private void fillChip(String selectedArchitecture, String selectedVendor) {
        board.setEnabled(false);
        board.setText(""); //$NON-NLS-1$
        chip.setEnabled(true);
        chip.removeAll();
        core.setEnabled(false);
        core.removeAll();
        board.removeAll();
        
        chipDecoration = showDecoration(chip, chipDecoration);
        coreDecoration = showDecoration(core, coreDecoration);

        for (String entry : getDirList("/" + selectedArchitecture + "/" + selectedVendor, "*.xml")) { //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
            chip.add(entry.substring(0, entry.length() - 4));
        }

        chip.setText(""); //$NON-NLS-1$
        descriptionText.setText(""); //$NON-NLS-1$
        selChip = null;
    }

    /** */
    private void fillBoard(String selectedArchitecture, String selectedVendor, String selectedChip) {
        board.removeAll();
        board.add("---  none ---"); //$NON-NLS-1$
        // Check if boards are listed ...
        SAXBuilder builder = new SAXBuilder(XMLReaders.NONVALIDATING);
        //builder.setValidation(false);

        for (String bundleName : registerDescriptionPaths.keySet()) {
            String directory = registerDescriptionPaths.get(bundleName);
            Bundle bundle = Platform.getBundle(bundleName);
            URL fileURL = bundle.getEntry(
                    directory + "/" + selectedArchitecture + "/" + selectedVendor + "/" + selectedChip + ".xml"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
            if (fileURL == null) {
                continue;
            }

            Document doc;
            boolean containsBoards = false;
            try {
                doc = builder.build(fileURL);
                Element root = doc.getRootElement();
                if (!PropertiesKeys.DEVICE.equals(root.getName())) {
                    List<Element> grouplist = root.getChildren("boards"); //$NON-NLS-1$
                    for (Element group : grouplist) {
                        List<Element> boardlist = group.getChildren();
                        for (Element boardElement : boardlist) {
                            containsBoards = true;
                            Attribute attr_bname = boardElement.getAttribute("id"); //$NON-NLS-1$
                            String bname;
                            if (attr_bname != null) {
                                bname = attr_bname.getValue();
                            } else {
                                bname = "-1"; //$NON-NLS-1$
                            }
                            board.add(bname);
                        }
                    }
                    Element chip_description = root.getChild("chip_description"); //$NON-NLS-1$
                    if (chip_description != null) {
                        descriptionText.setText(chip_description.getText());
                    }
                } else {
                    Element chip_description = root.getChild(PropertiesKeys.DESCRIPTION); 
                    if (chip_description != null) {
                        this.descriptionText.setText(chip_description.getText());
                    }
                }
            } catch (JDOMException e) {
            	Activator.log(0, e.getMessage(), e);
                this.descriptionText.setText(Messages.file_SVD_bad);                
            } catch (IOException e) {
            	Activator.log(0, e.getMessage(), e);
                this.descriptionText.setText(Messages.file_SVD_bad);                
            }

            // enable only if there are boards to show
            if (containsBoards) {
                board.setEnabled(true);
            } else {
                board.setEnabled(false);
            }

            if (board.getItemCount() > 0) {
                board.setText(board.getItem(0));
            } else {
                board.setText(""); //$NON-NLS-1$
                selBoard = null;
            }
        }

    }

    private List<String> getDirList(String path, String pattern) {
        List<String> dirList = new ArrayList<>();

        for (String bundleName : registerDescriptionPaths.keySet()) {
            Bundle bundle = Platform.getBundle(bundleName);
            String directory = registerDescriptionPaths.get(bundleName);
            String fullPath = directory + path;
            Enumeration<URL> entries = bundle.findEntries(fullPath, pattern, false);

            if (entries != null) {
                while (entries.hasMoreElements()) {
                    URL entry = entries.nextElement();

                    File x = new File(entry.getFile());
                    try {
                        String filename = x.getCanonicalFile().getName();
                        if (!filename.startsWith(".") && !dirList.contains(filename)) { //$NON-NLS-1$
                            dirList.add(filename);
                        }
                    } catch (IOException e) {
                    	Activator.log(0, e.getMessage(), e);
                    }
                }
            }

        }

        return dirList;
    }

    private static ControlDecoration createErrorDecoration(Control control) {
        ControlDecoration decoration = new ControlDecoration(control, SWT.LEFT | SWT.TOP);
        decoration.setDescriptionText(Messages.select_value_message);
        int margin = 2;
        decoration.setMarginWidth(margin);
        ImageDescriptor descriptor = PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(
                ISharedImages.IMG_DEC_FIELD_ERROR);
        Image image = descriptor.createImage();
        decoration.setImage(image);
        return decoration;
    }

    private void disposeWithChildren(Control control) {
        if (control.isDisposed()) {
            return;
        }

        if (control instanceof Composite) {
            for (Control child : ((Composite) control).getChildren()) {
                disposeWithChildren(child);
            }
        }

        control.dispose();
    }

    private static void disposeDecoration(ControlDecoration decoration) {
        if (decoration == null) {
            return;
        }
        decoration.hide();
        Image image = decoration.getImage();
        decoration.dispose();
        if (!image.isDisposed()) {
            image.dispose();
        }
    }

    private static ControlDecoration showDecoration(Control control, ControlDecoration decoration) {
    	if (!control.isVisible()) {
    		return null;
    	}
        if (decoration != null) {
            decoration.show();
            return decoration;
        }
		return createErrorDecoration(control);
    }

    private void hideDecoration(Combo control, ControlDecoration decoration) {
        if (control.getSelectionIndex() != -1 && decoration != null) {
            decoration.hide();
        }        
    }

    private void disposeDecorations() {
        disposeDecoration(archDecoration);
        disposeDecoration(vendorDecoration);
        disposeDecoration(chipDecoration);
        disposeDecoration(coreDecoration);
    }

}
