package org.eclipse.cdt.embsysregview.properties;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.eclipse.cdt.embsysregview.Activator;
import org.eclipse.cdt.embsysregview.Messages;
import org.eclipse.cdt.embsysregview.PropertiesKeys;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.ILock;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.osgi.util.NLS;

public class PropertiesHolder {
    private static Map<String, PropertiesHolder> instances = new HashMap<>();
    private ILock lock;

    public static PropertiesHolder getInstance(IProject project) {
        String key = project == null ? "common_properties" : project.getName();
        PropertiesHolder localInstance = instances.get(key);
        if (localInstance == null) {
            synchronized (PropertiesHolder.class) {
                localInstance = instances.get(key);
                if (localInstance == null) {
                    localInstance = new PropertiesHolder(project);
                    instances.put(key, localInstance);
                }
            }
        }

        return localInstance;
    }

    @SuppressWarnings("serial")
	private final Set<Property> properties = Collections.unmodifiableSet(new HashMap<String, Boolean>() {{		    		   
	    	Stream.of(PropertiesKeys.ARCHITECTURE, PropertiesKeys.VENDOR, PropertiesKeys.CHIP).forEach(p -> put(p, true));
	    	Stream.of(PropertiesKeys.BOARD, PropertiesKeys.CORE).forEach(p -> put(p, false));
	    }}
	    .entrySet()
	    .stream()	    
	    .map(e -> new Property(new QualifiedName(PropertiesKeys.REGISTER_PREFIX + e.getKey(), e.getKey()), e.getValue()))
	    .collect(Collectors.toSet()));
    /**
	 * @param project not used yet 
	 */
    private PropertiesHolder(IProject project) {
    }

    public Set<Property> getPropertySet() {
        return properties;
    }

    public Map<String, String> getValueMap() {
    	return properties.stream().collect(Collectors.toMap(Property::getName, Property::getValue));
    }
    
    public Map<String, Property> getPropertyMap() {
        return properties.stream().collect(Collectors.toMap(Property::getName, Function.identity()));
    }
    
    /** */
    public void setPropertyValue(IProject project, QualifiedName qualifiedName, String newValue) {
        Property property = getPropertyMap().get(qualifiedName.getLocalName());

        try {
            project.setPersistentProperty(qualifiedName, newValue);
            scheduleUpdateCProjectJob(project, qualifiedName, newValue);
            property.setValue(newValue);
        } catch (CoreException e) {
          Activator.log(IStatus.ERROR, "Write property problem", e);   //$NON-NLS-1$
        }
    }

    private Job storeSettingsJob(Function<IProgressMonitor, IStatus> func) {
        if (lock == null) {
            lock = Job.getJobManager().newLock();
        }

        return new Job(PropertiesKeys.store_settings_job_name) {
        	{
                setSystem(true);
                schedule();
        	}
        	
            @Override
            protected IStatus run(IProgressMonitor monitor) {
                try {
                    lock.acquire();
                    return func.apply(monitor);
                } finally {
                    lock.release();
                }
            }
        };
    }
    
    public void scheduleUpdateCProjectJob(IProject project, QualifiedName qualifiedName, String value) {
    	storeSettingsJob(m -> SettingsUtils.storePropertyInCProject(project, qualifiedName.getQualifier(), value, m));
    }

    public IStatus scheduleSetFromCProjectJob(IProject project, Property property) {
        Job j = storeSettingsJob(m -> SettingsUtils.setPropertyValueFromCProject(project, property));

        try {
            j.join();
        } catch (InterruptedException e) {
            Activator.log(IStatus.ERROR, e.getMessage(), e);
        }
        if ((j.getResult() != null && j.getResult().isOK()) || !property.isMandatory()) {
            return new Status(IStatus.OK, Activator.PLUGIN_ID, NLS.bind(Messages.property_value_was_set_from_cproject, property.getQualifier(), project.getName()));
        } else {
        	return new Status(IStatus.WARNING, Activator.PLUGIN_ID, NLS.bind(Messages.no_settings_were_found_in_cproject_file, property.getQualifier(), project.getName()));
        }
    }
    /**
     * 
     * */ 
    public void saveProjectSettings(IProject project) {
        try {
            for (Property property : properties) {
            	project.setPersistentProperty(property.getQualifiedName(), property.getValue());                
            }
        } catch (CoreException e) {
            Activator.log(0, e.getMessage(), e);
        }

    }

}
