/*******************************************************************************
 * Copyright (c) 2016 EmbSysRegView
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     ravenclaw78 - initial API and implementation
 *******************************************************************************/
package org.eclipse.cdt.embsysregview.preferences;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

import net.miginfocom.swt.MigLayout;

import org.eclipse.cdt.embsysregview.Activator;
import org.eclipse.cdt.embsysregview.PropertiesKeys;
import org.eclipse.cdt.embsysregview.internal.RegisterXMLCollector;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;
import org.jdom2.Attribute;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.input.SAXBuilder;
import org.jdom2.input.sax.XMLReaders;

/**
 * This class represents a preference page that is contributed to the
 * Preferences dialog. By subclassing <samp>FieldEditorPreferencePage</samp>, we
 * can use the field support built into JFace that allows us to create a page
 * that is small and knows how to save, restore and apply itself.
 * <p>
 * This page is used to modify preferences only. They are stored in the
 * preference store that belongs to the main plug-in class. That way,
 * preferences can be accessed directly via the preference store.
 */

public class PreferencePageEmbSys extends PreferencePage implements IWorkbenchPreferencePage {
    Combo architecture;
    Combo vendor;
    Combo chip;
    Combo board;
    Text descriptionText;

    private Map<String, String> registerDescriptionPaths = new HashMap<>(); // bundle,
                                                                                          // directory

    public PreferencePageEmbSys() {
        super();
        this.noDefaultAndApplyButton();
        setPreferenceStore(Activator.getDefault().getPreferenceStore());
        setDescription("A Periperal Register View for embedded system"); //$NON-NLS-1$
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.eclipse.ui.IWorkbenchPreferencePage#init(org.eclipse.ui.IWorkbench)
     */
    @Override
	public void init(IWorkbench workbench) {
        registerDescriptionPaths = new RegisterXMLCollector().collectPaths();
    }

    @Override
    public boolean performOk() {
        if (chip != null && chip.getSelectionIndex() == -1) {            
            return false;           
        }
        IPreferenceStore store = getPreferenceStore();
        if(architecture != null && vendor != null && chip != null) {
            if (architecture.getSelectionIndex() != -1)
                store.setValue(PropertiesKeys.ARCHITECTURE, architecture.getItem(architecture.getSelectionIndex()));
            else
                store.setValue(PropertiesKeys.ARCHITECTURE, ""); //$NON-NLS-1$
  
            if (vendor.getSelectionIndex() != -1)
                store.setValue(PropertiesKeys.VENDOR, vendor.getItem(vendor.getSelectionIndex()));
            else
                store.setValue(PropertiesKeys.VENDOR, ""); //$NON-NLS-1$

            if (chip.getSelectionIndex() != -1)
                store.setValue(PropertiesKeys.CHIP, chip.getItem(chip.getSelectionIndex()));
            else
                store.setValue(PropertiesKeys.CHIP, ""); //$NON-NLS-1$
            // if arch+vendor+chip is selected then return ok ...
            return architecture.getSelectionIndex() != -1 && vendor.getSelectionIndex() != -1
                    && chip.getSelectionIndex() != -1;
        }        
        if (board != null) {
            if (board.getSelectionIndex() != -1) {
                store.setValue(PropertiesKeys.BOARD, board.getItem(board.getSelectionIndex()));
            } else {
                store.setValue(PropertiesKeys.BOARD, ""); //$NON-NLS-1$
            }
        }
           
        return true;
    }

    @SuppressWarnings("serial")
	private List<String> getDirList(String path, String pattern) {
        return new ArrayList<String>() {{
	        for (String bundleName : registerDescriptionPaths.keySet()) {
	            String fullPath = registerDescriptionPaths.get(bundleName) + path;
	            Enumeration<URL> entries = Platform.getBundle(bundleName).findEntries(fullPath, pattern, false);
	
	            if (entries != null) {
	                while (entries.hasMoreElements()) {
	                    URL entry = entries.nextElement();
	
	                    try {
	                        String filename = new File(entry.getFile()).getCanonicalFile().getName();
	                        if (!filename.startsWith(".") && !contains(filename)) { //$NON-NLS-1$
	                            add(filename);
	                        }
	                    } catch (IOException e) {
	                        Activator.log(0, e.getMessage(), e);
	                    }
	                }
	            }
	        }
        }};
    }

    private void restoreStoredSettings() {
        // Try to fill out Combos with values from the Store
        IPreferenceStore store = getPreferenceStore();
        String store_architecture = store.getString(PropertiesKeys.ARCHITECTURE);
        String store_vendor = store.getString(PropertiesKeys.VENDOR);
        String store_chip = store.getString(PropertiesKeys.CHIP);
        String store_board = store.getString(PropertiesKeys.BOARD);

        int index = architecture.indexOf(store_architecture);
        if (index != -1) {
            architecture.select(index);
            fillVendor(store_architecture);
            vendor.setEnabled(true);
            index = vendor.indexOf(store_vendor);
            if (index != -1) {
                vendor.select(index);
                fillChip(store_architecture, store_vendor);
                chip.setEnabled(true);
                index = chip.indexOf(store_chip);
                if (index != -1) {
                    chip.select(index);
                    fillBoard(store_architecture, store_vendor, store_chip);
                    index = board.indexOf(store_board);
                    if (index != -1)
                        board.select(index);
                }
            } else
                vendor.setText(""); //$NON-NLS-1$
        } else
            architecture.setText(""); //$NON-NLS-1$
    }

    private void fillArchitecture() {
    	getDirList("", "*").stream().filter(s -> !"embsysregview.dtd".equals(s)).forEach(architecture::add);
        restoreStoredSettings();
    }

    private void fillVendor(String selectedArchitecture) {
        chip.setEnabled(false);
        board.setEnabled(false);
        vendor.setEnabled(true);
        chip.setText(""); //$NON-NLS-1$
        board.setText(""); //$NON-NLS-1$
        vendor.removeAll();
        chip.removeAll();
        board.removeAll();
        getDirList("/" + selectedArchitecture, "*").stream().forEach(vendor::add);
        vendor.setText(""); //$NON-NLS-1$
        descriptionText.setText(""); //$NON-NLS-1$
    }

    private void fillChip(String selectedArchitecture, String selectedVendor) {
        board.setEnabled(false);
        board.setText(""); //$NON-NLS-1$
        chip.setEnabled(true);
        chip.removeAll();
        board.removeAll();

        for (String entry : getDirList("/" + selectedArchitecture + "/" + selectedVendor, "*.xml")) { //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
            chip.add(entry.substring(0, entry.length() - 4));
        }

        chip.setText(""); //$NON-NLS-1$
        descriptionText.setText(""); //$NON-NLS-1$
    }

    /** */
    private void fillBoard(String selectedArchitecture, String selectedVendor, String selectedChip) {
        board.removeAll();
        board.add("---  none ---"); //$NON-NLS-1$
        // Check if boards are listed ...
        SAXBuilder builder = new SAXBuilder(XMLReaders.NONVALIDATING);
        //builder.setValidation(false);

        for (String bundleName : registerDescriptionPaths.keySet()) {
            URL fileURL = Platform.getBundle(bundleName).getEntry(registerDescriptionPaths.get(bundleName) + "/" + selectedArchitecture + "/" + selectedVendor + "/" + selectedChip + ".xml"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
            if (fileURL == null)
                continue;

            boolean containsBoards = false;
            try {
                Element root = builder.build(fileURL).getRootElement();
                if (root.getName() != "device") { //$NON-NLS-1$
                     //$NON-NLS-1$
                    for (Element group : root.getChildren("boards")) {
                        for (Element boardElement : group.getChildren()) {
                            containsBoards = true;
                            Attribute attr_bname = boardElement.getAttribute("id"); //$NON-NLS-1$
                            board.add(attr_bname == null ? "-1" : attr_bname.getValue());
                        }
                    }

                    Element chip_description = root.getChild("chip_description"); //$NON-NLS-1$
                    if (chip_description != null)
                        descriptionText.setText(chip_description.getText());
                } else {
                    Element chip_description = root.getChild("description"); //$NON-NLS-1$
                    if (chip_description != null)
                        descriptionText.setText(chip_description.getText());
                }
            } catch (JDOMException | IOException e) {
            	Activator.log(0, e.getMessage(), e);
            }

            // enable only if there are boards to show
            board.setEnabled(containsBoards);
            board.setText(board.getItemCount() > 0 ? board.getItem(0) : "");
        }

    }

    private static SelectionAdapter selectionAdapter(Consumer<SelectionEvent> consumer) {
    	return new SelectionAdapter() {
            @Override
			public void widgetSelected(SelectionEvent event) {
                consumer.accept(event);
            }
        };
    }
    
    @Override
    protected Control createContents(Composite parent) {
        Composite composite = new Composite(parent, SWT.NONE);
        Composite left = new Composite(composite, SWT.NONE);
        Composite right = new Composite(composite, SWT.NONE);

        MigLayout migLayout = new MigLayout("fill", "[180,grow 0][fill,grow]", "top"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
        composite.setLayout(migLayout);

        left.setLayoutData("width 100:180:180"); //$NON-NLS-1$
        left.setLayout(new GridLayout(1, false));

        right.setLayoutData("grow,hmin 0,wmin 0"); //$NON-NLS-1$
        right.setLayout(new FillLayout());

        Label architectureLabel = new Label(left, SWT.LEFT);
        architectureLabel.setText("Architecture:"); //$NON-NLS-1$
        architecture = new Combo(left, SWT.DROP_DOWN | SWT.READ_ONLY);
        architecture.setVisibleItemCount(10);
        architecture.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

        Label vendorLabel = new Label(left, SWT.LEFT);
        vendorLabel.setText("Vendor:"); //$NON-NLS-1$
        vendor = new Combo(left, SWT.DROP_DOWN | SWT.READ_ONLY);
        vendor.setVisibleItemCount(10);
        vendor.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

        Label chipLabel = new Label(left, SWT.LEFT);
        chipLabel.setText("Chip:"); //$NON-NLS-1$
        chip = new Combo(left, SWT.DROP_DOWN | SWT.READ_ONLY);
        chip.setVisibleItemCount(20);
        chip.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

        Label boardLabel = new Label(left, SWT.LEFT);
        boardLabel.setText("Board:"); //$NON-NLS-1$
        board = new Combo(left, SWT.DROP_DOWN | SWT.READ_ONLY);
        board.setVisibleItemCount(10);
        board.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

        Group descriptionGroup = new Group(right, SWT.NONE);
        descriptionGroup.setText("Chip description"); //$NON-NLS-1$
        descriptionGroup.setLayout(new MigLayout("fill")); //$NON-NLS-1$

        descriptionText = new Text(descriptionGroup,
                SWT.H_SCROLL | SWT.V_SCROLL | SWT.MULTI | SWT.READ_ONLY | SWT.WRAP);
        descriptionText.setLayoutData("height 100%,width 100%,hmin 0,wmin 400"); //$NON-NLS-1$
        descriptionText.setText(""); //$NON-NLS-1$
        FontData[] fD = descriptionText.getFont().getFontData();
        fD[0].setName("Lucida Console"); //$NON-NLS-1$
        Font f = new Font(Display.getCurrent(), fD[0]);
        descriptionText.setFont(f);

        architecture.addSelectionListener(selectionAdapter(e -> fillVendor(architecture.getItem(architecture.getSelectionIndex()))));
        vendor.addSelectionListener(selectionAdapter(e -> fillChip(architecture.getItem(architecture.getSelectionIndex()), vendor.getItem(vendor.getSelectionIndex()))));
        chip.addSelectionListener(selectionAdapter(e -> fillBoard(architecture.getItem(architecture.getSelectionIndex()), vendor.getItem(vendor.getSelectionIndex()), chip.getItem(chip.getSelectionIndex()))));
        
        architecture.setSize(500, 30);

        vendor.setSize(500, 30);
        vendor.setEnabled(false);

        chip.setSize(500, 30);
        chip.setEnabled(false);

        board.setSize(500, 30);
        board.setEnabled(false);

        fillArchitecture();
        parent.pack();
        return composite;
    }

}
