/*******************************************************************************
 * Copyright (c) 2016 EmbSysRegView
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     ravenclaw78 - initial API and implementation
 *******************************************************************************/
package org.eclipse.cdt.embsysregview.internal.utils;

import java.math.BigInteger;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.debug.core.model.MemoryByte;

/**
 * 
 * */
public class Utils {	

	private Utils() {		
	}
	
    public static <T, U extends Comparable<U>> int compare(T a, T b, Function<T, U> f) {
    	return f.apply(a).compareTo(f.apply(b));
    }
	
    /**
     * Converts from big endian to little endian and the other way round
     * 
     * @param value
     *            32 bit big endian value
     * @return 32 bit little endian value
     * 
     *         In: 0xAABBCCDD Out: 0xDDCCBBAA
     * 
     */

    public static long convertbig2little(long value) {
        long a, b, c, d;

        a = value >> 24;
        b = (value & (~0xFF00FFFFL)) >> 8;
        c = (value & (~0xFFFF00FFL)) << 8;
        d = (value & (~0xFFFFFF00L)) << 24;

        return (a | b | c | d);
    }

    private static Matcher getLineMatcher(String fulltext) {
        return Pattern.compile("^(.*)$", Pattern.MULTILINE).matcher(fulltext);
    }

    public static String getTextLine(int number, String fulltext) {
        Matcher matcher = getLineMatcher(fulltext);

        for (int i = 1; i <= number; i++) {
            if (matcher.find() && i == number)
                return matcher.group(1);
        }
        return ""; //$NON-NLS-1$
    }

    public static String getFirstNotEmptyTextLine(String fulltext) {
        Matcher matcher = getLineMatcher(fulltext);

        while (matcher.find()) {
            if (!matcher.group(1).equals("")) //$NON-NLS-1$
                return matcher.group(1);
        }
        return ""; //$NON-NLS-1$
    }

    public static int countTextLines(String fulltext) {
        int i = 0;

        Matcher matcher = getLineMatcher(fulltext);

        while (matcher.find())
            i++;

        return i;
    }

    /**
     * 
     * */
    public static int getBitFromValue(int bitnr, long value) {
        return (int) ((value >> bitnr) & 1);
    }

    /**
     * 
     * */
    public static String longtobinarystring(long wert, int bits) {
        StringBuilder text = new StringBuilder();
        String x = Long.toBinaryString(wert);

        int missingLen = bits - x.length();

        if (missingLen < 0) {
            String value = x.substring(-missingLen);
            text.append(value.toUpperCase());
        } else {
            for (int i = 0; i < missingLen; i++)
                text.append('0');
            text.append(x.toUpperCase());
        }
        return text.toString();
    }

    public static String longtoHexString(long wert, int bits) {

        int hexdigits = bits / 4;
        if (bits % 4 != 0)
            hexdigits++;

        StringBuilder hex = new StringBuilder();
        hex.append("0x"); //$NON-NLS-1$
        String x = Long.toHexString(wert);
        int missingLen = hexdigits - x.length();

        if (missingLen < 0) {
            String value = x.substring(-missingLen);
            hex.append(value.toUpperCase());
        } else {
            for (int i = 0; i < missingLen; i++)
                hex.append('0');
            hex.append(x.toUpperCase());
        }

        return hex.toString();
    }
    
    public static boolean convertToUnsigned(final MemoryByte[] bytes, final int regOffset, final int regByteSize, final long[] ret) {
      if (bytes[regOffset].isEndianessKnown() && bytes[regOffset].isReadable()) {
  	     byte[] arraybytes = new byte[regByteSize+1];
         if (!(bytes[regOffset].isBigEndian())) {
            for (int k = 0; k < regByteSize; k++) {
                arraybytes[k+1] = bytes[regOffset + regByteSize - 1 - k].getValue();
            }
         } else {
            for (int k = 0; k < regByteSize; k++) {
                arraybytes[k+1] = bytes[regOffset + k].getValue();
            }
         }
        //no sign anymore! 
        arraybytes[0] = 0;
        BigInteger big = new BigInteger(arraybytes);
        ret[0] = Long.parseLong(big.toString());
        return true;
      }
      return false;
    }

	public static IProject getProjectByName(String name) {
		return ResourcesPlugin.getWorkspace().getRoot().getProject(name);
	}
    
}
