package org.eclipse.cdt.embsysregview.internal.utils;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import org.eclipse.cdt.embsysregview.Activator;
import org.eclipse.cdt.embsysregview.Messages;
import org.eclipse.cdt.embsysregview.internal.model.TreeElement;
import org.eclipse.cdt.embsysregview.internal.model.TreeGroup;
import org.eclipse.cdt.embsysregview.internal.model.TreeParent;
import org.eclipse.cdt.embsysregview.internal.model.TreeRegister;
import org.eclipse.cdt.embsysregview.internal.model.TreeRegisterGroup;
import org.eclipse.cdt.embsysregview.internal.ui.view.ImportRegDialog;
import org.eclipse.cdt.embsysregview.views.EmbSysRegView;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IViewPart;
import org.jdom2.Attribute;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.input.SAXBuilder;

public class ImportRegisters {

	public static void importRegisterValues(IViewPart view) {
		if (!(view instanceof EmbSysRegView)) {
			return;
		}
		EmbSysRegView embView = (EmbSysRegView) view;
        if (embView.isSessionTerminated()) {
            showMessage(view.getViewSite().getShell(), SWT.ICON_WARNING | SWT.OK, Messages.import_title_no_debug,
                    Messages.import_message_no_debug);
            return;
        }

		String filename = null;
		ImportRegDialog dialog = new ImportRegDialog(null);
		dialog.create();
		if (dialog.open() == Window.OK) {
			filename = dialog.getFilename();
		} else {
			return;
		}
		if (filename == null || filename.isEmpty()) {
			return;
		}
		TreeParent modelRoot = embView.getViewer().getEngine().getModel().getRoot();
		HashMap<String, TreeElement> regMap = new HashMap<>();
		TreeElement[] group = modelRoot.getChildren();
		for (TreeElement treeGroupElement : group) {
			if (treeGroupElement instanceof TreeGroup) {
				TreeElement[] treeGroup = ((TreeGroup) treeGroupElement).getChildren();
				for (TreeElement treeRegGroupElement : treeGroup) {
					if (treeRegGroupElement instanceof TreeRegisterGroup) {
						Stream.of(((TreeRegisterGroup) treeRegGroupElement).getChildren())
								.forEach(r -> regMap.put(r.getName() + ((TreeRegister) r).getRegisterAddrString(), r));
					}
				}
			}
		}

		List<String> failSet = writeToModel(filename, regMap, embView);
		if (failSet == null) {
			showMessage(embView.getViewSite().getShell(), SWT.ICON_WARNING | SWT.OK, Messages.import_title_no_reg,
					Messages.import_message_no_reg);
		} else if (!failSet.isEmpty()) {
			String message = String.join("\n", failSet);
			StatusDialog statusDialog = new StatusDialog(view.getViewSite().getShell());
			statusDialog.create();
			statusDialog.fillText(message);
			statusDialog.open();
		} else {
			showMessage(view.getViewSite().getShell(), SWT.ICON_INFORMATION | SWT.OK, Messages.import_title_success,
					Messages.import_message_success);
		}
	}

	public static List<Element> parseXML(String filename) {
		Document document = null;
		List<Element> importList = new ArrayList<>();
		try {
			File inputFile = new File(filename);
			SAXBuilder saxBuilder = new SAXBuilder();
			document = saxBuilder.build(inputFile);
			Element rootElement = document.getRootElement();
			importList = collectExportRegisters(rootElement);
		} catch (JDOMException | IOException e) {
			Activator.log(IStatus.ERROR, Messages.import_parse_error, e);
		}
		return importList;
	}

    public static List<String> writeToModel(String filename, Map<String, TreeElement> regMap,
			EmbSysRegView embView) {
		List<Element> importList = parseXML(filename);
		if (importList == null | importList.isEmpty()) {
			return null;
		}
		List<String> failSet = new ArrayList<String>();
		for (Element element : importList) {
			String a = element.getName();
			if (a.equals(Messages.export_register)) {
				Attribute attribute = element.getAttribute(Messages.export_name);
				String regName = attribute.getValue();
				String address = element.getChild(Messages.export_address).getText();
				String value = element.getChild(Messages.export_value).getText();
				if (value.contains("?")) {
					failSet.add(regName);
					continue;
				}
				TreeElement regElem = regMap.get(regName + address);
				if (!(regElem instanceof TreeRegister)) {
					failSet.add(regName);
					continue;
				}
				TreeRegister register = (TreeRegister) regElem;
				if (register == null || register.getPrintValue().equalsIgnoreCase(value)) {
					failSet.add(regName);
					continue;
				}
				long lvalue = -1;
				if (value.startsWith("0x")) { //$NON-NLS-1$
					lvalue = Long.valueOf(value.substring(2, value.length()), 16);
					if (!embView.getViewer().getEngine().setValue(register, lvalue)) {
						failSet.add(regName);
					}
				}
			}
		}
		return failSet;
	}

	private static List<Element> collectExportRegisters(Element element) {
		if (element == null) {
			return new ArrayList<>();
		}
		if (element.getName().equals(Messages.export_register)) {
			return Arrays.asList(element);
		}
		List<Element> fullList = new ArrayList<>();
		List<Element> children = element.getChildren();
		if (!children.isEmpty()) {
			for (Element child : children) {
				fullList.addAll(collectExportRegisters(child));
			}
		}
		return fullList;
	}

	private static void showMessage(Shell shell, int style, String title, String message) {
		MessageBox messageDialog = new MessageBox(shell, style);
		messageDialog.setText(title);
		messageDialog.setMessage(message);
		messageDialog.open();
		return;
	}

	private static class StatusDialog extends TitleAreaDialog {
		private Text skippedReg;

		public StatusDialog(Shell parentShell) {
			super(parentShell);
		}

		@Override
		public void create() {
			super.create();
			setTitle(Messages.import_title_success);
		}

        @Override
        protected void configureShell(Shell newShell) {
            super.configureShell(newShell);
            newShell.setText(Messages.import_title);
        }

		@Override
		protected Control createDialogArea(Composite parent) {
			Composite area = (Composite) super.createDialogArea(parent);
			Composite container = new Composite(area, SWT.NONE);
			container.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
			container.setLayout(new GridLayout(1, false));
			createText(container);
			return area;
		}

		private void createText(Composite container) {
			Label lbl = new Label(container, SWT.NONE);
			lbl.setText(Messages.import_label_skip);
			skippedReg = new Text(container, SWT.BORDER | SWT.READ_ONLY | SWT.V_SCROLL);
			GridData dataGrd = new GridData(SWT.FILL, SWT.FILL, true, true);
			skippedReg.setLayoutData(dataGrd);
		}

		public void fillText(String reg) {
			skippedReg.setText(reg);
		}

		@Override
		protected boolean isResizable() {
			return false;
		}
	}

}
