package org.eclipse.cdt.embsysregview.internal.utils;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Stream;

import org.eclipse.cdt.embsysregview.Activator;
import org.eclipse.cdt.embsysregview.Messages;
import org.eclipse.cdt.embsysregview.internal.model.TreeElement;
import org.eclipse.cdt.embsysregview.internal.model.TreeField;
import org.eclipse.cdt.embsysregview.internal.model.TreeGroup;
import org.eclipse.cdt.embsysregview.internal.model.TreeParent;
import org.eclipse.cdt.embsysregview.internal.model.TreeRegister;
import org.eclipse.cdt.embsysregview.internal.model.TreeRegisterGroup;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.output.Format;
import org.jdom2.output.XMLOutputter;

public class ExportRegisters {

	public static IStatus exportToFile(IStructuredSelection select, String exportFileName, boolean exportAllRegisters) {
		List<TreeElement> registers = selectRegistersForExport(select, exportAllRegisters);
		if (!registers.isEmpty()) {
            return export(registers, exportFileName);
		} else {
			return Status.CANCEL_STATUS;
		}
	}

	public static List<TreeElement> selectRegistersForExport(IStructuredSelection selection,
			boolean exportAllRegisters) {
		List<TreeElement> registerList = new ArrayList<>();
		if (exportAllRegisters) {
			Object obj = selection.getFirstElement();
			for (;;) {
				// find root element from any selected element
				if (obj instanceof TreeGroup) {
					TreeElement elem = (TreeElement) obj;
					elem = elem.getParent();
					registerList.addAll(addRegisterGroup((TreeParent) elem));
					break;
				} else {
					obj = ((TreeElement) obj).getParent();
				}
			}
		} else
			for (Object obj : selection.toArray()) {
				if (obj instanceof TreeGroup) {
					registerList.addAll(addRegisterGroup((TreeParent) obj));
				} else if (obj instanceof TreeRegisterGroup) {
					registerList.addAll(addRegisterGroup((TreeParent) obj));
				} else if (obj instanceof TreeRegister) {
					registerList.add((TreeRegister) obj);
				} else if (obj instanceof TreeField) {
					registerList.add(((TreeField) obj).getParent());
				} else {
					// this selection is not supported
					return registerList;
				}
			}
		return registerList;
	}


	public static IStatus export(List<TreeElement> registerList, String filename) {
		IStatus status = Status.OK_STATUS;
		Element root = new Element(Messages.export_root);
		for (TreeElement element : registerList) {
			if (element instanceof TreeGroup) {
				TreeGroup treeGroup = (TreeGroup) element;
				Element treeGroupElement = new Element(Messages.export_group);
				treeGroupElement.setAttribute(Messages.export_name, treeGroup.getName());
				Stream.of(treeGroup.getChildren()).forEach(r -> treeGroupElement.addContent(treeRegGroupToElement(r)));
				root.addContent(treeGroupElement);
			} else if (element instanceof TreeRegisterGroup) {
				Element treeRegGroupElement = treeRegGroupToElement(element);
				root.addContent(treeRegGroupElement);
			} else if (element instanceof TreeRegister) {
				root.addContent(registerToElement(element));
			}
		}

		Document doc = new Document();
		doc.setRootElement(root);
		XMLOutputter outter = new XMLOutputter();
        try (FileWriter fw = new FileWriter(new File(filename))) {
			outter.setFormat(Format.getPrettyFormat());
            outter.output(doc, fw);
		} catch (IOException e) {
			status = new Status(Status.ERROR, Activator.PLUGIN_ID, e.getMessage());
			Activator.log(status);
		}
		return status;
	}

	private static List<TreeElement> addRegisterGroup(final TreeParent group) {
		return Arrays.asList(group.getChildren());
	}

	private static Element treeRegGroupToElement(final TreeElement element) {
		if (element instanceof TreeRegisterGroup) {
			TreeRegisterGroup treeRegGroup = (TreeRegisterGroup) element;
			Element treeRegGroupElement = new Element(Messages.export_register_group);
			treeRegGroupElement.setAttribute(Messages.export_name, treeRegGroup.getName());
			Stream.of(treeRegGroup.getChildren()).forEach(r -> treeRegGroupElement.addContent(registerToElement(r)));
			return treeRegGroupElement;
		}
		return new Element(element.getName());
	}

	private static Element registerToElement(final TreeElement element) {
		if (element instanceof TreeRegister) {
			TreeRegister register = (TreeRegister) element;
			return getRegisterAsElement(register);
		}
		return new Element(element.getName());
	}

	public static Element getRegisterAsElement(TreeRegister register) {
		Element registerElement = new Element(Messages.export_register);
		registerElement.setAttribute(Messages.export_name, register.getName());

		Element value = new Element(Messages.export_value);
		value.addContent(register.getPrintValue());

		Element address = new Element(Messages.export_address);
		if (!register.isSPR()) {
			address.addContent(register.getRegisterAddrString());
		}

		Element description = new Element(Messages.export_description);
		description.addContent(register.getDescription());

		registerElement.addContent(value);
		registerElement.addContent(address);
		registerElement.addContent(description);
		return registerElement;
	}

}
