package org.eclipse.cdt.embsysregview.internal.utils;

import java.util.ArrayList;

import java.util.List;
import java.util.concurrent.RejectedExecutionException;
import org.eclipse.cdt.dsf.concurrent.ConfinedToDsfExecutor;
import org.eclipse.cdt.dsf.concurrent.DsfRunnable;
import org.eclipse.cdt.dsf.service.DsfServicesTracker;
import org.eclipse.cdt.dsf.service.DsfSession;
import org.eclipse.cdt.embsysregview.Activator;

/**
 * DSF session state object.
 * 
 * Encapsulates and manages DsfSession we're currently tracking.
 */
public class DSFSessionState {
    // --- members ---
    /** Current session ID. */
    protected final String m_sessionId;

    /** Current set of session event listeners. */
    protected final List<Object> m_sessionListeners;

    /** Services tracker, used to access services. */
    protected final DsfServicesTracker m_servicesTracker;    

    // --- constructors/destructors ---

    public DSFSessionState(final DsfSession session)  {
        //do not put debug context here    	
        m_sessionId = session.getId();
        m_sessionListeners = new ArrayList<>();
        m_servicesTracker = new DsfServicesTracker(Activator.getDefault().getBundle().getBundleContext(), m_sessionId);        
    }

    /** Dispose method. */
    public void dispose() {
        if (m_sessionId != null) {
            removeAllServiceEventListeners();
            // m_sessionId = null;
            // m_sessionListeners = null;
        }

        if (m_servicesTracker != null) {
            m_servicesTracker.dispose();
            // m_servicesTracker = null;
        }
    }

    // --- accessors ---

    /** Returns session ID. */
    public String getSessionID() {
        return m_sessionId;
    }
    
    // --- listener management ---

    /** Adds a service event listener. */
    public void addServiceEventListener(Object listener) {
        final Object listener_f = listener;
        final DsfSession session_f = getDsfSession();
        if (session_f != null) {
            try {
                session_f.getExecutor().execute(new DsfRunnable() {
                    @Override
                    public void run() {
                        session_f.addServiceEventListener(listener_f, null);
                        m_sessionListeners.add(listener_f);
                    }
                });
            } catch (RejectedExecutionException e) {
                // Session is shut down.
            	Activator.log(0, e.getMessage(), e);
            }
        }
    }

    /** Removes a service event listener. */
    public void removeServiceEventListener(Object listener) {
        final Object listener_f = listener;
        final DsfSession session_f = getDsfSession();
        if (session_f != null) {
            try {
                session_f.getExecutor().execute(new DsfRunnable() {
                    @Override
                    public void run() {
                        if (m_sessionListeners != null) {
                            session_f.removeServiceEventListener(listener_f);
                            m_sessionListeners.remove(listener_f);
                        }
                    }
                });
            } catch (RejectedExecutionException e) {
                // Session is shut down.
            	Activator.log(0, e.getMessage(), e);
            }
        }
    }

    /** Removes all service event listeners. */
    public void removeAllServiceEventListeners() {
        final DsfSession session_f = getDsfSession();
        if (session_f != null) {
            try {
                session_f.getExecutor().execute(new DsfRunnable() {
                    @Override
                    public void run() {
                        if (m_sessionListeners != null) {
                            for (Object listener : m_sessionListeners) {
                                session_f.removeServiceEventListener(listener);
                            }
                            m_sessionListeners.clear();
                        }
                    }
                });
            } catch (RejectedExecutionException e) {
                // Session is shut down.
            	Activator.log(0, e.getMessage(), e);
            }
        }
    }

    // --- methods ---

    /** Gets current DsfSession, if it's still active. */
    protected DsfSession getDsfSession() {
        return DsfSession.getSession(m_sessionId);
    }

    /** Executes DsfRunnable. */
    public void execute(DsfRunnable runnable) {
        try {
            DsfSession session = getDsfSession();
            if (session == null) {                
            	Activator.log(0, "session is already null");
            } else {
                session.getExecutor().execute(runnable);
            }
        } catch (RejectedExecutionException e) {           
        	Activator.log(0, e.getMessage(), e);            
        }
    }

    /** Gets service of the specified type. */
    @ConfinedToDsfExecutor("getDsfSession().getExecutor()")
    public <V> V getService(Class<V> serviceClass) {
        return (m_servicesTracker == null) ? null : m_servicesTracker.getService(serviceClass);
    }	
}
