package org.eclipse.cdt.embsysregview.internal.utils;

import java.util.concurrent.TimeoutException;

import org.eclipse.cdt.core.IAddress;
import org.eclipse.cdt.dsf.concurrent.ConfinedToDsfExecutor;
import org.eclipse.cdt.dsf.concurrent.DataRequestMonitor;
import org.eclipse.cdt.dsf.concurrent.RequestMonitor;
import org.eclipse.cdt.dsf.datamodel.DMContexts;
import org.eclipse.cdt.dsf.datamodel.IDMContext;
import org.eclipse.cdt.dsf.debug.service.IFormattedValues;
import org.eclipse.cdt.dsf.debug.service.IFormattedValues.FormattedValueDMContext;
import org.eclipse.cdt.dsf.debug.service.IFormattedValues.FormattedValueDMData;
import org.eclipse.cdt.dsf.debug.service.IFormattedValues.IFormattedDataDMContext;
import org.eclipse.cdt.dsf.debug.service.IMemory;
import org.eclipse.cdt.dsf.debug.service.IMemory.IMemoryDMContext;
import org.eclipse.cdt.dsf.debug.service.IProcesses;
import org.eclipse.cdt.dsf.debug.service.IProcesses.IThreadDMContext;
import org.eclipse.cdt.dsf.debug.service.IRegisters;
import org.eclipse.cdt.dsf.debug.service.IRegisters.IBitFieldDMData;
import org.eclipse.cdt.dsf.debug.service.IRegisters.IRegisterDMContext;
import org.eclipse.cdt.dsf.debug.service.IRegisters.IRegisterGroupDMContext;
import org.eclipse.cdt.dsf.debug.service.IStack;
import org.eclipse.cdt.dsf.debug.service.IStack.IFrameDMContext;
import org.eclipse.cdt.dsf.gdb.service.IGDBMemory;
import org.eclipse.cdt.dsf.gdb.service.IGDBProcesses;
import org.eclipse.cdt.dsf.mi.service.IMIContainerDMContext;
import org.eclipse.cdt.dsf.mi.service.IMIExecutionDMContext;
import org.eclipse.cdt.dsf.mi.service.MIRegisters;
import org.eclipse.cdt.dsf.mi.service.MIRegisters.MIRegisterDMC;
import org.eclipse.cdt.dsf.mi.service.MIRegisters.MIRegisterGroupDMC;
import org.eclipse.cdt.dsf.service.DsfSession;
import org.eclipse.cdt.embsysregview.Activator;
import org.eclipse.cdt.embsysregview.internal.model.TreeRange;
import org.eclipse.cdt.embsysregview.internal.model.TreeRegister;
import org.eclipse.cdt.utils.Addr64;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.model.MemoryByte;
import org.eclipse.debug.ui.DebugUITools;

/**
 * 
 * */
public class DSFAccessor {
	/**
	 * Requests data of a EmbSysRegister. Calls back to
	 * getEmbSysRegisterDataDone() on listener.
	 * 
	 * @throws TimeoutException
	 */
	@ConfinedToDsfExecutor("getSession().getExecutor()")
	public static void getEmbSysRegisterData(IDMContext context, DSFSessionState sessionState,
			final TreeRegister register, final DSFDebugModelListener listener, final Object arg) {
		assert register != null;
		if (register.isSPR()) {
			readSprRegister(context, sessionState, register, listener, arg);
		} else {
			readMemoryRegister(context, sessionState, register, listener, arg);
		}
	}

	@ConfinedToDsfExecutor("getSession().getExecutor()")
	private static void readSprRegister(IDMContext dmc, DSFSessionState sessionState, final TreeRegister register,
			final DSFDebugModelListener listener, final Object arg) {
		final IMIContainerDMContext containerDmc = DMContexts.getAncestorOfType(dmc, IMIContainerDMContext.class);
		IMIExecutionDMContext[] exec = sessionState.getService(IGDBProcesses.class).getExecutionContexts(containerDmc);
		if (exec.length > 0) {
			DsfSession session = sessionState.getDsfSession();
			sessionState.getService(IStack.class).getTopFrame(exec[0],
					new DataRequestMonitor<IFrameDMContext>(session.getExecutor(), null) {
						@Override
						protected void handleSuccess() {
							MIRegisters registers = (MIRegisters) sessionState.getService(IRegisters.class);
							IFormattedDataDMContext registerDmc = new MIRegisterDMC(registers, null, getData(), (int) register.getRegisterAddress(), register.getName());
							FormattedValueDMContext formattedDmc = registers.getFormattedValueContext(registerDmc, IFormattedValues.NATURAL_FORMAT);
							registers.getFormattedExpressionValue(formattedDmc,
									new DataRequestMonitor<FormattedValueDMData>(registers.getExecutor(), null) {
										@Override
										protected void handleCompleted() {
											if (getStatus().isOK()) {
												register.setValue(Long.parseLong(getData().getFormattedValue()));
											}
											listener.getEmbSysRegisterDataDone(getStatus(), register, arg);
										}
									});
						}

						@Override
						protected void handleFailure() {
							listener.getEmbSysRegisterDataDone(getStatus(), register, arg);
						}
					});
		}
	}

	@ConfinedToDsfExecutor("getSession().getExecutor()")
	private static void readMemoryRegister(IDMContext cont, DSFSessionState sessionState, final TreeRegister register,
			final DSFDebugModelListener listener, final Object arg) {
		final IMemoryDMContext context = DMContexts.getAncestorOfType(cont, IMemoryDMContext.class);
		final IMemory fMemReg = sessionState.getService(IMemory.class);
		final IStatus status = new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Service down"); //$NON-NLS-1$
		if (null == fMemReg) {
			listener.getEmbSysRegisterDataDone(status, register, arg);
			return;
		}
		final IAddress address1 = new Addr64(register.getRegisterAddrString());
		fMemReg.getMemory(context, address1, 0, 1, register.getByteSize(),
				new DataRequestMonitor<MemoryByte[]>(fMemReg.getExecutor(), null) {
					@Override
					protected void handleSuccess() {
						MemoryByte[] bytes = getData();
						long[] ret = new long[] { -1 };
						if (Utils.convertToUnsigned(bytes, 0, register.getByteSize(), ret)) {
							register.setValue(ret[0]);
							listener.getEmbSysRegisterDataDone(getStatus(), register, arg);
						} else {
							final IStatus statusRm = new Status(IStatus.ERROR, Activator.PLUGIN_ID,
									"Unable to convert"); //$NON-NLS-1$
							listener.getEmbSysRegisterDataDone(statusRm, register, arg);
						}
					}

					@Override
					protected void handleFailure() {
						final IStatus statusRm = new Status(0, Activator.PLUGIN_ID, "Read register fail"); //$NON-NLS-1$
						listener.getEmbSysRegisterDataDone(statusRm, register, arg);
					}
				});

	}

	@ConfinedToDsfExecutor("getSession().getExecutor()")
	public static void writeEmbSysRegisterData(IDMContext context, DSFSessionState sessionState,
			final TreeRegister register, final DSFDebugModelListener listener, final Object arg) {
		if (register.isSPR()) {
			writeSprRegisterData(sessionState, register, listener, arg);
		} else {
			writeMemoryRegisterData(context, sessionState, register, listener, arg);
		}
	}

	@ConfinedToDsfExecutor("getSession().getExecutor()")
	private static void writeSprRegisterData(DSFSessionState sessionState, final TreeRegister register,
			final DSFDebugModelListener listener, final Object arg) {
		IDMContext dmc = (IDMContext) DebugUITools.getDebugContext().getAdapter(IDMContext.class);
		MIRegisters regService = (MIRegisters) sessionState.getService(IRegisters.class);
		DsfSession session = sessionState.getDsfSession();
		regService.getRegisterGroups(dmc,
				new DataRequestMonitor<IRegisterGroupDMContext[]>(session.getExecutor(), null) {
					@Override
					protected void handleSuccess() {
						// daisy chain contexts
						MIRegisterGroupDMC groupCtx = (MIRegisterGroupDMC) getData()[0];
						// create frameless context for SPR
						IRegisterDMContext registerDmc = new MIRegisterDMC(regService, groupCtx,
								(int) register.getRegisterAddress(), register.getName());
						regService.writeRegister(registerDmc, Integer.toString((int) register.getValue()),
								IFormattedValues.NATURAL_FORMAT,
								new DataRequestMonitor<IBitFieldDMData>(regService.getExecutor(), null) {
									@Override
									protected void handleSuccess() {
										listener.writeEmbSysRegisterDataDone(getStatus(), register, arg);
									}
								});
					}

					@Override
					protected void handleFailure() {
						final IStatus status = new Status(IStatus.ERROR, Activator.PLUGIN_ID,
								"Write SPR register fail"); //$NON-NLS-1$
						listener.writeEmbSysRegisterDataDone(status, register, arg);
					}
				});

	}

	@ConfinedToDsfExecutor("getSession().getExecutor()")
	private static void writeMemoryRegisterData(final IDMContext cont, DSFSessionState sessionState,
			final TreeRegister register, final DSFDebugModelListener listener, final Object arg) {
		final IMemoryDMContext context = DMContexts.getAncestorOfType(cont, IMemoryDMContext.class);
		final IGDBMemory fMemReg = sessionState.getService(IGDBMemory.class);
		final IAddress address = new Addr64(register.getRegisterAddrString());
		final boolean isBigEndian = fMemReg.isBigEndian(context);
		// N.B. this call also reads back in order to update internal memory
		// service cache.
		// this may give unwanted effect for W/O registers.
		fMemReg.setMemory(context, address, 0, 1, register.getByteSize(), register.getByteBuffer(isBigEndian),
				new RequestMonitor(fMemReg.getExecutor(), null) {
					@Override
					protected void handleSuccess() {
						fMemReg.getMemory(context, address, 0, 1, register.getByteSize(),
								new DataRequestMonitor<MemoryByte[]>(fMemReg.getExecutor(), null) {
									@Override
									protected void handleCompleted() {
										MemoryByte[] bytes = getData();
										long[] ret = new long[] { -1 };
										if (Utils.convertToUnsigned(bytes, 0, register.getByteSize(), ret)) {
											register.setValue(ret[0]);
											listener.writeEmbSysRegisterDataDone(getStatus(), register, arg);
										} else {
											IStatus status;
											if (register.isWriteOnly()) {
												status = new Status(IStatus.INFO, Activator.PLUGIN_ID,
														"Cannot read back"); //$NON-NLS-1$
											} else {
												status = new Status(IStatus.ERROR, Activator.PLUGIN_ID,
														"Unable to convert"); //$NON-NLS-1$
											}
											listener.writeEmbSysRegisterDataDone(status, register, arg);
										}
									}
								});
					}

					@Override
					protected void handleFailure() {
						final IStatus status = new Status(IStatus.ERROR, Activator.PLUGIN_ID,
								"Write memory-mapped register fail"); //$NON-NLS-1$
						listener.writeEmbSysRegisterDataDone(status, register, arg);
					}
				});
	}

	/**
	 * 
	 * */
	public static void getEmbSysDeviceData(final IDMContext cont, DSFSessionState sessionState, final TreeRange range,
			final DSFDebugModelListener listener, final Object arg) {
		final IMemoryDMContext context = DMContexts.getAncestorOfType(cont, IMemoryDMContext.class);
		final IMemory fMemReg = sessionState.getService(IMemory.class);
		final IAddress address1 = new Addr64(range.getRangeLowAddrString());

		fMemReg.getMemory(context, address1, 0, 1, range.getRangeByteCount(),
				new DataRequestMonitor<MemoryByte[]>(fMemReg.getExecutor(), null) {
					@Override
					protected void handleSuccess() {
						MemoryByte[] bytes = getData();
						listener.readEmbSysRangeDataDone(getStatus(), range, bytes, arg);
					}

					@Override
					protected void handleFailure() {
						final IStatus status = new Status(0, Activator.PLUGIN_ID, "Read memory fail"); //$NON-NLS-1$
						listener.readEmbSysRangeDataDone(status, range, null, arg);
					}
				});
	}
}
