package org.eclipse.cdt.embsysregview.internal.ui.view;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.eclipse.cdt.core.CProjectNature;
//import org.eclipse.cdt.core.index.export.Messages;
import org.eclipse.cdt.dsf.service.DsfSession;
import org.eclipse.cdt.embsysregview.Activator;
import org.eclipse.cdt.embsysregview.Messages;
import org.eclipse.cdt.embsysregview.core.EmbSysRegCore;
import org.eclipse.cdt.embsysregview.core.EmbSysRegEngine;
import org.eclipse.cdt.embsysregview.internal.model.TreeElement;
import org.eclipse.cdt.embsysregview.internal.model.TreeField;
import org.eclipse.cdt.embsysregview.internal.model.TreeParent;
import org.eclipse.cdt.embsysregview.internal.model.TreeRegister;
import org.eclipse.cdt.embsysregview.internal.model.TreeRegisterGroup;
import org.eclipse.cdt.embsysregview.internal.utils.GUIUtils;
import org.eclipse.cdt.embsysregview.properties.PropertiesHolder;
import org.eclipse.cdt.embsysregview.views.EmbSysRegView;
import org.eclipse.cdt.embsysregview.views.EmbSysRegView.RegSet;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ColumnViewerToolTipSupport;
import org.eclipse.jface.viewers.ComboBoxViewerCellEditor;
import org.eclipse.jface.viewers.EditingSupport;
import org.eclipse.jface.viewers.ICellEditorListener;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.swt.widgets.Widget;

import net.miginfocom.swt.MigLayout;

/**
 * 
 * */
public class EmbSysRegViewer {
    private static final int TABLE_UPDATE_INTERVAL = 200;
    private TreeViewer m_viewer;
    private Composite header;
    private Label infoLabel;

    /** Parent control. */
    private Composite m_parent = null;

    final private EmbSysRegEngine m_engine;

    private final RegSet kind;

    private static final QualifiedName EMBSYS_REGVIEW_STATE = new QualifiedName(null, "EmbSysTreeViewState"); //$NON-NLS-1$
    private static final QualifiedName SPR_REGVIEW_STATE = new QualifiedName(null, "SprTreeViewState"); //$NON-NLS-1$
    private static final QualifiedName ALL_REGVIEW_STATE = new QualifiedName(null, "AllTreeViewState"); //$NON-NLS-1$

    private static final String EMPTY = ""; //$NON-NLS-1$
    private static final String ARCH_TXT = "Arch: ";     //$NON-NLS-1$ 
    private static final String VEND_TXT = "  Vendor: "; //$NON-NLS-1$
    private static final String CHIP_TXT = "  Chip: ";   //$NON-NLS-1$
    private static final String CORE_TXT = "  Core: ";   //$NON-NLS-1$
    private static final String BOARD_TXT= "  Board: ";  //$NON-NLS-1$
    /**
     * 
     * */
    protected static class AutoDropComboBoxViewerCellEditor extends ComboBoxViewerCellEditor {

        public AutoDropComboBoxViewerCellEditor(Composite parent) {
            super(parent, SWT.NONE);
            setActivationStyle(DROP_DOWN_ON_MOUSE_ACTIVATION);
        }

        @Override
        protected Control createControl(Composite parent) {
            final Control control = super.createControl(parent);
            CCombo cCombo = super.getViewer().getCCombo();
            cCombo.addSelectionListener(new SelectionAdapter() {
                @Override
                public void widgetSelected(SelectionEvent e) {
                    // if the list is not visible, assume the user is done
                    if (!((CCombo) getControl()).getListVisible()) {
              
                        // since I cannot access applyEditorValueAndDeactivate()
                        focusLost();
                    }
                }
            });
            return control;
        }

    }

    /**
     * 
     **/
    protected static class HexCellEditor extends TextCellEditor {
    	public HexCellEditor(Tree tree) {
			super(tree);
		}

		@Override
    	protected void focusLost() {
			//--we leave editor by click on another line (see ENGR00379350)
			if (isActivated()) {
				fireCancelEditor();
			}
    	}
		
    } 

    /**
     * @param embSysRegView
     *            not used
     * 
     */
    public EmbSysRegViewer(EmbSysRegView embSysRegView, Composite parent, RegSet regSet) {
        m_parent = parent;
        kind = regSet;
        m_engine = new EmbSysRegEngine(this, regSet);        
        init();
    }

    /**
     * 
     * */
    public void requestUpdate() {
        GUIUtils.exec(new Runnable() {
            @Override
            public void run() {
            	update();
            }
        });

    }

    /**
     * 
     * */
    private void addExpandCollapseGroupListeners() {
        Tree tree = m_viewer.getTree();
        // prevents expanding node by double-click
        tree.addListener(SWT.MeasureItem, new Listener() {
            @Override
            public void handleEvent(Event event) {
            	//just override to empty
            }
        });

        tree.addListener(SWT.Expand, new Listener() {
            @Override
            public void handleEvent(Event event) {
                Widget item = event.item;
                final Object obj = item.getData();
                if (obj instanceof TreeRegisterGroup) {
                    TreeRegisterGroup treeRegisterGroup = ((TreeRegisterGroup) obj);
                    treeRegisterGroup.setExpanded(true);
                }
            }
        });

        tree.addListener(SWT.Collapse, new Listener() {
            @Override
            public void handleEvent(Event event) {
                Widget item = event.item;
                final Object obj = item.getData();
                if (obj instanceof TreeRegisterGroup) {
                    TreeRegisterGroup treeRegisterGroup = ((TreeRegisterGroup) obj);
                    treeRegisterGroup.setExpanded(false);
                }

            }
        });
    }

    /** Compose the actual layout of viewer. */
    protected void init() {

        TreeViewerColumn column;
        m_parent.setLayout(new MigLayout("fill", EMPTY, EMPTY)); //$NON-NLS-1$
        header = new Composite(m_parent, SWT.NONE);
        header.setLayoutData("dock north,height 16px,width 100%,wmin 0,hmin 16,gap 1 1 -5 1"); //$NON-NLS-1$

        RowLayout rowLayout = new RowLayout(SWT.HORIZONTAL);

        rowLayout.marginLeft = 1;
        rowLayout.marginRight = 0;
        rowLayout.marginTop = -1;
        rowLayout.marginBottom = 0;
        rowLayout.fill = true;
        rowLayout.wrap = false;
        rowLayout.spacing = 5;
        header.setLayout(rowLayout);

        infoLabel = new Label(header, SWT.WRAP);

        m_viewer = m_engine.createTreeViewer(m_parent);
        // create HEX and BIN cell editors
        final TextCellEditor textCellEditor = new HexCellEditor(m_viewer.getTree());
        textCellEditor.setValidator(new HexCellEditorValidator(m_viewer));
        final ComboBoxViewerCellEditor comboBoxCellEditor = new AutoDropComboBoxViewerCellEditor(m_viewer.getTree());
        
        // Registername
        column = new TreeViewerColumn(m_viewer, SWT.NONE);
        column.getColumn().setWidth(250);
        column.getColumn().setMoveable(false);
        column.getColumn().setText("Register"); //$NON-NLS-1$
        ColumnLabelProvider registerColumnLabelProvider = EmbSysRegUi.getRegisterColumnLabelProvider(m_parent);
        column.setLabelProvider(registerColumnLabelProvider);

        // Hex Value
        column = new TreeViewerColumn(m_viewer, SWT.NONE);
        column.getColumn().setWidth(80);
        column.getColumn().setMoveable(false);
        column.getColumn().setText("Hex"); //$NON-NLS-1$
        ColumnLabelProvider hexColumnLabelProvider = EmbSysRegUi.getHexColumnLabelProvider(m_parent);
        column.setLabelProvider(hexColumnLabelProvider);
        HexColumnEditingSupport hexColumnEditingSupport = getHexColumnEditingSupport(comboBoxCellEditor, textCellEditor);
        column.setEditingSupport(hexColumnEditingSupport);

        // Binary Value
        column = new TreeViewerColumn(m_viewer, SWT.NONE);
        column.getColumn().setWidth(200);
        column.getColumn().setMoveable(false);
        column.getColumn().setText("Bin"); //$NON-NLS-1$
        ColumnLabelProvider binColumnLabelProvider = EmbSysRegUi.getBinColumnLabelProvider(m_parent);
        column.setLabelProvider(binColumnLabelProvider);
        EditingSupport binEditingSupport = getBinEditingSupport();
        column.setEditingSupport(binEditingSupport);

        //
        comboBoxCellEditor.setValidator(new HexCellEditorValidator(m_viewer));
        ICellEditorListener cellEditorListener = getCellEditorListener(comboBoxCellEditor, hexColumnEditingSupport.getCurrentEditedElement());
        comboBoxCellEditor.addListener(cellEditorListener);
        
        
        // Register Reset Value (hex)
        column = new TreeViewerColumn(m_viewer, SWT.NONE);
        column.getColumn().setWidth(80);
        column.getColumn().setMoveable(false);
        column.getColumn().setText("Reset"); //$NON-NLS-1$
        ColumnLabelProvider resetColumnLabelProvider = EmbSysRegUi.getResetColumnLabelProvider();
        column.setLabelProvider(resetColumnLabelProvider);

        // Register Access
        column = new TreeViewerColumn(m_viewer, SWT.NONE);
        column.getColumn().setAlignment(SWT.CENTER);
        column.getColumn().setWidth(50);
        column.getColumn().setMoveable(false);
        column.getColumn().setText("Access"); //$NON-NLS-1$
        ColumnLabelProvider accessColumnLabelProvider = EmbSysRegUi.getAccessColumnLabelProvider();
        column.setLabelProvider(accessColumnLabelProvider);

        if (kind.embsys) {
	        // Register Address (hex)
	        column = new TreeViewerColumn(m_viewer, SWT.NONE);
	        column.getColumn().setWidth(80);
	        column.getColumn().setMoveable(false);
	        column.getColumn().setText("Address"); //$NON-NLS-1$
	        column.setLabelProvider(EmbSysRegUi.getAddressColumnLabelProvider());
        }
		
        // Description
        column = new TreeViewerColumn(m_viewer, SWT.NONE);
        column.getColumn().setWidth(300);
        column.getColumn().setMoveable(false);
        column.getColumn().setText("Description"); //$NON-NLS-1$
        ColumnViewerToolTipSupport.enableFor(m_viewer);
        CellLabelProvider descriptionCellLabelProvider = EmbSysRegUi.getDescriptionCellLabelProvider();
        column.setLabelProvider(descriptionCellLabelProvider);

        // Double click action
        m_viewer.getTree().addMouseListener(new MouseListener() {

            @Override
            public void mouseDoubleClick(MouseEvent event) {
                handleDoubleClick(event);
            }

            @Override
            public void mouseDown(MouseEvent arg0) {
                // TODO Auto-generated method stub
            }

            @Override
            public void mouseUp(MouseEvent arg0) {
                saveState(m_engine.getModel().getProject());
            }
        });
        // Update
        update();
        //
        addExpandCollapseGroupListeners();        
    }

    /**
     * 
     * */
    protected void update() {
        if (!m_viewer.getControl().isDisposed()) {
            m_viewer.refresh();
        }
    }

    /**
     * We handle only if double clicked on first column. So we need to find
     * column and thus low-level mouse handler.
     */
    protected void handleDoubleClick(MouseEvent event) {
        Tree tree = (Tree) event.getSource();
        Point point = new Point(event.x, event.y);
        int selectedColumn = -1;
        TreeItem item = null;
        for (int i = 0; i < tree.getColumnCount(); i++) {
            item = tree.getItem(point);
            if (item != null) {
                if (item.getBounds(i).contains(point)) {
                    selectedColumn = i;
                    break;
                }
            }
        }
        if (selectedColumn == 0 && item != null) {
            final Object obj = item.getData();
            if (obj instanceof TreeElement) {
                if (!TreeElement.isModelAccessGranted()) {
                    if (obj instanceof TreeRegisterGroup) {
                        m_engine.toggleSelectedGroupOff((TreeRegisterGroup) obj);
                    } else if (obj instanceof TreeRegister) {
                        m_engine.toggleSelectedRegisterOff((TreeRegister) obj);
                    }
                    m_engine.getEmbSysRegViewer().requestUpdate();
                    return;
                }
            }
            if (obj instanceof TreeRegisterGroup) {
                final TreeRegisterGroup treeGroup = ((TreeRegisterGroup) obj);
                m_engine.toggleSelectGroup(treeGroup);
            }
            // only TreeRegister level makes sense
            if (obj instanceof TreeRegister) {
                final TreeRegister treeRegister = ((TreeRegister) obj);
                m_engine.toggleOnLineState(treeRegister);
            }

            if (obj instanceof TreeField) {
                final TreeField treeField = ((TreeField) obj);
                TreeRegister treeRegister = ((TreeRegister) treeField.getParent());
                if (!treeRegister.isWriteOnly()) {
                    // updater.readRegisterValue(treeField.getParent(),
                    // treeRegister);
                }
            }
        }

    }

    /**
     * Handles current workbench selection.
     */
    @SuppressWarnings("unchecked")
    public IStatus updateView(IProject newProject) {
        IContentProvider contentProvider = m_viewer.getContentProvider();
        if (contentProvider != null) {
            contentProvider.inputChanged(m_viewer, null, newProject);
            m_engine.updateDebugContext();
        }

        if (m_viewer.getControl().isDisposed()) {
            return new Status(IStatus.INFO, Activator.PLUGIN_ID, Messages.EmbSysRegViewer_disposed);
        }
        m_viewer.setInput(m_engine.getModel().getRoot());
        if (newProject != null) {
            // restore expanded nodes may be saved in project
            Set<String> expandedPaths = null;
            try {
                Object objs = newProject.getSessionProperty(getTreeViewStateQualifier());
                if (objs != null) {
                    expandedPaths = (Set<String>) objs;
                } else {
                    expandedPaths = new HashSet<>();
                }

            } catch (CoreException e) {
                return new Status(IStatus.ERROR, Activator.PLUGIN_ID, Messages.EmbSysRegViewer_cannot_get_property, e);
            }
            for (final TreeItem treeItem : m_viewer.getTree().getItems()) {
                expandTreeItems(treeItem, expandedPaths);
            }
        }
        requestUpdate();
        return Status.OK_STATUS;
    }

    /**
    *
    * */
   private void expandTreeItems(final TreeItem treeItem, final Set<String> paths) { 
   	   final TreeElement node = (TreeElement)treeItem.getData(); 
       if (node == null) {
    	   return; 
       }
        
       if (paths.contains(node.getPath())) { 
       	m_viewer.setExpandedState(node, true); 
       	m_viewer.refresh(node, true); 
       } 

       for (final TreeItem child : treeItem.getItems()) { 
           expandTreeItems(child, paths); 
       } 
   }  

   /**
   *
   * */
	public void collapseAll() {
		for (final TreeItem treeItem : m_viewer.getTree().getItems()) {
			collapseAllTreeItems(treeItem);
		}
		m_viewer.refresh(m_viewer.getTree(), true);
		saveState(m_engine.getModel().getProject());
	}

	/**
	*
	* */
	private void collapseAllTreeItems(final TreeItem treeItem) {
		final TreeElement node = (TreeElement) treeItem.getData();
		if (node != null) {
			m_viewer.setExpandedState(node, false);

			for (final TreeItem child : treeItem.getItems()) {
				collapseAllTreeItems(child);
			}
		}
	}

    /**
     *  
     * */
    public IStatus saveState(final IProject oldProject) {
    	if (m_viewer.getControl().isDisposed()) {
    		return new Status(IStatus.INFO, Activator.PLUGIN_ID, Messages.EmbSysRegViewer_disposed);
    	}
    	if (oldProject != null) {    		
    		 Set<String> expandedPaths = new HashSet<>();
    		 
             for (Object obj : m_viewer.getExpandedElements()) { 
        	    TreeParent expandedNode = (TreeParent)obj; 
                expandedPaths.add(expandedNode.getPath()); 
            }
            if (!expandedPaths.isEmpty()) {
                try {
                    oldProject.setSessionProperty(getTreeViewStateQualifier(), expandedPaths);
                } catch (CoreException e) {
                    return new Status(IStatus.ERROR, Activator.PLUGIN_ID, Messages.EmbSysRegViewer_cannot_set_property, e);
                }
            }
        }
		return Status.OK_STATUS;
    } 

    /** */
    private HexColumnEditingSupport getHexColumnEditingSupport(final ComboBoxViewerCellEditor comboBoxCellEditor,
            final TextCellEditor textCellEditor) {
        return new HexColumnEditingSupport(m_viewer, comboBoxCellEditor, textCellEditor, m_engine);                
    }

    /**  */
    private ICellEditorListener getCellEditorListener(final ComboBoxViewerCellEditor comboBoxCellEditor, 
    		                                          final TreeElement element) {
        return new ICellEditorListener() {
            /** fired on just leave combo box */
            // Check against WidgetSelected
            @Override
            public void applyEditorValue() {
                int selectionIndex = ((CCombo) comboBoxCellEditor.getControl()).getSelectionIndex();
                TreeElement obj = element;
                if (obj instanceof TreeField) {
                    final TreeField registerField = (TreeField) obj;
                    long value = -1;

                    if (selectionIndex != -1) {
                        value = registerField.getInterpretations()
                                .getValue(((CCombo) comboBoxCellEditor.getControl()).getItem(selectionIndex));
                    } else {
                        String svalue = ((CCombo) comboBoxCellEditor.getControl()).getText();
                        if (svalue.startsWith("0x")) //$NON-NLS-1$
                            value = Long.valueOf(svalue.substring(2, svalue.length()), 16);
                    }
                    if (value != -1) {                        
                        m_engine.updateRegisterByField(registerField, value);
                    }

                }
            }

            @Override
            public void cancelEditor() {
            	//
            }

            @Override
            public void editorValueChanged(boolean oldValidState, boolean newValidState) {
            	//
            }
        };
        
    }

    /**  */
    private EditingSupport getBinEditingSupport() {
        return new EditingSupport(m_viewer) {
            @Override
            protected boolean canEdit(Object element) {
                return EmbSysRegCore.checkEditRules(element);
            }
            @Override
            protected CellEditor getCellEditor(Object element) {
                return new BinaryButtonsCellEditor(m_viewer.getTree(), m_engine);
            }
            @Override
            protected Object getValue(Object element) {
                return element;
            }
            @Override
            protected void setValue(Object element, Object value) {
                m_viewer.refresh(element);
            }
        };        
    }

    /**
     * 
     * */
    public void updateInfoLabel() {
    	if (infoLabel.isDisposed()) {
    	  return;	
    	}

        final IProject project = m_engine.getModel().getProject();
        if (project == null) {
        	setInfoLabel(Messages.EmbSysRegViewer_No_project_selected);
            return;
        }
        
        try {
			if (!project.isNatureEnabled(CProjectNature.C_NATURE_ID)){
				setInfoLabel(String.format(Messages.EmbSysRegViewer_project, project.getName()) + Messages.EmbSysRegViewer_non_c_project);
				return;
			}
		} catch (CoreException e) {
			setInfoLabel(String.format(Messages.EmbSysRegViewer_project_does_not_exist_or_closed, project.getName()));
			Activator.log(e.getStatus());
			return;
		}

        Map<String, String> props = PropertiesHolder.getInstance(project).getValueMap();
        String storedArchitecture = props.get("architecture"); //$NON-NLS-1$
        String storedVendor = props.get("vendor"); //$NON-NLS-1$
        String storedChip = props.get("chip");   //$NON-NLS-1$
        String storedBoard = props.get("board"); //$NON-NLS-1$
        String storedCore = props.get("core");   //$NON-NLS-1$
        final String projectTitle = String.format(Messages.EmbSysRegViewer_project, project.getName());
        String infoBanner = TreeParent.isModelAccessGranted() ? Messages.EmbSysRegViewer_click_to_read :
                                                                Messages.EmbSysRegViewer_select_context;
        if (m_engine.isEmpty() || storedArchitecture == null || storedVendor == null || storedChip == null) {
        	setInfoLabel(projectTitle
                    + Messages.EmbSysRegViewer_select_chip);
        } else if (storedBoard == null || storedBoard.equals(EMPTY)) {
        	setInfoLabel(projectTitle + ARCH_TXT + storedArchitecture + VEND_TXT + storedVendor 
        			+ CHIP_TXT + storedChip + infoBanner);
        } else if (kind.spr) { 
        	setInfoLabel(projectTitle + ARCH_TXT + storedArchitecture + VEND_TXT + storedVendor 
        			+ CHIP_TXT + storedChip + CORE_TXT + storedCore + infoBanner);
        } else {
        	setInfoLabel(projectTitle + ARCH_TXT + storedArchitecture + VEND_TXT + storedVendor 
        			+ CHIP_TXT + storedChip + BOARD_TXT + storedBoard + infoBanner);
        }
    }

    private void setInfoLabel(String s) {
    	infoLabel.setText(s);
    	infoLabel.setToolTipText(s);
    	infoLabel.pack();
    }
    
    /**
     * Invoked by EmbSysRegView when currently selected presentation, if any,
     * should take the focus.
     */
    public boolean setFocus() {
        boolean result = false;
        if (m_viewer != null) {
            // Tell current visualizer's control to take the focus.
            m_viewer.getControl().setFocus();
        } else {
            // Otherwise, let viewer take the focus.
            // result = super.setFocus();
        }
        return result;
    }

    /** */
    public void setReadOnly(final boolean contextFlag, final boolean sessionFlag) {
        getEngine().getModel().setRootReadOnly(contextFlag);
        getEngine().getModel().cleanList(sessionFlag);
        updateInfoLabel();
        requestUpdate();
    }

    public ISelectionProvider getViewer() {
        return m_viewer;
    }

	public EmbSysRegEngine getEngine() {		
		return m_engine;
	}

	public void setTrackingSession(DsfSession session) {
		getEngine().setSessionTrack(session);		
	}

    public boolean isDisposed() {
        return (m_viewer == null || m_viewer.getControl().isDisposed());
    }

    private QualifiedName getTreeViewStateQualifier() {
        switch (kind) {
            case EMBSYS:
                return EMBSYS_REGVIEW_STATE;
            case SPR:
                return SPR_REGVIEW_STATE;
            default:
                return ALL_REGVIEW_STATE;
        }
    }
}
