/*******************************************************************************
 * Copyright (c) 2016 EmbSysRegView
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     ravenclaw78 - initial API and implementation
 *******************************************************************************/
package org.eclipse.cdt.embsysregview.internal.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.cdt.embsysregview.internal.utils.Utils;


/**
 * Represents physical unit register set.
 */
public class TreeRegisterGroup extends TreeParent {
    TreeRange currentRange;
    // List<TreeRange> rangeList;
    private boolean isExpanded;
    private boolean isChecked;
    private final List<TreeRegister> regsList = new ArrayList<>();

    public TreeRegisterGroup(String name, String description) {
        super(name, description);
        currentRange = new TreeRange(name + "Range0", ""); //$NON-NLS-1$ //$NON-NLS-2$
    }

    public boolean isExpanded() {
        return isExpanded;
    }

    public void setExpanded(boolean isExpanded) {
        this.isExpanded = isExpanded;
    }

    /**
     * Buffers registers to list.
     */
    @Override
    public void addChild(TreeElement child) {
        regsList.add((TreeRegister) child);
    }

    /**
     *  Model can contain "overlapped" registers.
     *  But biggest in size should come first.    
     * */
    final public void extractRanges() {
        // first, sort Device registers by address and then by size
        Collections.sort(regsList, (a, b) -> {
              int addrComp = Utils.compare(a, b, TreeRegister::getRegisterAddress);
              if (addrComp != 0) {
                  return addrComp;
              } else {
            	  return Utils.compare(a, b, TreeRegister::getByteSize);
              }            	               
        });
        // iterate register list
        for (TreeRegister reg : regsList) {
            if (currentRange.hasChildren()) {
                final long lastKnown = currentRange.getLastAddress();
                // check for continuous
                if (reg.getRegisterAddress() - lastKnown > 0) {
                    // store old range
                    closeRange();
                    // open new range
                    currentRange = new TreeRange(Long.toHexString(lastKnown), ""); //$NON-NLS-1$
                }
            }
            //add to model 
            currentRange.addModelChild(reg);
            //add to visual tree regardless of duplication address
            super.addChild(reg);
        }
        closeRange();
    }

    /**  */
    private void closeRange() {
        currentRange.setParent(this);
        super.addModelChild(currentRange);
    }

    /** */
    final public void toggleChecked() {
        isChecked = !isChecked;
    }

    /** */
    final public boolean isChecked() {
        return isChecked;
    }

    public void setChecked(boolean isChecked) {
        this.isChecked = isChecked;
    }

}
